"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeBuildSpecs = exports.BuildSpec = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * BuildSpec for CodeBuild projects
 */
class BuildSpec {
    constructor() {
    }
    static fromObject(value) {
        return new ObjectBuildSpec(value);
    }
    /**
     * Use a file from the source as buildspec
     *
     * Use this if you want to use a file different from 'buildspec.yml'`
     */
    static fromSourceFilename(filename) {
        return new FilenameBuildSpec(filename);
    }
}
exports.BuildSpec = BuildSpec;
/**
 * BuildSpec that just returns the input unchanged
 */
class FilenameBuildSpec extends BuildSpec {
    constructor(filename) {
        super();
        this.filename = filename;
        this.isImmediate = false;
    }
    toBuildSpec() {
        return this.filename;
    }
    toString() {
        return `<buildspec file: ${this.filename}>`;
    }
}
/**
 * BuildSpec that understands about structure
 */
class ObjectBuildSpec extends BuildSpec {
    constructor(spec) {
        super();
        this.spec = spec;
        this.isImmediate = true;
    }
    toBuildSpec() {
        // We have to pretty-print the buildspec, otherwise
        // CodeBuild will not recognize it as an inline buildspec.
        return core_1.Lazy.stringValue({
            produce: (ctx) => core_1.Stack.of(ctx.scope).toJsonString(this.spec, 2),
        });
    }
}
/**
 * Merge two buildspecs into a new BuildSpec
 *
 * NOTE: will currently only merge commands, not artifact
 * declarations, environment variables, secrets, or any
 * other configuration elements.
 *
 * Internal for now because it's not complete/good enough
 * to expose on the objects directly, but we need to it to
 * keep feature-parity for Project.
 *
 * @internal
 */
function mergeBuildSpecs(lhs, rhs) {
    if (!(lhs instanceof ObjectBuildSpec) || !(rhs instanceof ObjectBuildSpec)) {
        throw new Error('Can only merge buildspecs created using BuildSpec.fromObject()');
    }
    return new ObjectBuildSpec(copyCommands(lhs.spec, rhs.spec));
}
exports.mergeBuildSpecs = mergeBuildSpecs;
/**
 * Extend buildSpec phases with the contents of another one
 */
function copyCommands(buildSpec, extend) {
    if (buildSpec.version === '0.1') {
        throw new Error('Cannot extend buildspec at version "0.1". Set the version to "0.2" or higher instead.');
    }
    const ret = Object.assign({}, buildSpec); // Return a copy
    ret.phases = Object.assign({}, ret.phases);
    for (const phaseName of Object.keys(extend.phases)) {
        const phase = ret.phases[phaseName] = Object.assign({}, ret.phases[phaseName]);
        phase.commands = [...phase.commands || [], ...extend.phases[phaseName].commands];
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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