"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginAccessIdentity = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
class OriginAccessIdentityBase extends cdk.Resource {
    /**
     * The ARN to include in S3 bucket policy to allow CloudFront access
     */
    arn() {
        return cdk.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            account: 'cloudfront',
            resource: 'user',
            resourceName: `CloudFront Origin Access Identity ${this.originAccessIdentityName}`,
        });
    }
}
/**
 * An origin access identity is a special CloudFront user that you can
 * associate with Amazon S3 origins, so that you can secure all or just some of
 * your Amazon S3 content.
 *
 * @resource AWS::CloudFront::CloudFrontOriginAccessIdentity
 */
class OriginAccessIdentity extends OriginAccessIdentityBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // Comment has a max length of 128.
        const comment = ((_a = props === null || props === void 0 ? void 0 : props.comment) !== null && _a !== void 0 ? _a : 'Allows CloudFront to reach the bucket').substr(0, 128);
        this.resource = new cloudfront_generated_1.CfnCloudFrontOriginAccessIdentity(this, 'Resource', {
            cloudFrontOriginAccessIdentityConfig: { comment },
        });
        // physical id - OAI name
        this.originAccessIdentityName = this.getResourceNameAttribute(this.resource.ref);
        // Canonical user to whitelist in S3 Bucket Policy
        this.cloudFrontOriginAccessIdentityS3CanonicalUserId = this.resource.attrS3CanonicalUserId;
        // The principal for must be either the canonical user or a special ARN
        // with the CloudFront Origin Access Id (see `arn()` method). For
        // import/export the OAI is anyway required so the principal is constructed
        // with it. But for the normal case the S3 Canonical User as a nicer
        // interface and does not require constructing the ARN.
        this.grantPrincipal = new iam.CanonicalUserPrincipal(this.cloudFrontOriginAccessIdentityS3CanonicalUserId);
    }
    /**
     * Creates a OriginAccessIdentity by providing the OriginAccessIdentityName
     */
    static fromOriginAccessIdentityName(scope, id, originAccessIdentityName) {
        class Import extends OriginAccessIdentityBase {
            constructor(s, i) {
                super(s, i, { physicalName: originAccessIdentityName });
                this.originAccessIdentityName = originAccessIdentityName;
                this.grantPrincipal = new iam.ArnPrincipal(this.arn());
            }
        }
        return new Import(scope, id);
    }
}
exports.OriginAccessIdentity = OriginAccessIdentity;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3JpZ2luX2FjY2Vzc19pZGVudGl0eS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm9yaWdpbl9hY2Nlc3NfaWRlbnRpdHkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEscUNBQXFDLENBQUMsbURBQW1EO0FBQ3pGLGtDQUFrQyxDQUFDLGdEQUFnRDtBQUNuRixpRUFBMkU7QUFxQjNFLE1BQWUsd0JBQXlCLFNBQVEsR0FBRyxDQUFDLFFBQVE7SUFTeEQ7O09BRUc7SUFDTyxHQUFHO1FBQ1QsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDaEMsT0FBTyxFQUFFLEtBQUs7WUFDZCxNQUFNLEVBQUUsRUFBRTtZQUNWLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLFFBQVEsRUFBRSxNQUFNO1lBQ2hCLFlBQVksRUFBRSxxQ0FBcUMsSUFBSSxDQUFDLHdCQUF3QixFQUFFO1NBQ3JGLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjtBQUNEOzs7Ozs7R0FNRztBQUNILE1BQWEsb0JBQXFCLFNBQVEsd0JBQXdCO0lBb0M5RCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQWlDOztRQUMzRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLG1DQUFtQztRQUNuQyxNQUFNLE9BQU8sR0FBRyxPQUFDLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxPQUFPLG1DQUFJLHVDQUF1QyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUMzRixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksd0RBQWlDLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwRSxvQ0FBb0MsRUFBRSxFQUFFLE9BQU8sRUFBRTtTQUNwRCxDQUFDLENBQUM7UUFDSCx5QkFBeUI7UUFDekIsSUFBSSxDQUFDLHdCQUF3QixHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2pGLGtEQUFrRDtRQUNsRCxJQUFJLENBQUMsK0NBQStDLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztRQUMzRix1RUFBdUU7UUFDdkUsaUVBQWlFO1FBQ2pFLDJFQUEyRTtRQUMzRSxvRUFBb0U7UUFDcEUsdURBQXVEO1FBQ3ZELElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLCtDQUErQyxDQUFDLENBQUM7SUFDL0csQ0FBQztJQXBERDs7T0FFRztJQUNJLE1BQU0sQ0FBQyw0QkFBNEIsQ0FBQyxLQUFvQixFQUFFLEVBQVUsRUFBRSx3QkFBZ0M7UUFDekcsTUFBTSxNQUFPLFNBQVEsd0JBQXdCO1lBR3pDLFlBQVksQ0FBZ0IsRUFBRSxDQUFTO2dCQUNuQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLFlBQVksRUFBRSx3QkFBd0IsRUFBRSxDQUFDLENBQUM7Z0JBSDVDLDZCQUF3QixHQUFHLHdCQUF3QixDQUFDO2dCQUNwRCxtQkFBYyxHQUFHLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUdsRSxDQUFDO1NBQ0o7UUFDRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNqQyxDQUFDO0NBeUNKO0FBdERELG9EQXNEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIGNkayBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IENmbkNsb3VkRnJvbnRPcmlnaW5BY2Nlc3NJZGVudGl0eSB9IGZyb20gJy4vY2xvdWRmcm9udC5nZW5lcmF0ZWQnO1xuLyoqXG4gKiBQcm9wZXJ0aWVzIG9mIENsb3VkRnJvbnQgT3JpZ2luQWNjZXNzSWRlbnRpdHlcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBPcmlnaW5BY2Nlc3NJZGVudGl0eVByb3BzIHtcbiAgICAvKipcbiAgICAgKiBBbnkgY29tbWVudHMgeW91IHdhbnQgdG8gaW5jbHVkZSBhYm91dCB0aGUgb3JpZ2luIGFjY2VzcyBpZGVudGl0eS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFwiQWxsb3dzIENsb3VkRnJvbnQgdG8gcmVhY2ggdGhlIGJ1Y2tldFwiXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29tbWVudD86IHN0cmluZztcbn1cbi8qKlxuICogSW50ZXJmYWNlIGZvciBDbG91ZEZyb250IE9yaWdpbkFjY2Vzc0lkZW50aXR5XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSU9yaWdpbkFjY2Vzc0lkZW50aXR5IGV4dGVuZHMgY2RrLklSZXNvdXJjZSwgaWFtLklHcmFudGFibGUge1xuICAgIC8qKlxuICAgICAqIFRoZSBPcmlnaW4gQWNjZXNzIElkZW50aXR5IE5hbWVcbiAgICAgKi9cbiAgICByZWFkb25seSBvcmlnaW5BY2Nlc3NJZGVudGl0eU5hbWU6IHN0cmluZztcbn1cbmFic3RyYWN0IGNsYXNzIE9yaWdpbkFjY2Vzc0lkZW50aXR5QmFzZSBleHRlbmRzIGNkay5SZXNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogVGhlIE9yaWdpbiBBY2Nlc3MgSWRlbnRpdHkgTmFtZSAocGh5c2ljYWwgaWQpXG4gICAgICovXG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IG9yaWdpbkFjY2Vzc0lkZW50aXR5TmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIERlcml2ZWQgcHJpbmNpcGFsIHZhbHVlIGZvciBidWNrZXQgYWNjZXNzXG4gICAgICovXG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcbiAgICAvKipcbiAgICAgKiBUaGUgQVJOIHRvIGluY2x1ZGUgaW4gUzMgYnVja2V0IHBvbGljeSB0byBhbGxvdyBDbG91ZEZyb250IGFjY2Vzc1xuICAgICAqL1xuICAgIHByb3RlY3RlZCBhcm4oKTogc3RyaW5nIHtcbiAgICAgICAgcmV0dXJuIGNkay5TdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgc2VydmljZTogJ2lhbScsXG4gICAgICAgICAgICByZWdpb246ICcnLFxuICAgICAgICAgICAgYWNjb3VudDogJ2Nsb3VkZnJvbnQnLFxuICAgICAgICAgICAgcmVzb3VyY2U6ICd1c2VyJyxcbiAgICAgICAgICAgIHJlc291cmNlTmFtZTogYENsb3VkRnJvbnQgT3JpZ2luIEFjY2VzcyBJZGVudGl0eSAke3RoaXMub3JpZ2luQWNjZXNzSWRlbnRpdHlOYW1lfWAsXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbi8qKlxuICogQW4gb3JpZ2luIGFjY2VzcyBpZGVudGl0eSBpcyBhIHNwZWNpYWwgQ2xvdWRGcm9udCB1c2VyIHRoYXQgeW91IGNhblxuICogYXNzb2NpYXRlIHdpdGggQW1hem9uIFMzIG9yaWdpbnMsIHNvIHRoYXQgeW91IGNhbiBzZWN1cmUgYWxsIG9yIGp1c3Qgc29tZSBvZlxuICogeW91ciBBbWF6b24gUzMgY29udGVudC5cbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpDbG91ZEZyb250OjpDbG91ZEZyb250T3JpZ2luQWNjZXNzSWRlbnRpdHlcbiAqL1xuZXhwb3J0IGNsYXNzIE9yaWdpbkFjY2Vzc0lkZW50aXR5IGV4dGVuZHMgT3JpZ2luQWNjZXNzSWRlbnRpdHlCYXNlIGltcGxlbWVudHMgSU9yaWdpbkFjY2Vzc0lkZW50aXR5IHtcbiAgICAvKipcbiAgICAgKiBDcmVhdGVzIGEgT3JpZ2luQWNjZXNzSWRlbnRpdHkgYnkgcHJvdmlkaW5nIHRoZSBPcmlnaW5BY2Nlc3NJZGVudGl0eU5hbWVcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21PcmlnaW5BY2Nlc3NJZGVudGl0eU5hbWUoc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIG9yaWdpbkFjY2Vzc0lkZW50aXR5TmFtZTogc3RyaW5nKTogSU9yaWdpbkFjY2Vzc0lkZW50aXR5IHtcbiAgICAgICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgT3JpZ2luQWNjZXNzSWRlbnRpdHlCYXNlIHtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBvcmlnaW5BY2Nlc3NJZGVudGl0eU5hbWUgPSBvcmlnaW5BY2Nlc3NJZGVudGl0eU5hbWU7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWwgPSBuZXcgaWFtLkFyblByaW5jaXBhbCh0aGlzLmFybigpKTtcbiAgICAgICAgICAgIGNvbnN0cnVjdG9yKHM6IGNkay5Db25zdHJ1Y3QsIGk6IHN0cmluZykge1xuICAgICAgICAgICAgICAgIHN1cGVyKHMsIGksIHsgcGh5c2ljYWxOYW1lOiBvcmlnaW5BY2Nlc3NJZGVudGl0eU5hbWUgfSk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogVGhlIEFtYXpvbiBTMyBjYW5vbmljYWwgdXNlciBJRCBmb3IgdGhlIG9yaWdpbiBhY2Nlc3MgaWRlbnRpdHksIHVzZWQgd2hlblxuICAgICAqIGdpdmluZyB0aGUgb3JpZ2luIGFjY2VzcyBpZGVudGl0eSByZWFkIHBlcm1pc3Npb24gdG8gYW4gb2JqZWN0IGluIEFtYXpvblxuICAgICAqIFMzLlxuICAgICAqXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBjbG91ZEZyb250T3JpZ2luQWNjZXNzSWRlbnRpdHlTM0Nhbm9uaWNhbFVzZXJJZDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIERlcml2ZWQgcHJpbmNpcGFsIHZhbHVlIGZvciBidWNrZXQgYWNjZXNzXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcbiAgICAvKipcbiAgICAgKiBUaGUgT3JpZ2luIEFjY2VzcyBJZGVudGl0eSBOYW1lIChwaHlzaWNhbCBpZClcbiAgICAgKlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgb3JpZ2luQWNjZXNzSWRlbnRpdHlOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogQ0RLIEwxIHJlc291cmNlXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZTogQ2ZuQ2xvdWRGcm9udE9yaWdpbkFjY2Vzc0lkZW50aXR5O1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wcz86IE9yaWdpbkFjY2Vzc0lkZW50aXR5UHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgLy8gQ29tbWVudCBoYXMgYSBtYXggbGVuZ3RoIG9mIDEyOC5cbiAgICAgICAgY29uc3QgY29tbWVudCA9IChwcm9wcz8uY29tbWVudCA/PyAnQWxsb3dzIENsb3VkRnJvbnQgdG8gcmVhY2ggdGhlIGJ1Y2tldCcpLnN1YnN0cigwLCAxMjgpO1xuICAgICAgICB0aGlzLnJlc291cmNlID0gbmV3IENmbkNsb3VkRnJvbnRPcmlnaW5BY2Nlc3NJZGVudGl0eSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBjbG91ZEZyb250T3JpZ2luQWNjZXNzSWRlbnRpdHlDb25maWc6IHsgY29tbWVudCB9LFxuICAgICAgICB9KTtcbiAgICAgICAgLy8gcGh5c2ljYWwgaWQgLSBPQUkgbmFtZVxuICAgICAgICB0aGlzLm9yaWdpbkFjY2Vzc0lkZW50aXR5TmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHRoaXMucmVzb3VyY2UucmVmKTtcbiAgICAgICAgLy8gQ2Fub25pY2FsIHVzZXIgdG8gd2hpdGVsaXN0IGluIFMzIEJ1Y2tldCBQb2xpY3lcbiAgICAgICAgdGhpcy5jbG91ZEZyb250T3JpZ2luQWNjZXNzSWRlbnRpdHlTM0Nhbm9uaWNhbFVzZXJJZCA9IHRoaXMucmVzb3VyY2UuYXR0clMzQ2Fub25pY2FsVXNlcklkO1xuICAgICAgICAvLyBUaGUgcHJpbmNpcGFsIGZvciBtdXN0IGJlIGVpdGhlciB0aGUgY2Fub25pY2FsIHVzZXIgb3IgYSBzcGVjaWFsIEFSTlxuICAgICAgICAvLyB3aXRoIHRoZSBDbG91ZEZyb250IE9yaWdpbiBBY2Nlc3MgSWQgKHNlZSBgYXJuKClgIG1ldGhvZCkuIEZvclxuICAgICAgICAvLyBpbXBvcnQvZXhwb3J0IHRoZSBPQUkgaXMgYW55d2F5IHJlcXVpcmVkIHNvIHRoZSBwcmluY2lwYWwgaXMgY29uc3RydWN0ZWRcbiAgICAgICAgLy8gd2l0aCBpdC4gQnV0IGZvciB0aGUgbm9ybWFsIGNhc2UgdGhlIFMzIENhbm9uaWNhbCBVc2VyIGFzIGEgbmljZXJcbiAgICAgICAgLy8gaW50ZXJmYWNlIGFuZCBkb2VzIG5vdCByZXF1aXJlIGNvbnN0cnVjdGluZyB0aGUgQVJOLlxuICAgICAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gbmV3IGlhbS5DYW5vbmljYWxVc2VyUHJpbmNpcGFsKHRoaXMuY2xvdWRGcm9udE9yaWdpbkFjY2Vzc0lkZW50aXR5UzNDYW5vbmljYWxVc2VySWQpO1xuICAgIH1cbn1cbiJdfQ==