"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeoRestriction = void 0;
/**
 * Controls the countries in which content is distributed.
 */
class GeoRestriction {
    /**
     * Creates an instance of GeoRestriction for internal use
     *
     * @param restrictionType Specifies the restriction type to impose (whitelist or blacklist)
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist/blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    constructor(restrictionType, locations) {
        this.restrictionType = restrictionType;
        this.locations = locations;
    }
    /**
     * Whitelist specific countries which you want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static whitelist(...locations) {
        return new GeoRestriction('whitelist', GeoRestriction.validateLocations(locations));
    }
    /**
     * Blacklist specific countries which you don't want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static blacklist(...locations) {
        return new GeoRestriction('blacklist', GeoRestriction.validateLocations(locations));
    }
    static validateLocations(locations) {
        if (locations.length === 0) {
            throw new Error('Should provide at least 1 location');
        }
        locations.forEach(location => {
            if (!GeoRestriction.LOCATION_REGEX.test(location)) {
                // eslint-disable-next-line max-len
                throw new Error(`Invalid location format for location: ${location}, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code`);
            }
        });
        return locations;
    }
}
exports.GeoRestriction = GeoRestriction;
GeoRestriction.LOCATION_REGEX = /^[A-Z]{2}$/;
//# sourceMappingURL=data:application/json;base64,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