"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DnsValidatedCertificate = void 0;
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A certificate managed by AWS Certificate Manager.  Will be automatically
 * validated using DNS validation against the specified Route 53 hosted zone.
 *
 * @resource AWS::CertificateManager::Certificate
 * @experimental
 */
class DnsValidatedCertificate extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.domainName = props.domainName;
        this.normalizedZoneName = props.hostedZone.zoneName;
        // Remove trailing `.` from zone name
        if (this.normalizedZoneName.endsWith('.')) {
            this.normalizedZoneName = this.normalizedZoneName.substring(0, this.normalizedZoneName.length - 1);
        }
        // Remove any `/hostedzone/` prefix from the Hosted Zone ID
        this.hostedZoneId = props.hostedZone.hostedZoneId.replace(/^\/hostedzone\//, '');
        const requestorFunction = new lambda.Function(this, 'CertificateRequestorFunction', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '..', 'lambda-packages', 'dns_validated_certificate_handler', 'lib')),
            handler: 'index.certificateRequestHandler',
            runtime: lambda.Runtime.NODEJS_10_X,
            timeout: cdk.Duration.minutes(15),
            role: props.customResourceRole,
        });
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['acm:RequestCertificate', 'acm:DescribeCertificate', 'acm:DeleteCertificate'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:GetChange'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:changeResourceRecordSets'],
            resources: [`arn:${cdk.Stack.of(requestorFunction).partition}:route53:::hostedzone/${this.hostedZoneId}`],
        }));
        const certificate = new cdk.CustomResource(this, 'CertificateRequestorResource', {
            serviceToken: requestorFunction.functionArn,
            properties: {
                DomainName: props.domainName,
                SubjectAlternativeNames: cdk.Lazy.listValue({ produce: () => props.subjectAlternativeNames }, { omitEmpty: true }),
                HostedZoneId: this.hostedZoneId,
                Region: props.region,
                Route53Endpoint: props.route53Endpoint,
            },
        });
        this.certificateArn = certificate.getAtt('Arn').toString();
    }
    validate() {
        const errors = [];
        // Ensure the zone name is a parent zone of the certificate domain name
        if (!cdk.Token.isUnresolved(this.normalizedZoneName) &&
            this.domainName !== this.normalizedZoneName &&
            !this.domainName.endsWith('.' + this.normalizedZoneName)) {
            errors.push(`DNS zone ${this.normalizedZoneName} is not authoritative for certificate domain name ${this.domainName}`);
        }
        return errors;
    }
}
exports.DnsValidatedCertificate = DnsValidatedCertificate;
//# sourceMappingURL=data:application/json;base64,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