"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const batch_generated_1 = require("./batch.generated");
/**
 * Property to specify if the compute environment
 * uses On-Demand or SpotFleet compute resources.
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ComputeResourceType["ON_DEMAND"] = "EC2";
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    ComputeResourceType["SPOT"] = "SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
class ComputeEnvironment extends core_1.Resource {
    constructor(scope, id, props = { enabled: true, managed: true }) {
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        this.validateProps(props);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                allocationStrategy: props.computeResources.allocationStrategy
                    || (props.computeResources.type === ComputeResourceType.SPOT
                        ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                        : AllocationStrategy.BEST_FIT),
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                instanceRole: props.computeResources.instanceRole
                    ? props.computeResources.instanceRole
                    : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                        roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                minvCpus: props.computeResources.minvCpus || 0,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole ? spotFleetRole.roleArn : undefined,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: props.serviceRole
                ? props.serviceRole.roleArn
                : new iam.Role(this, 'Resource-Service-Instance-Role', {
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                    ],
                    assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
                }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.parseArn(computeEnvironmentArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        return props.managed === undefined ? true : props.managed;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        // Setting a bid percentage is only allowed on SPOT resources +
        // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
        if (props.computeResources) {
            if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                // VALIDATE FOR ON_DEMAND
                // Bid percentage is not allowed
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                }
                // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                    throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                }
            }
            else {
                // VALIDATE FOR SPOT
                // Bid percentage must be from 0 - 100
                if (props.computeResources.bidPercentage !== undefined &&
                    (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                    throw new Error('Bid percentage can only be a value between 0 and 100');
                }
            }
            if (props.computeResources.minvCpus) {
                // minvCpus cannot be less than 0
                if (props.computeResources.minvCpus < 0) {
                    throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                }
                // minvCpus cannot exceed max vCpus
                if (props.computeResources.maxvCpus &&
                    props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                    throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _a;
        if (((_a = props.computeResources) === null || _a === void 0 ? void 0 : _a.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
//# sourceMappingURL=data:application/json;base64,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