"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupResource = exports.TagOperation = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * An operation that is applied to a key-value pair
 */
var TagOperation;
(function (TagOperation) {
    /**
     * StringEquals
     */
    TagOperation["STRING_EQUALS"] = "STRINGEQUALS";
    /**
     * Dummy member
     */
    TagOperation["DUMMY"] = "dummy";
})(TagOperation = exports.TagOperation || (exports.TagOperation = {}));
/**
 * A resource to backup
 */
class BackupResource {
    constructor(resource, tagCondition, construct) {
        this.resource = resource;
        this.tagCondition = tagCondition;
        this.construct = construct;
    }
    /**
     * Adds all supported resources in a construct
     *
     * @param construct The construct containing resources to backup
     */
    static fromConstruct(construct) {
        return new BackupResource(undefined, undefined, construct);
    }
    /**
     * A DynamoDB table
     */
    static fromDynamoDbTable(table) {
        return BackupResource.fromArn(table.tableArn);
    }
    /**
     * An EC2 instance
     */
    static fromEc2Instance(instance) {
        return BackupResource.fromArn(core_1.Stack.of(instance).formatArn({
            service: 'ec2',
            resource: 'instance',
            resourceName: instance.instanceId,
        }));
    }
    /**
     * An EFS file system
     */
    static fromEfsFileSystem(fileSystem) {
        return BackupResource.fromArn(core_1.Stack.of(fileSystem).formatArn({
            service: 'elasticfilesystem',
            resource: 'file-system',
            resourceName: fileSystem.fileSystemId,
        }));
    }
    /**
     * A RDS database instance
     */
    static fromRdsDatabaseInstance(instance) {
        return BackupResource.fromArn(instance.instanceArn);
    }
    /**
     * A list of ARNs or match patterns such as
     * `arn:aws:ec2:us-east-1:123456789012:volume/*`
     */
    static fromArn(arn) {
        return new BackupResource(arn);
    }
    /**
     * A tag condition
     */
    static fromTag(key, value, operation) {
        return new BackupResource(undefined, {
            key,
            value,
            operation,
        });
    }
}
exports.BackupResource = BackupResource;
//# sourceMappingURL=data:application/json;base64,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