"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PredefinedMetric = exports.TargetTrackingScalingPolicy = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error('Exactly one of \'customMetric\' or \'predefinedMetric\' must be specified.');
        }
        if (props.predefinedMetric === PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET && !props.resourceLabel) {
            throw new Error('When tracking the ALBRequestCountPerTarget metric, the ALB identifier must be supplied in resourceLabel');
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error('Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.');
        }
        super(scope, id);
        this.resource = new autoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyType: 'TargetTrackingScaling',
            autoScalingGroupName: props.autoScalingGroup.autoScalingGroupName,
            cooldown: props.cooldown && props.cooldown.toSeconds().toString(),
            estimatedInstanceWarmup: props.estimatedInstanceWarmup && props.estimatedInstanceWarmup.toSeconds(),
            targetTrackingConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                targetValue: props.targetValue,
            },
        });
        this.scalingPolicyArn = this.resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    /**
     * Average CPU utilization of the Auto Scaling group
     */
    PredefinedMetric["ASG_AVERAGE_CPU_UTILIZATION"] = "ASGAverageCPUUtilization";
    /**
     * Average number of bytes received on all network interfaces by the Auto Scaling group
     */
    PredefinedMetric["ASG_AVERAGE_NETWORK_IN"] = "ASGAverageNetworkIn";
    /**
     * Average number of bytes sent out on all network interfaces by the Auto Scaling group
     */
    PredefinedMetric["ASG_AVERAGE_NETWORK_OUT"] = "ASGAverageNetworkOut";
    /**
     * Number of requests completed per target in an Application Load Balancer target group
     *
     * Specify the ALB to look at in the `resourceLabel` field.
     */
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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