"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricAggregationType = exports.AdjustmentType = exports.StepScalingAction = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        const resource = new autoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyType: 'StepScaling',
            autoScalingGroupName: props.autoScalingGroup.autoScalingGroupName,
            cooldown: props.cooldown && props.cooldown.toSeconds().toString(),
            estimatedInstanceWarmup: props.estimatedInstanceWarmup && props.estimatedInstanceWarmup.toSeconds(),
            adjustmentType: props.adjustmentType,
            minAdjustmentMagnitude: props.minAdjustmentMagnitude,
            metricAggregationType: props.metricAggregationType,
            stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["AVERAGE"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["MINIMUM"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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