"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.GroupMetric = exports.GroupMetrics = exports.Monitoring = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
/**
 * A set of group metrics
 */
class GroupMetrics {
    constructor(...metrics) {
        /**
         * @internal
         */
        this._metrics = new Set();
        metrics === null || metrics === void 0 ? void 0 : metrics.forEach(metric => this._metrics.add(metric));
    }
    /**
     * Report all group metrics.
     */
    static all() {
        return new GroupMetrics();
    }
}
exports.GroupMetrics = GroupMetrics;
/**
 * Group metrics that an Auto Scaling group sends to Amazon CloudWatch.
 */
class GroupMetric {
    constructor(name) {
        this.name = name;
    }
}
exports.GroupMetric = GroupMetric;
/**
 * The minimum size of the Auto Scaling group
 */
GroupMetric.MIN_SIZE = new GroupMetric('GroupMinSize');
/**
 * The maximum size of the Auto Scaling group
 */
GroupMetric.MAX_SIZE = new GroupMetric('GroupMaxSize');
/**
 * The number of instances that the Auto Scaling group attempts to maintain
 */
GroupMetric.DESIRED_CAPACITY = new GroupMetric('GroupDesiredCapacity');
/**
 * The number of instances that are running as part of the Auto Scaling group
 * This metric does not include instances that are pending or terminating
 */
GroupMetric.IN_SERVICE_INSTANCES = new GroupMetric('GroupInServiceInstances');
/**
 * The number of instances that are pending
 * A pending instance is not yet in service, this metric does not include instances that are in service or terminating
 */
GroupMetric.PENDING_INSTANCES = new GroupMetric('GroupPendingInstances');
/**
 * The number of instances that are in a Standby state
 * Instances in this state are still running but are not actively in service
 */
GroupMetric.STANDBY_INSTANCES = new GroupMetric('GroupStandbyInstances');
/**
 * The number of instances that are in the process of terminating
 * This metric does not include instances that are in service or pending
 */
GroupMetric.TERMINATING_INSTANCES = new GroupMetric('GroupTerminatingInstances');
/**
 * The total number of instances in the Auto Scaling group
 * This metric identifies the number of instances that are in service, pending, and terminating
 */
GroupMetric.TOTAL_INSTANCES = new GroupMetric('GroupTotalInstances');
class AutoScalingGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
    }
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: props.targetRequestsPerSecond,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
    addUserData(..._commands) {
        // do nothing
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.autoScalingGroupName,
        });
        this.securityGroups = [];
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.groupMetrics = [];
        this.notifications = [];
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: props.allowAllOutbound !== false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        if (props.groupMetrics) {
            this.groupMetrics.push(...props.groupMetrics);
        }
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
            securityGroups: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            associatePublicIpAddress: props.associatePublicIpAddress,
            spotPrice: props.spotPrice,
            blockDeviceMappings: (props.blockDevices !== undefined ?
                synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
        });
        launchConfig.node.addDependency(this.role);
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity !== undefined ? props.minCapacity : 1;
        const maxCapacity = props.maxCapacity !== undefined ? props.maxCapacity :
            desiredCapacity !== undefined ? desiredCapacity : Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            core_1.Annotations.of(this).addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        if (this.maxInstanceLifetime &&
            (this.maxInstanceLifetime.toSeconds() < 604800 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 7 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => {
                var _a;
                return ({
                    topic: nc.topic,
                    scalingEvents: (_a = nc.scalingEvents) !== null && _a !== void 0 ? _a : ScalingEvents.ALL,
                });
            });
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            autoScalingGroupName: this.physicalName,
            cooldown: props.cooldown !== undefined ? props.cooldown.toSeconds().toString() : undefined,
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            launchConfigurationName: launchConfig.ref,
            loadBalancerNames: core_1.Lazy.listValue({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.listValue({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            metricsCollection: core_1.Lazy.anyValue({ produce: () => this.renderMetricsCollection() }),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.osType = imageConfig.osType;
        this.autoScalingGroupName = this.getResourceNameAttribute(this.autoScalingGroup.ref),
            this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                service: 'autoscaling',
                resource: 'autoScalingGroup:*:autoScalingGroupName',
                resourceName: this.autoScalingGroupName,
            });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props);
        this.spotPrice = props.spotPrice;
    }
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
                this.osType = ec2.OperatingSystemType.UNKNOWN;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        // undefined is treated as 'true'
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
    renderMetricsCollection() {
        if (this.groupMetrics.length === 0) {
            return undefined;
        }
        return this.groupMetrics.map(group => {
            var _a;
            return ({
                granularity: '1Minute',
                metrics: ((_a = group._metrics) === null || _a === void 0 ? void 0 : _a.size) !== 0 ? [...group._metrics].map(m => m.name) : undefined,
            });
        });
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
/**
 * The type of update to perform on instances in this AutoScalingGroup
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launched
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custom group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
/**
 * Health check settings
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined ? true : false;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toISOString(),
        suspendProcesses: config.suspendProcesses !== undefined ? config.suspendProcesses :
            // Recommended list of processes to suspend from here:
            // https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
            [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
                ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS],
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                core_1.Annotations.of(construct).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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