"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const fs_1 = require("fs");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const private_1 = require("./private");
const schema_intermediate_1 = require("./schema-intermediate");
;
/**
 * The Schema for a GraphQL Api
 *
 * If no options are configured, schema will be generated
 * code-first.
 */
class Schema {
    constructor(options) {
        if (options === null || options === void 0 ? void 0 : options.filePath) {
            this.mode = private_1.SchemaMode.FILE;
            this.definition = fs_1.readFileSync(options.filePath).toString('utf-8');
        }
        else {
            this.mode = private_1.SchemaMode.CODE;
            this.definition = '';
        }
        this.types = [];
    }
    /**
     * Generate a Schema from file
     *
     * @returns `SchemaAsset` with immutable schema defintion
     * @param filePath the file path of the schema file
     */
    static fromAsset(filePath) {
        return new Schema({ filePath });
    }
    /**
     * Called when the GraphQL Api is initialized to allow this object to bind
     * to the stack.
     *
     * @param api The binding GraphQL Api
     */
    bind(api) {
        if (!this.schema) {
            this.schema = new appsync_generated_1.CfnGraphQLSchema(api, 'Schema', {
                apiId: api.apiId,
                definition: this.mode === private_1.SchemaMode.CODE ?
                    core_1.Lazy.stringValue({
                        produce: () => this.types.reduce((acc, type) => { return `${acc}${type._bindToGraphqlApi(api).toString()}\n`; }, `${this.declareSchema()}${this.definition}`),
                    })
                    : this.definition,
            });
        }
        return this.schema;
    }
    /**
     * Escape hatch to add to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition, delimiter) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter !== null && delimiter !== void 0 ? delimiter : '';
        this.definition = `${this.definition}${sep}${addition}\n`;
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add query. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.query) {
            this.query = new schema_intermediate_1.ObjectType('Query', { definition: {} });
            this.addType(this.query);
        }
        ;
        this.query.addField({ fieldName, field });
        return this.query;
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add mutation. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.mutation) {
            this.mutation = new schema_intermediate_1.ObjectType('Mutation', { definition: {} });
            this.addType(this.mutation);
        }
        ;
        this.mutation.addField({ fieldName, field });
        return this.mutation;
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        var _a, _b;
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add subscription. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.subscription) {
            this.subscription = new schema_intermediate_1.ObjectType('Subscription', { definition: {} });
            this.addType(this.subscription);
        }
        const directives = (_b = (_a = field.fieldOptions) === null || _a === void 0 ? void 0 : _a.directives) === null || _b === void 0 ? void 0 : _b.filter((directive) => directive.mutationFields);
        if (directives && directives.length > 1) {
            throw new Error(`Subscription fields must not have more than one @aws_subscribe directives. Received: ${directives.length}`);
        }
        this.subscription.addField({ fieldName, field });
        return this.subscription;
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.types.push(type);
        return type;
    }
    /**
     * Set the root types of this schema if they are defined.
     *
     * For example:
     * schema {
     *   query: Query
     *   mutation: Mutation
     *   subscription: Subscription
     * }
     */
    declareSchema() {
        if (!this.query && !this.mutation && !this.subscription) {
            return '';
        }
        const list = ['query', 'mutation', 'subscription'];
        return private_1.shapeAddition({
            prefix: 'schema',
            fields: list.map((key) => { var _a; return this[key] ? `${key}: ${(_a = this[key]) === null || _a === void 0 ? void 0 : _a.name}` : ''; })
                .filter((field) => field != ''),
        }) + '\n';
    }
}
exports.Schema = Schema;
//# sourceMappingURL=data:application/json;base64,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