"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Type = exports.Directive = void 0;
const graphqlapi_1 = require("./graphqlapi");
/**
 * Directives for types
 *
 * i.e. @aws_iam or @aws_subscribe
 *
 * @experimental
 */
class Directive {
    constructor(statement, options) {
        this.statement = statement;
        this.mode = options === null || options === void 0 ? void 0 : options.mode;
        this.mutationFields = options === null || options === void 0 ? void 0 : options.mutationFields;
    }
    /**
     * Add the @aws_iam directive
     */
    static iam() {
        return new Directive('@aws_iam', { mode: graphqlapi_1.AuthorizationType.IAM });
    }
    /**
     * Add the @aws_oidc directive
     */
    static oidc() {
        return new Directive('@aws_oidc', { mode: graphqlapi_1.AuthorizationType.OIDC });
    }
    /**
     * Add the @aws_api_key directive
     */
    static apiKey() {
        return new Directive('@aws_api_key', { mode: graphqlapi_1.AuthorizationType.API_KEY });
    }
    /**
     * Add the @aws_auth or @aws_cognito_user_pools directive
     *
     * @param groups the groups to allow access to
     */
    static cognito(...groups) {
        if (groups.length === 0) {
            throw new Error(`Cognito authorization requires at least one Cognito group to be supplied. Received: ${groups.length}`);
        }
        // this function creates the cognito groups as a string (i.e. ["group1", "group2", "group3"])
        const stringify = (array) => {
            return array.reduce((acc, element) => `${acc}"${element}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_auth(cognito_groups: [${stringify(groups)}])`, { mode: graphqlapi_1.AuthorizationType.USER_POOL });
    }
    /**
     * Add the @aws_subscribe directive. Only use for top level Subscription type.
     *
     * @param mutations the mutation fields to link to
     */
    static subscribe(...mutations) {
        if (mutations.length === 0) {
            throw new Error(`Subscribe directive requires at least one mutation field to be supplied. Received: ${mutations.length}`);
        }
        // this function creates the subscribe directive as a string (i.e. ["mutation_field_1", "mutation_field_2"])
        const stringify = (array) => {
            return array.reduce((acc, mutation) => `${acc}"${mutation}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_subscribe(mutations: [${stringify(mutations)}])`, { mutationFields: mutations });
    }
    /**
     * Add a custom directive
     *
     * @param statement - the directive statement to append
     */
    static custom(statement) {
        return new Directive(statement);
    }
    /**
     * Method called when the stringifying Directive for schema generation
     *
     * @param modes the authorization modes
     *
     * @internal
     */
    _bindToAuthModes(modes) {
        this.modes = modes;
        return this;
    }
    /**
     * Generate the directive statement
     */
    toString() {
        if (this.modes && this.mode && !this.modes.some((mode) => mode === this.mode)) {
            throw new Error(`No Authorization Type ${this.mode} declared in GraphQL Api.`);
        }
        if (this.mode === graphqlapi_1.AuthorizationType.USER_POOL && this.modes && this.modes.length > 1) {
            this.statement = this.statement.replace('@aws_auth', '@aws_cognito_user_pools');
        }
        return this.statement;
    }
}
exports.Directive = Directive;
/**
 * Enum containing the Types that can be used to define ObjectTypes
 */
var Type;
(function (Type) {
    /**
     * `ID` scalar type is a unique identifier. `ID` type is serialized similar to `String`.
     *
     * Often used as a key for a cache and not intended to be human-readable.
     */
    Type["ID"] = "ID";
    /**
     * `String` scalar type is a free-form human-readable text.
     */
    Type["STRING"] = "String";
    /**
     * `Int` scalar type is a signed non-fractional numerical value.
     */
    Type["INT"] = "Int";
    /**
     * `Float` scalar type is a signed double-precision fractional value.
     */
    Type["FLOAT"] = "Float";
    /**
     * `Boolean` scalar type is a boolean value: true or false.
     */
    Type["BOOLEAN"] = "Boolean";
    /**
     * `AWSDate` scalar type represents a valid extended `ISO 8601 Date` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DD`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Calendar_dates
     */
    Type["AWS_DATE"] = "AWSDate";
    /**
     * `AWSTime` scalar type represents a valid extended `ISO 8601 Time` string.
     *
     * In other words, accepts date strings in the form of `hh:mm:ss.sss`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Times
     */
    Type["AWS_TIME"] = "AWSTime";
    /**
     * `AWSDateTime` scalar type represents a valid extended `ISO 8601 DateTime` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DDThh:mm:ss.sssZ`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Combined_date_and_time_representations
     */
    Type["AWS_DATE_TIME"] = "AWSDateTime";
    /**
     * `AWSTimestamp` scalar type represents the number of seconds since `1970-01-01T00:00Z`.
     *
     * Timestamps are serialized and deserialized as numbers.
     */
    Type["AWS_TIMESTAMP"] = "AWSTimestamp";
    /**
     * `AWSEmail` scalar type represents an email address string (i.e.`username@example.com`)
     */
    Type["AWS_EMAIL"] = "AWSEmail";
    /**
     * `AWSJson` scalar type represents a JSON string.
     */
    Type["AWS_JSON"] = "AWSJSON";
    /**
     * `AWSURL` scalar type represetns a valid URL string.
     *
     * URLs wihtout schemes or contain double slashes are considered invalid.
     */
    Type["AWS_URL"] = "AWSURL";
    /**
     * `AWSPhone` scalar type represents a valid phone number. Phone numbers maybe be whitespace delimited or hyphenated.
     *
     * The number can specify a country code at the beginning, but is not required for US phone numbers.
     */
    Type["AWS_PHONE"] = "AWSPhone";
    /**
     * `AWSIPAddress` scalar type respresents a valid `IPv4` of `IPv6` address string.
     */
    Type["AWS_IP_ADDRESS"] = "AWSIPAddress";
    /**
     * Type used for Intermediate Types
     * (i.e. an interface or an object type)
     */
    Type["INTERMEDIATE"] = "INTERMEDIATE";
})(Type = exports.Type || (exports.Type = {}));
//# sourceMappingURL=data:application/json;base64,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