"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Values = exports.AttributeValuesStep = exports.AttributeValues = exports.PartitionKey = exports.PrimaryKey = exports.SortKeyStep = exports.PartitionKeyStep = exports.Assign = exports.KeyCondition = void 0;
const private_1 = require("./private");
/**
 * Factory class for DynamoDB key conditions.
 */
class KeyCondition {
    constructor(cond) {
        this.cond = cond;
    }
    /**
     * Condition k = arg, true if the key attribute k is equal to the Query argument
     */
    static eq(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '=', arg));
    }
    /**
     * Condition k < arg, true if the key attribute k is less than the Query argument
     */
    static lt(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '<', arg));
    }
    /**
     * Condition k <= arg, true if the key attribute k is less than or equal to the Query argument
     */
    static le(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '<=', arg));
    }
    /**
     * Condition k > arg, true if the key attribute k is greater than the the Query argument
     */
    static gt(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '>', arg));
    }
    /**
     * Condition k >= arg, true if the key attribute k is greater or equal to the Query argument
     */
    static ge(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '>=', arg));
    }
    /**
     * Condition (k, arg). True if the key attribute k begins with the Query argument.
     */
    static beginsWith(keyName, arg) {
        return new KeyCondition(new private_1.BeginsWith(keyName, arg));
    }
    /**
     * Condition k BETWEEN arg1 AND arg2, true if k >= arg1 and k <= arg2.
     */
    static between(keyName, arg1, arg2) {
        return new KeyCondition(new private_1.Between(keyName, arg1, arg2));
    }
    /**
     * Conjunction between two conditions.
     */
    and(keyCond) {
        return new KeyCondition(this.cond.and(keyCond.cond));
    }
    /**
     * Renders the key condition to a VTL string.
     */
    renderTemplate() {
        return `"query" : {
            "expression" : "${this.cond.renderCondition()}",
            "expressionNames" : {
                ${this.cond.renderExpressionNames()}
            },
            "expressionValues" : {
                ${this.cond.renderExpressionValues()}
            }
        }`;
    }
}
exports.KeyCondition = KeyCondition;
/**
 * Utility class representing the assigment of a value to an attribute.
 */
class Assign {
    constructor(attr, arg) {
        this.attr = attr;
        this.arg = arg;
    }
    /**
     * Renders the assignment as a VTL string.
     */
    renderAsAssignment() {
        return `"${this.attr}" : $util.dynamodb.toDynamoDBJson(${this.arg})`;
    }
    /**
     * Renders the assignment as a map element.
     */
    putInMap(map) {
        return `$util.qr($${map}.put("${this.attr}", ${this.arg}))`;
    }
}
exports.Assign = Assign;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a partition key.
 */
class PartitionKeyStep {
    constructor(key) {
        this.key = key;
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    is(val) {
        return new PartitionKey(new Assign(this.key, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    auto() {
        return new PartitionKey(new Assign(this.key, '$util.autoId()'));
    }
}
exports.PartitionKeyStep = PartitionKeyStep;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a sort key.
 */
class SortKeyStep {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    is(val) {
        return new PrimaryKey(this.pkey, new Assign(this.skey, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    auto() {
        return new PrimaryKey(this.pkey, new Assign(this.skey, '$util.autoId()'));
    }
}
exports.SortKeyStep = SortKeyStep;
/**
 * Specifies the assignment to the primary key. It either
 * contains the full primary key or only the partition key.
 */
class PrimaryKey {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Allows assigning a value to the partition key.
     */
    static partition(key) {
        return new PartitionKeyStep(key);
    }
    /**
     * Renders the key assignment to a VTL string.
     */
    renderTemplate() {
        const assignments = [this.pkey.renderAsAssignment()];
        if (this.skey) {
            assignments.push(this.skey.renderAsAssignment());
        }
        return `"key" : {
      ${assignments.join(',')}
    }`;
    }
}
exports.PrimaryKey = PrimaryKey;
/**
 * Specifies the assignment to the partition key. It can be
 * enhanced with the assignment of the sort key.
 */
class PartitionKey extends PrimaryKey {
    constructor(pkey) {
        super(pkey);
    }
    /**
     * Allows assigning a value to the sort key.
     */
    sort(key) {
        return new SortKeyStep(this.pkey, key);
    }
}
exports.PartitionKey = PartitionKey;
/**
 * Specifies the attribute value assignments.
 */
class AttributeValues {
    constructor(container, assignments = []) {
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    attribute(attr) {
        return new AttributeValuesStep(attr, this.container, this.assignments);
    }
    /**
     * Renders the variables required for `renderTemplate`.
     */
    renderVariables() {
        return `#set($input = ${this.container})
      ${this.assignments.map(a => a.putInMap('input')).join('\n')}`;
    }
    /**
     * Renders the attribute value assingments to a VTL string.
     */
    renderTemplate() {
        return '"attributeValues": $util.dynamodb.toMapValuesJson($input)';
    }
}
exports.AttributeValues = AttributeValues;
/**
 * Utility class to allow assigning a value to an attribute.
 */
class AttributeValuesStep {
    constructor(attr, container, assignments) {
        this.attr = attr;
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Assign the value to the current attribute.
     */
    is(val) {
        this.assignments.push(new Assign(this.attr, val));
        return new AttributeValues(this.container, this.assignments);
    }
}
exports.AttributeValuesStep = AttributeValuesStep;
/**
 * Factory class for attribute value assignments.
 */
class Values {
    /**
     * Treats the specified object as a map of assignments, where the property
     * names represent attribute names. It’s opinionated about how it represents
     * some of the nested objects: e.g., it will use lists (“L”) rather than sets
     * (“SS”, “NS”, “BS”). By default it projects the argument container ("$ctx.args").
     */
    static projecting(arg) {
        return new AttributeValues('$ctx.args' + (arg ? `.${arg}` : ''));
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    static attribute(attr) {
        return new AttributeValues('{}').attribute(attr);
    }
}
exports.Values = Values;
//# sourceMappingURL=data:application/json;base64,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