"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualNodeBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.backends = new Array();
        this.listeners = new Array();
    }
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to
     */
    addBackends(...props) {
        for (const s of props) {
            this.backends.push({
                virtualService: {
                    virtualServiceName: s.virtualServiceName,
                },
            });
        }
    }
    /**
     * Utility method to add an inbound listener for this virtual node
     */
    addListeners(...listeners) {
        if (this.listeners.length + listeners.length > 1) {
            throw new Error('VirtualNode may have at most one listener');
        }
        for (const listener of listeners) {
            const portMapping = listener.portMapping || { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP };
            this.listeners.push({
                portMapping,
                healthCheck: renderHealthCheck(listener.healthCheck, portMapping),
            });
        }
    }
}
/**
 * Minimum and maximum thresholds for HeathCheck numeric properties
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_HealthCheckPolicy.html
 */
const HEALTH_CHECK_PROPERTY_THRESHOLDS = {
    healthyThreshold: [2, 10],
    intervalMillis: [5000, 300000],
    port: [1, 65535],
    timeoutMillis: [2000, 60000],
    unhealthyThreshold: [2, 10],
};
function renderHealthCheck(hc, pm) {
    if (hc === undefined) {
        return undefined;
    }
    if (hc.protocol === shared_interfaces_1.Protocol.TCP && hc.path) {
        throw new Error('The path property cannot be set with Protocol.TCP');
    }
    if (hc.protocol === shared_interfaces_1.Protocol.GRPC && hc.path) {
        throw new Error('The path property cannot be set with Protocol.GRPC');
    }
    const healthCheck = {
        healthyThreshold: hc.healthyThreshold || 2,
        intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
        path: hc.path || (hc.protocol === shared_interfaces_1.Protocol.HTTP ? '/' : undefined),
        port: hc.port || pm.port,
        protocol: hc.protocol || pm.protocol,
        timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
        unhealthyThreshold: hc.unhealthyThreshold || 2,
    };
    Object.keys(healthCheck)
        .filter((key) => HEALTH_CHECK_PROPERTY_THRESHOLDS[key] &&
        typeof healthCheck[key] === 'number' &&
        !cdk.Token.isUnresolved(healthCheck[key])).map((key) => {
        const [min, max] = HEALTH_CHECK_PROPERTY_THRESHOLDS[key];
        const value = healthCheck[key];
        if (value < min) {
            throw new Error(`The value of '${key}' is below the minimum threshold (expected >=${min}, got ${value})`);
        }
        if (value > max) {
            throw new Error(`The value of '${key}' is above the maximum threshold (expected <=${max}, got ${value})`);
        }
    });
    return healthCheck;
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 */
class VirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.mesh = props.mesh;
        this.addBackends(...props.backends || []);
        this.addListeners(...props.listener ? [props.listener] : []);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                backends: cdk.Lazy.anyValue({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.anyValue({ produce: () => this.listeners }, { omitEmptyArray: true }),
                serviceDiscovery: {
                    dns: props.dnsHostName !== undefined ? { hostname: props.dnsHostName } : undefined,
                    awsCloudMap: props.cloudMapService !== undefined ? {
                        serviceName: props.cloudMapService.serviceName,
                        namespaceName: props.cloudMapService.namespace.namespaceName,
                        attributes: renderAttributes(props.cloudMapServiceInstanceAttributes),
                    } : undefined,
                },
                logging: {
                    accessLog: {
                        file: {
                            path: '/dev/stdout',
                        },
                    },
                },
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualNode given an ARN
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new ImportedVirtualNode(scope, id, { virtualNodeArn });
    }
    /**
     * Import an existing VirtualNode given its name
     */
    static fromVirtualNodeName(scope, id, meshName, virtualNodeName) {
        return new ImportedVirtualNode(scope, id, {
            meshName,
            virtualNodeName,
        });
    }
}
exports.VirtualNode = VirtualNode;
function renderAttributes(attrs) {
    if (attrs === undefined) {
        return undefined;
    }
    return Object.entries(attrs).map(([key, value]) => ({ key, value }));
}
/**
 * Used to import a VirtualNode and read it's properties
 */
class ImportedVirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualNodeArn) {
            this.virtualNodeArn = props.virtualNodeArn;
            this.virtualNodeName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualNodeArn).resourceName));
        }
        else if (props.virtualNodeName && props.meshName) {
            this.virtualNodeName = props.virtualNodeName;
            this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualNode`,
                resourceName: this.virtualNodeName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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