"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route = exports.RouteType = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * Type of route
 */
var RouteType;
(function (RouteType) {
    /**
     * HTTP route
     */
    RouteType["HTTP"] = "http";
    /**
     * TCP route
     */
    RouteType["TCP"] = "tcp";
})(RouteType = exports.RouteType || (exports.RouteType = {}));
/**
 * Route represents a new or existing route attached to a VirtualRouter and Mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
 */
class Route extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.routeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.weightedTargets = new Array();
        this.virtualRouter = props.virtualRouter;
        const routeType = props.routeType !== undefined ? props.routeType :
            props.prefix !== undefined ? RouteType.HTTP :
                RouteType.TCP;
        if (routeType === RouteType.HTTP) {
            this.httpRoute = this.renderHttpRoute(props);
        }
        else {
            this.tcpRoute = this.renderTcpRoute(props);
        }
        const route = new appmesh_generated_1.CfnRoute(this, 'Resource', {
            routeName: this.physicalName,
            meshName: this.virtualRouter.mesh.meshName,
            virtualRouterName: this.virtualRouter.virtualRouterName,
            spec: {
                tcpRoute: this.tcpRoute,
                httpRoute: this.httpRoute,
            },
        });
        this.routeName = this.getResourceNameAttribute(route.attrRouteName);
        this.routeArn = this.getResourceArnAttribute(route.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter/${props.virtualRouter.virtualRouterName}/route`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing route given an ARN
     */
    static fromRouteArn(scope, id, routeArn) {
        return new ImportedRoute(scope, id, { routeArn });
    }
    /**
     * Import an existing route given its name
     */
    static fromRouteName(scope, id, meshName, virtualRouterName, routeName) {
        return new ImportedRoute(scope, id, { meshName, virtualRouterName, routeName });
    }
    /**
     * Utility method to add weighted route targets to an existing route
     */
    renderWeightedTargets(props) {
        for (const t of props) {
            this.weightedTargets.push({
                virtualNode: t.virtualNode.virtualNodeName,
                weight: t.weight || 1,
            });
        }
        return this.weightedTargets;
    }
    renderHttpRoute(props) {
        return {
            match: {
                prefix: props.prefix || '/',
            },
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
    renderTcpRoute(props) {
        return {
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
}
exports.Route = Route;
/**
 * Represents and imported IRoute
 */
class ImportedRoute extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.routeArn) {
            this.routeArn = props.routeArn;
            this.routeName = cdk.Fn.select(4, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.routeArn).resourceName));
        }
        else if (props.routeName && props.meshName && props.virtualRouterName) {
            this.routeName = props.routeName;
            this.routeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter/${props.virtualRouterName}/route`,
                resourceName: this.routeName,
            });
        }
        else {
            throw new Error('Need either arn or three names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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