import { Construct, Resource } from "../../../core";
import { IIntegration } from '../common';
import { IHttpApi } from './api';
import { HttpMethod, IHttpRoute } from './route';
/**
 * Represents an Integration for an HTTP API.
 */
export interface IHttpIntegration extends IIntegration {
    /** The HTTP API associated with this integration */
    readonly httpApi: IHttpApi;
}
/**
 * Supported integration types
 */
export declare enum HttpIntegrationType {
    /**
     * Integration type is a Lambda proxy
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-develop-integrations-lambda.html
     */
    LAMBDA_PROXY = "AWS_PROXY",
    /**
     * Integration type is an HTTP proxy
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-develop-integrations-lambda.html
     */
    HTTP_PROXY = "HTTP_PROXY"
}
/**
 * Payload format version for lambda proxy integration
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-develop-integrations-lambda.html
 */
export declare class PayloadFormatVersion {
    /** Version 1.0 */
    static readonly VERSION_1_0: PayloadFormatVersion;
    /** Version 2.0 */
    static readonly VERSION_2_0: PayloadFormatVersion;
    /**
     * A custom payload version.
     * Typically used if there is a version number that the CDK doesn't support yet
     */
    static custom(version: string): PayloadFormatVersion;
    /** version as a string */
    readonly version: string;
    private constructor();
}
/**
 * The integration properties
 */
export interface HttpIntegrationProps {
    /**
     * The HTTP API to which this integration should be bound.
     */
    readonly httpApi: IHttpApi;
    /**
     * Integration type
     */
    readonly integrationType: HttpIntegrationType;
    /**
     * Integration URI.
     * This will be the function ARN in the case of `HttpIntegrationType.LAMBDA_PROXY`,
     * or HTTP URL in the case of `HttpIntegrationType.HTTP_PROXY`.
     */
    readonly integrationUri: string;
    /**
     * The HTTP method to use when calling the underlying HTTP proxy
     * @default - none. required if the integration type is `HttpIntegrationType.HTTP_PROXY`.
     */
    readonly method?: HttpMethod;
    /**
     * The version of the payload format
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-develop-integrations-lambda.html
     * @default - defaults to latest in the case of HttpIntegrationType.LAMBDA_PROXY`, irrelevant otherwise.
     */
    readonly payloadFormatVersion?: PayloadFormatVersion;
}
/**
 * The integration for an API route.
 * @resource AWS::ApiGatewayV2::Integration
 */
export declare class HttpIntegration extends Resource implements IHttpIntegration {
    readonly integrationId: string;
    readonly httpApi: IHttpApi;
    constructor(scope: Construct, id: string, props: HttpIntegrationProps);
}
/**
 * Options to the HttpRouteIntegration during its bind operation.
 */
export interface HttpRouteIntegrationBindOptions {
    /**
     * The route to which this is being bound.
     */
    readonly route: IHttpRoute;
    /**
     * The current scope in which the bind is occurring.
     * If the `HttpRouteIntegration` being bound creates additional constructs,
     * this will be used as their parent scope.
     */
    readonly scope: Construct;
}
/**
 * The interface that various route integration classes will inherit.
 */
export interface IHttpRouteIntegration {
    /**
     * Bind this integration to the route.
     */
    bind(options: HttpRouteIntegrationBindOptions): HttpRouteIntegrationConfig;
}
/**
 * Config returned back as a result of the bind.
 */
export interface HttpRouteIntegrationConfig {
    /**
     * Integration type.
     */
    readonly type: HttpIntegrationType;
    /**
     * Integration URI
     */
    readonly uri: string;
    /**
     * The HTTP method that must be used to invoke the underlying proxy.
     * Required for `HttpIntegrationType.HTTP_PROXY`
     * @default - undefined
     */
    readonly method?: HttpMethod;
    /**
     * Payload format version in the case of lambda proxy integration
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-develop-integrations-lambda.html
     * @default - undefined
     */
    readonly payloadFormatVersion: PayloadFormatVersion;
}
