"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionType = exports.PassthroughBehavior = exports.IntegrationType = exports.ContentHandling = exports.Integration = void 0;
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
        const options = this.props.options || {};
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        return {
            options: this.props.options,
            type: this.props.type,
            uri: this.props.uri,
            integrationHttpMethod: this.props.integrationHttpMethod,
        };
    }
}
exports.Integration = Integration;
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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