"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const crypto = require("crypto");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
        if (props.api instanceof restapi_1.RestApiBase) {
            props.api._attachDeployment(this);
        }
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Quoting from CloudFormation's docs:
     *
     *   If you create an AWS::ApiGateway::RestApi resource and its methods (using
     *   AWS::ApiGateway::Method) in the same template as your deployment, the
     *   deployment must depend on the RestApi's methods. To create a dependency,
     *   add a DependsOn attribute to the deployment. If you don't, AWS
     *   CloudFormation creates the deployment right after it creates the RestApi
     *   resource that doesn't contain any methods, and AWS CloudFormation
     *   encounters the following error: The REST API doesn't contain any methods.
     *
     * @param method The method to add as a dependency of the deployment
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
     * @see https://github.com/aws/aws-cdk/pull/6165
     * @internal
     */
    _addMethodDependency(method) {
        // adding a dependency between the constructs using `node.addDependency()`
        // will create additional dependencies between `AWS::ApiGateway::Deployment`
        // and the `AWS::Lambda::Permission` resources (children under Method),
        // causing cyclic dependency errors. Hence, falling back to declaring
        // dependencies between the underlying CfnResources.
        this.node.addDependency(method.node.defaultChild);
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = this.stack.getLogicalId(this);
        this.overrideLogicalId(core_1.Lazy.stringValue({ produce: () => this.calculateLogicalId() }));
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    calculateLogicalId() {
        const hash = [...this.hashComponents];
        if (this.api instanceof restapi_1.RestApi || this.api instanceof restapi_1.SpecRestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            hash.push(this.stack.resolve(cfnRestApiCF));
        }
        let lid = this.originalLogicalId;
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (hash.length > 0) {
            const md5 = crypto.createHash('md5');
            hash.map(x => this.stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            lid += md5.digest('hex');
        }
        return lid;
    }
}
//# sourceMappingURL=data:application/json;base64,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