"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RequestAuthorizer = exports.TokenAuthorizer = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("../apigateway.generated");
const authorizer_1 = require("../authorizer");
class LambdaAuthorizer extends authorizer_1.Authorizer {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.handler = props.handler;
        this.role = props.assumeRole;
        if (props.resultsCacheTtl && ((_a = props.resultsCacheTtl) === null || _a === void 0 ? void 0 : _a.toSeconds()) > 3600) {
            throw new Error('Lambda authorizer property \'resultsCacheTtl\' must not be greater than 3600 seconds (1 hour)');
        }
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     */
    _attachToApi(restApi) {
        if (this.restApiId && this.restApiId !== restApi.restApiId) {
            throw new Error('Cannot attach authorizer to two different rest APIs');
        }
        this.restApiId = restApi.restApiId;
    }
    /**
     * Sets up the permissions necessary for the API Gateway service to invoke the Lambda function.
     */
    setupPermissions() {
        if (!this.role) {
            this.handler.addPermission(`${this.node.uniqueId}:Permissions`, {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.authorizerArn,
            });
        }
        else if (this.role instanceof iam.Role) { // i.e. not imported
            this.role.attachInlinePolicy(new iam.Policy(this, 'authorizerInvokePolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: [this.handler.functionArn],
                        actions: ['lambda:InvokeFunction'],
                    }),
                ],
            }));
        }
    }
    /**
     * Returns a token that resolves to the Rest Api Id at the time of synthesis.
     * Throws an error, during token resolution, if no RestApi is attached to this authorizer.
     */
    lazyRestApiId() {
        return core_1.Lazy.stringValue({
            produce: () => {
                if (!this.restApiId) {
                    throw new Error(`Authorizer (${this.node.path}) must be attached to a RestApi`);
                }
                return this.restApiId;
            },
        });
    }
}
/**
 * Token based lambda authorizer that recognizes the caller's identity as a bearer token,
 * such as a JSON Web Token (JWT) or an OAuth token.
 * Based on the token, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class TokenAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'TOKEN',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySource || 'method.request.header.Authorization',
            identityValidationExpression: props.validationRegex,
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.TokenAuthorizer = TokenAuthorizer;
/**
 * Request-based lambda authorizer that recognizes the caller's identity via request parameters,
 * such as headers, paths, query strings, stage variables, or context variables.
 * Based on the request, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class RequestAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        if ((props.resultsCacheTtl === undefined || props.resultsCacheTtl.toSeconds() !== 0) && props.identitySources.length === 0) {
            throw new Error('At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        }
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'REQUEST',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySources.map(is => is.toString()).join(','),
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.RequestAuthorizer = RequestAuthorizer;
/**
 * constructs the authorizerURIArn.
 */
function lambdaAuthorizerArn(handler) {
    return `arn:${core_1.Stack.of(handler).partition}:apigateway:${core_1.Stack.of(handler).region}:lambda:path/2015-03-31/functions/${handler.functionArn}/invocations`;
}
//# sourceMappingURL=data:application/json;base64,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