"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BasicAuth = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
/**
 * Basic Auth configuration
 */
class BasicAuth {
    constructor(props) {
        this.props = props;
    }
    /**
     * Creates a Basic Auth configuration from a username and a password
     *
     * @param username The username
     * @param password The password
     */
    static fromCredentials(username, password) {
        return new BasicAuth({ username, password });
    }
    /**
     * Creates a Basic Auth configuration with a password generated in Secrets
     * Manager.
     *
     * @param username The username
     * @param encryptionKey The encryption key to use to encrypt the password in
     * Secrets Manager
     */
    static fromGeneratedPassword(username, encryptionKey) {
        return new BasicAuth({ username, encryptionKey });
    }
    /**
     * Binds this Basic Auth configuration to an App
     */
    bind(scope, id) {
        const config = {
            enableBasicAuth: true,
            username: this.props.username,
        };
        if (this.props.password) {
            return {
                ...config,
                password: this.props.password.toString(),
            };
        }
        const secret = new secretsmanager.Secret(scope, id, {
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: this.props.username }),
                generateStringKey: 'password',
            },
        });
        return {
            ...config,
            password: secret.secretValueFromJson('password').toString(),
        };
    }
}
exports.BasicAuth = BasicAuth;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmFzaWMtYXV0aC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImJhc2ljLWF1dGgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsMkRBQTJELENBQUMsOERBQThEO0FBeUMxSDs7R0FFRztBQUNILE1BQWEsU0FBUztJQXFCbEIsWUFBNkIsS0FBcUI7UUFBckIsVUFBSyxHQUFMLEtBQUssQ0FBZ0I7SUFBSSxDQUFDO0lBcEJ2RDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxlQUFlLENBQUMsUUFBZ0IsRUFBRSxRQUFxQjtRQUNqRSxPQUFPLElBQUksU0FBUyxDQUFDLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUNEOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMscUJBQXFCLENBQUMsUUFBZ0IsRUFBRSxhQUF3QjtRQUMxRSxPQUFPLElBQUksU0FBUyxDQUFDLEVBQUUsUUFBUSxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUVEOztPQUVHO0lBQ0ksSUFBSSxDQUFDLEtBQWdCLEVBQUUsRUFBVTtRQUNwQyxNQUFNLE1BQU0sR0FBRztZQUNYLGVBQWUsRUFBRSxJQUFJO1lBQ3JCLFFBQVEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVE7U0FDaEMsQ0FBQztRQUNGLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUU7WUFDckIsT0FBTztnQkFDSCxHQUFHLE1BQU07Z0JBQ1QsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRTthQUMzQyxDQUFDO1NBQ0w7UUFDRCxNQUFNLE1BQU0sR0FBRyxJQUFJLGNBQWMsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNoRCxvQkFBb0IsRUFBRTtnQkFDbEIsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDO2dCQUN2RSxpQkFBaUIsRUFBRSxVQUFVO2FBQ2hDO1NBQ0osQ0FBQyxDQUFDO1FBQ0gsT0FBTztZQUNILEdBQUcsTUFBTTtZQUNULFFBQVEsRUFBRSxNQUFNLENBQUMsbUJBQW1CLENBQUMsVUFBVSxDQUFDLENBQUMsUUFBUSxFQUFFO1NBQzlELENBQUM7SUFDTixDQUFDO0NBQ0o7QUEvQ0QsOEJBK0NDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMga21zIGZyb20gXCIuLi8uLi9hd3Mta21zXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3Mta21zJ1xuaW1wb3J0ICogYXMgc2VjcmV0c21hbmFnZXIgZnJvbSBcIi4uLy4uL2F3cy1zZWNyZXRzbWFuYWdlclwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLXNlY3JldHNtYW5hZ2VyJ1xuaW1wb3J0IHsgQ29uc3RydWN0LCBTZWNyZXRWYWx1ZSB9IGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIEJhc2ljQXV0aFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEJhc2ljQXV0aFByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgdXNlcm5hbWVcbiAgICAgKi9cbiAgICByZWFkb25seSB1c2VybmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBwYXNzd29yZFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBIFNlY3JldHMgTWFuYWdlciBnZW5lcmF0ZWQgcGFzc3dvcmRcbiAgICAgKi9cbiAgICByZWFkb25seSBwYXNzd29yZD86IFNlY3JldFZhbHVlO1xuICAgIC8qKlxuICAgICAqIFRoZSBlbmNyeXB0aW9uIGtleSB0byB1c2UgdG8gZW5jcnlwdCB0aGUgcGFzc3dvcmQgd2hlbiBpdCdzIGdlbmVyYXRlZFxuICAgICAqIGluIFNlY3JldHMgTWFuYWdlclxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBkZWZhdWx0IG1hc3RlciBrZXlcbiAgICAgKi9cbiAgICByZWFkb25seSBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG59XG4vKipcbiAqIEEgQmFzaWMgQXV0aCBjb25maWd1cmF0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQmFzaWNBdXRoQ29uZmlnIHtcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIHRvIGVuYWJsZSBCYXNpYyBBdXRoXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW5hYmxlQmFzaWNBdXRoOiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFRoZSB1c2VybmFtZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHVzZXJuYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHBhc3N3b3JkXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGFzc3dvcmQ6IHN0cmluZztcbn1cbi8qKlxuICogQmFzaWMgQXV0aCBjb25maWd1cmF0aW9uXG4gKi9cbmV4cG9ydCBjbGFzcyBCYXNpY0F1dGgge1xuICAgIC8qKlxuICAgICAqIENyZWF0ZXMgYSBCYXNpYyBBdXRoIGNvbmZpZ3VyYXRpb24gZnJvbSBhIHVzZXJuYW1lIGFuZCBhIHBhc3N3b3JkXG4gICAgICpcbiAgICAgKiBAcGFyYW0gdXNlcm5hbWUgVGhlIHVzZXJuYW1lXG4gICAgICogQHBhcmFtIHBhc3N3b3JkIFRoZSBwYXNzd29yZFxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUNyZWRlbnRpYWxzKHVzZXJuYW1lOiBzdHJpbmcsIHBhc3N3b3JkOiBTZWNyZXRWYWx1ZSkge1xuICAgICAgICByZXR1cm4gbmV3IEJhc2ljQXV0aCh7IHVzZXJuYW1lLCBwYXNzd29yZCB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ3JlYXRlcyBhIEJhc2ljIEF1dGggY29uZmlndXJhdGlvbiB3aXRoIGEgcGFzc3dvcmQgZ2VuZXJhdGVkIGluIFNlY3JldHNcbiAgICAgKiBNYW5hZ2VyLlxuICAgICAqXG4gICAgICogQHBhcmFtIHVzZXJuYW1lIFRoZSB1c2VybmFtZVxuICAgICAqIEBwYXJhbSBlbmNyeXB0aW9uS2V5IFRoZSBlbmNyeXB0aW9uIGtleSB0byB1c2UgdG8gZW5jcnlwdCB0aGUgcGFzc3dvcmQgaW5cbiAgICAgKiBTZWNyZXRzIE1hbmFnZXJcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21HZW5lcmF0ZWRQYXNzd29yZCh1c2VybmFtZTogc3RyaW5nLCBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXkpIHtcbiAgICAgICAgcmV0dXJuIG5ldyBCYXNpY0F1dGgoeyB1c2VybmFtZSwgZW5jcnlwdGlvbktleSB9KTtcbiAgICB9XG4gICAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogQmFzaWNBdXRoUHJvcHMpIHsgfVxuICAgIC8qKlxuICAgICAqIEJpbmRzIHRoaXMgQmFzaWMgQXV0aCBjb25maWd1cmF0aW9uIHRvIGFuIEFwcFxuICAgICAqL1xuICAgIHB1YmxpYyBiaW5kKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcpOiBCYXNpY0F1dGhDb25maWcge1xuICAgICAgICBjb25zdCBjb25maWcgPSB7XG4gICAgICAgICAgICBlbmFibGVCYXNpY0F1dGg6IHRydWUsXG4gICAgICAgICAgICB1c2VybmFtZTogdGhpcy5wcm9wcy51c2VybmFtZSxcbiAgICAgICAgfTtcbiAgICAgICAgaWYgKHRoaXMucHJvcHMucGFzc3dvcmQpIHtcbiAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgLi4uY29uZmlnLFxuICAgICAgICAgICAgICAgIHBhc3N3b3JkOiB0aGlzLnByb3BzLnBhc3N3b3JkLnRvU3RyaW5nKCksXG4gICAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHNlY3JldCA9IG5ldyBzZWNyZXRzbWFuYWdlci5TZWNyZXQoc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICBnZW5lcmF0ZVNlY3JldFN0cmluZzoge1xuICAgICAgICAgICAgICAgIHNlY3JldFN0cmluZ1RlbXBsYXRlOiBKU09OLnN0cmluZ2lmeSh7IHVzZXJuYW1lOiB0aGlzLnByb3BzLnVzZXJuYW1lIH0pLFxuICAgICAgICAgICAgICAgIGdlbmVyYXRlU3RyaW5nS2V5OiAncGFzc3dvcmQnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAuLi5jb25maWcsXG4gICAgICAgICAgICBwYXNzd29yZDogc2VjcmV0LnNlY3JldFZhbHVlRnJvbUpzb24oJ3Bhc3N3b3JkJykudG9TdHJpbmcoKSxcbiAgICAgICAgfTtcbiAgICB9XG59XG4iXX0=