
# Copyright 2020, Peter Oberhofer (pob90)
# Copyright 2020,2021 Stefan Valouch (svalouch)
# SPDX-License-Identifier: GPL-3.0-only

import logging
import struct
from typing import Union

from .exceptions import FrameCRCMismatch, InvalidCommand, FrameLengthExceeded
from .types import Command, FrameType
from .utils import CRC16

#: Token that starts a frame
START_TOKEN = b'+'
#: Token that escapes the next value
ESCAPE_TOKEN = b'-'
#: Length of the header
FRAME_LENGTH_HEADER = 1
#: Length of a command
FRAME_LENGTH_COMMAND = 1
#: Length of the length information
FRAME_LENGTH_LENGTH = 2
#: Length of a frame, contains 1 byte header, 1 byte command and 2 bytes length
FRAME_HEADER_WITH_LENGTH = FRAME_LENGTH_HEADER + FRAME_LENGTH_COMMAND + FRAME_LENGTH_LENGTH
#: Length of the CRC16 checkum
FRAME_LENGTH_CRC16 = 2

#: Amount of bytes we need to have a command
BUFFER_LEN_COMMAND = 2


def make_frame(command: Command, id: int, payload: bytes = b'', address: int = 0,
               frame_type: FrameType = FrameType.STANDARD) -> bytes:
    '''
    Crafts the byte-stream representing the input values. The result of this function can be sent as-is to the target
    device.

    `payload` is ignored for ``READ`` commands. and the `address` is ignored for ``STANDARD`` frames.

    For a variant which stores the input values as well as the output, see :class:`~rctclient.frame.SendFrame`.

    .. versionadded:: 0.0.2

    :param command: The command to transmit.
    :param id: The object ID to target.
    :param payload: The payload to be transmitted. Use :func:`~rctclient.utils.encode_value` to generate valid
       payloads.
    :param address: Address for plant communication (untested, ignored for standard communication).
    :param frame_type: The type if frame to transmit (standard or plant).

    :return: byte object ready to be sent to a device.
    '''
    # start with the command
    buf = bytearray(struct.pack('B', command))

    # add frame type and length of payload
    if command in [Command.LONG_WRITE, Command.LONG_RESPONSE]:
        buf += struct.pack('>H', frame_type + len(payload))  # 2 bytes
    else:
        buf += struct.pack('>B', frame_type + len(payload))  # 1 byte

    # add address for plants
    if frame_type == FrameType.PLANT:
        buf += struct.pack('>I', address)  # 4 bytes

    # add the ID
    buf += struct.pack('>I', id)  # 4 bytes

    # add the payload unless it's a READ
    if command != Command.READ:
        buf += payload  # N bytes

    # calculate and append the checksum
    crc16 = CRC16(buf)
    buf += struct.pack('>H', crc16)

    data = bytearray(struct.pack('c', START_TOKEN))

    # go over the buffer and inject escape tokens

    for byt in buf:
        byte = bytes([byt])
        if byte in [START_TOKEN, ESCAPE_TOKEN]:
            data += ESCAPE_TOKEN

        data += byte

    return data


class SendFrame:
    '''
    A container for data to be transmitted to the target device. Instances of this class keep the input values so they
    can be retrieved later, if that is not a requirement it's easier to use :func:`~rctclient.frame.make_frame` which
    is called by this class internally to generate the byte stream. The byte stream stored by this class is generated on
    initialization and can be retrieved at any time using the ``data`` property.

    The frame byte stream that is generated by this class is meant to be sent to a device. The receiving side is
    implemented in :class:`ReceiveFrame`.

    `payload` needs to be encoded before it can be transmitted. See :func:`~rctclient.utils.encode_value`. It is
    ignored for ``READ`` commands.

    `address` is used for ``PLANT`` frames and otherwise ignored, when queried later using the ``address`` property, 0
    is returned for non-PLANT frames.

    :param command: The command to transmit.
    :param id: The message id.
    :param payload: Optional payload (ignored for read commands).
    :param address: Address for plant communication (untested, ignored for non-PLANT frame types).
    :param frame_type: Type of frame (standard or plant).
    '''

    _command: Command
    _id: int
    _address: int
    _frame_type: FrameType
    _payload: bytes

    _data: bytes

    def __init__(self, command: Command, id: int, payload: bytes = b'', address: int = 0,
                 frame_type: FrameType = FrameType.STANDARD) -> None:
        self._command = command
        self._id = id
        self._frame_type = frame_type

        self._payload = payload if command != Command.READ else b''
        self._address = address if frame_type == FrameType.PLANT else 0

        self._data = make_frame(self._command, self._id, self._payload, self._address, self._frame_type)

    def __repr__(self) -> str:
        return f'<SendFrame(command={self._command}, id=0x{self._id:X}, payload=0x{self._payload.hex()})>'

    @property
    def data(self) -> bytes:
        '''
        Returns the data after encoding, ready to be sent over the socket.
        '''
        return self._data

    @property
    def command(self) -> Command:
        '''
        Returns the command.
        '''
        return self._command

    @property
    def id(self) -> int:
        '''
        Returns the object ID.
        '''
        return self._id

    @property
    def address(self) -> int:
        '''
        Returns the address for plant communication. Note that this returns 0 unless plant-communication was requested.
        '''
        return self._address

    @property
    def frame_type(self) -> FrameType:
        '''
        Returns the type of communication frame.
        '''
        return self._frame_type

    @property
    def payload(self) -> bytes:
        '''
        Returns the payload (input data). To get the result to send to a device, use ``data``. Note that this returns
        an empty byte stream for ``READ`` commands, regardless of any input.
        '''
        return self._payload


class ReceiveFrame:
    '''
    Structure to receive and decode data received from the RCT device. Each instance can only consume a single frame
    and a new one needs to be constructed to receive the next one. Frames are decoded incrementally, by feeding data
    to the ``consume()`` function as it arrives over the network. The function returns the amount of bytes it consumed,
    and it automatically stops consumption the frame has been received completely.

    Use ``complete()`` to determin if the frame has been received in its entirety.

    When the frame has been received completely (``complete()`` returns *True*), it extracts the CRC and compares it
    with its own calculated checksum. Unless ``ignore_crc_match`` is set, a
    :class:`~rctclient.exceptions.FrameCRCMismatch` is raised if the checksums do **not** match.

    If the command encoded in the frame is not supported or invalid, a :class:`~rctclient.exceptions.InvalidCommand`
    will be raised unconditionally during consumption.

    Both exceptions carry the amount of consumed bytes in a field as to allow users to remove the already consumed
    bytes from their input buffer in order to start consumption with a new ReceiveFrame instance for the next frame.

    Some of the fields (such as command, id, frame_type, ...) are populated if enough data has arrived, even before the
    checksum has been received and compared. Until ``complete()`` returns *True*, this data may not be valid, but it
    may hint towards invalid frames (invalid length being the most common problem).

    To decode the payload, use :func:`~rctclient.utils.decode_value`.

    .. note::

       The parsing deviates from the protocol in the following ways:

       * arbitrary data (instead of just a single ``0x00``) before a start byte is ignored.
       * the distinction between normal and long commands is ignored. No error is reported if a frame that should be a
         ``LONG_RESPONSE`` is received with ``RESPONSE``, for example.

    .. versionchanged:: 0.0.3

       * The frame type is detected from the consumed data and the ``frame_type`` parameter was removed.
       * Debug logging is now handled using the Python logging framework. ``debug()`` was dropped as it served no use
         anymore.
       * The ``decode()`` method was removed as the functionality was integrated into ``consume()``.

    .. versionadded:: 0.0.3

       The ``ignore_crc_match`` parameter prevents raising an exception on CRC mismatch. Use the ``crc_ok`` property to
       figure out if the CRC matched when setting the parameter to ``True``.

    :param ignore_crc_mismatch: If not set, no exception is raised when the CRC checksums do **not** match. Use
       ``crc_ok`` to figure out if they matched.
    '''
    # frame complete yet?
    _complete: bool
    # did the crc match?
    _crc_ok: bool
    # parser in escape mode?
    _escaping: bool
    # received crc16 checksum data
    _crc16: int
    # frame length
    _frame_length: int
    # frame type
    _frame_type: FrameType
    # data buffer
    _buffer: bytearray
    # command
    _command: Command
    # amount of bytes consumed
    _consumed_bytes: int

    _frame_header_length: int

    # ID, once decoded
    _id: int
    # raw received payload
    _data: bytes
    # address for plant frames
    _address: int

    _ignore_crc_mismatch: bool

    def __init__(self, ignore_crc_mismatch: bool = False) -> None:
        self._log = logging.getLogger(__name__ + '.ReceiveFrame')
        self._complete = False
        self._crc_ok = False
        self._escaping = False
        self._crc16 = 0
        self._frame_length = 0
        self._frame_type = FrameType._NONE
        self._command = Command._NONE
        self._buffer = bytearray()
        self._ignore_crc_mismatch = ignore_crc_mismatch
        self._consumed_bytes = 0

        # set initially to the minimum length a frame header (i.e. everything before the data) can be.
        # 1 byte start, 1 byte command, 1 byte length, no address, 4 byte ID
        self._frame_header_length = 1 + 1 + 1 + 0 + 4

        # output data
        self._id = 0
        self._data = b''
        self._address = 0

    def __repr__(self) -> str:
        return f'<ReceiveFrame(cmd={self.command.name}, id={self.id:x}, address={self.address:x}, ' \
               f'data={self.data.hex()})>'

    @property
    def address(self) -> int:
        '''
        Returns the address if the frame is a plant frame (``FrameType.PLANT``) or 0.

        :raises FrameNotComplete: If the frame has not been fully received.
        '''
        return self._address

    @property
    def command(self) -> Command:
        '''
        Returns the command.
        '''
        return self._command

    def complete(self) -> bool:
        '''
        Returns whether the frame has been received completely. If this returns True, do **not** ``consume()`` any more
        data with this instance, but instead create a new instance of this class for further consumption of data.
        '''
        return self._complete

    @property
    def consumed_bytes(self) -> int:
        '''
        Returns how many bytes the frame has consumed over its lifetime. This includes data that was consumed before
        the start of a frame was found, so the amount reported here may be larger than the amount of data that makes up
        the frame.

        .. versionadded:: 0.0.3
        '''
        return self._consumed_bytes

    @property
    def crc_ok(self) -> bool:
        '''
        Returns whether the CRC is valid. The value is only valid after a complete frame has arrived.

        .. versionadded:: 0.0.3
        '''
        return self._crc_ok

    @property
    def data(self) -> bytes:
        '''
        Returns the received data payload. This is empty if there has been no data received or the CRC did not match.
        '''
        return bytes(self._data)

    @property
    def frame_type(self) -> FrameType:
        '''
        Returns the frame type if enough data has been received to decode it, and ``FrameType._NONE`` otherwise.

        .. versionadded:: 0.0.3
        '''
        return self._frame_type

    @property
    def id(self) -> int:
        '''
        Returns the ID. If the frame has been received but the checksum does not match up, 0 is returned.
        '''
        return self._id

    @property
    def ignore_crc_mismatch(self) -> bool:
        '''
        Returns whether CRC mismatches are ignored during decoding.

        .. versionadded:: 0.0.3
        '''
        return self._ignore_crc_mismatch

    @ignore_crc_mismatch.setter
    def ignore_crc_mismatch(self, newval: bool) -> None:
        '''
        Changes whether CRC mismatches are ignored during decoding.

        .. versionadded:: 0.0.3
        '''
        self._ignore_crc_mismatch = newval

    @property
    def frame_length(self) -> int:
        '''
        Returns the length of the frame. This is ``0`` until the header containing the length field has been received.
        Note that this is not the length field of the protocol but rather the length of the frame in its entirety,
        from start byte to the end of the CRC.

        .. versionadded:: 0.0.3
        '''
        return self._frame_length

    def consume(self, data: Union[bytes, bytearray]) -> int:  # pylint: disable=too-many-branches,too-many-statements
        '''
        Consumes data until the frame is complete. Returns the number of consumed bytes. Exceptions raised also carry
        the amount of consumed bytes.

        :param data: Data to consume.
        :return: The amount of bytes consumed from the input data.
        :raises FrameCRCMismatch: If the checksum didn't match and ``ignore_crc_mismatch`` was not set.
        :raises FrameLengthExceeded: If the parser read past the frames advertised length.
        :raises InvalidCommand: If the command byte is invalid or can't be decoded (such as ``EXTENSION``).
        '''

        i = 0
        for d_byte in data:  # pylint: disable=too-many-nested-blocks  # ugly parser is ugly
            self._consumed_bytes += 1
            i += 1
            c = bytes([d_byte])  # pylint: disable=invalid-name
            self._log.debug('read: 0x%s', c.hex())

            # sync to start_token
            if len(self._buffer) == 0:
                self._log.debug('      buffer empty')
                if c == START_TOKEN:
                    self._log.debug('      start token found')
                    self._buffer += c
                continue

            if self._escaping:
                self._log.debug('      resetting escape')
                self._escaping = False
            else:
                if c == ESCAPE_TOKEN:
                    self._log.debug('      setting escape')
                    # set the escape mode and ignore the byte at hand.
                    self._escaping = True
                    continue

            self._buffer += c
            self._log.debug('      adding to buffer')

            blen = len(self._buffer)  # cache

            if blen == BUFFER_LEN_COMMAND:
                cmd = struct.unpack('B', bytes([self._buffer[1]]))[0]
                try:
                    self._command = Command(cmd)
                except ValueError as exc:
                    raise InvalidCommand(str(exc), cmd, i) from exc

                if self._command == Command.EXTENSION:
                    raise InvalidCommand('EXTENSION is not supported', cmd, i)

                self._log.debug('      have command: 0x%x, is_plant: %s', self._command,
                                Command.is_plant(self._command))
                if Command.is_plant(self._command):
                    self._frame_header_length += 4
                    self._log.debug('      plant frame, extending header length by 4 to %d', self._frame_header_length)
                if Command.is_long(self._command):
                    self._frame_header_length += 1
                    self._log.debug('      long cmd, extending header length by 1 to %d', self._frame_header_length)
            elif blen == self._frame_header_length:
                self._log.debug('      buffer length %d indicates that it contains entire header', blen)
                if Command.is_long(self._command):
                    data_length = struct.unpack('>H', self._buffer[2:4])[0]
                    address_idx = 4
                else:
                    data_length = struct.unpack('>B', bytes([self._buffer[2]]))[0]
                    address_idx = 3

                if Command.is_plant(self._command):
                    # length field includes address and id length == 8 bytes
                    self._frame_length = (self._frame_header_length - 8) + data_length + FRAME_LENGTH_CRC16
                    self._address = struct.unpack('>I', self._buffer[address_idx:address_idx + 4])[0]
                    oid_idx = address_idx + 4

                else:
                    # length field includes id length == 4 bytes
                    self._frame_length = (self._frame_header_length - 4) + data_length + FRAME_LENGTH_CRC16
                    oid_idx = address_idx

                self._log.debug('      data_length: %d bytes, frame_length: %d', data_length, self._frame_length)

                self._id = struct.unpack('>I', self._buffer[oid_idx:oid_idx + 4])[0]
                self._log.debug('      oid index: %d, OID: 0x%X', oid_idx, self._id)

            elif self._frame_length > 0 and blen == self._frame_length:
                self._log.debug('      buffer contains full frame')
                self._log.debug('      buffer: %s', self._buffer.hex())
                self._complete = True

                self._crc16 = struct.unpack('>H', self._buffer[-2:])[0]
                calc_crc16 = CRC16(self._buffer[len(START_TOKEN):-FRAME_LENGTH_CRC16])
                self._crc_ok = self._crc16 == calc_crc16
                self._log.debug('      crc: %04x calculated: %04x match: %s', self._crc16, calc_crc16, self._crc_ok)

                self._data = self._buffer[self._frame_header_length:-FRAME_LENGTH_CRC16]

                if not self._crc_ok and not self._ignore_crc_mismatch:
                    raise FrameCRCMismatch('CRC mismatch', self._crc16, calc_crc16, i)
                return i
            elif self._frame_length > 0 and blen > self._frame_length:
                raise FrameLengthExceeded('Frame consumed more than it should', i)
        return i
