"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const alias_1 = require("../lib/alias");
const key_1 = require("../lib/key");
module.exports = {
    'default alias'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new key_1.Key(stack, 'Key');
        new alias_1.Alias(stack, 'Alias', { targetKey: key, aliasName: 'alias/foo' });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Alias', {
            AliasName: 'alias/foo',
            TargetKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] }
        }));
        test.done();
    },
    'add "alias/" prefix if not given.'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new key_1.Key(stack, 'Key', {
            enableKeyRotation: true,
            enabled: false
        });
        new alias_1.Alias(stack, 'Alias', {
            aliasName: 'foo',
            targetKey: key
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Alias', {
            AliasName: 'alias/foo',
            TargetKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] }
        }));
        test.done();
    },
    'can create alias directly while creating the key'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        new key_1.Key(stack, 'Key', {
            enableKeyRotation: true,
            enabled: false,
            alias: 'foo',
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Alias', {
            AliasName: 'alias/foo',
            TargetKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] }
        }));
        test.done();
    },
    'fails if alias is "alias/" (and nothing more)'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new key_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false
        });
        test.throws(() => new alias_1.Alias(stack, 'Alias', {
            aliasName: 'alias/',
            targetKey: key
        }));
        test.done();
    },
    'fails if alias contains illegal characters'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new key_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false
        });
        test.throws(() => new alias_1.Alias(stack, 'Alias', {
            aliasName: 'alias/@Nope',
            targetKey: key
        }), 'a-zA-Z0-9:/_-');
        test.done();
    },
    'fails if alias starts with "alias/aws/"'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new key_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false
        });
        test.throws(() => new alias_1.Alias(stack, 'Alias1', {
            aliasName: 'alias/aws/',
            targetKey: key
        }), /Alias cannot start with alias\/aws\/: alias\/aws\//);
        test.throws(() => new alias_1.Alias(stack, 'Alias2', {
            aliasName: 'alias/aws/Awesome',
            targetKey: key
        }), /Alias cannot start with alias\/aws\/: alias\/aws\/Awesome/);
        test.throws(() => new alias_1.Alias(stack, 'Alias3', {
            aliasName: 'alias/AWS/awesome',
            targetKey: key
        }), /Alias cannot start with alias\/aws\/: alias\/AWS\/awesome/);
        test.done();
    },
    'can be used wherever a key is expected'(test) {
        const stack = new core_1.Stack();
        const myKey = new key_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false
        });
        const myAlias = new alias_1.Alias(stack, 'MyAlias', {
            targetKey: myKey,
            aliasName: 'alias/myAlias',
        });
        class MyConstruct extends core_1.Construct {
            constructor(scope, id, key) {
                super(scope, id);
                new core_1.CfnOutput(stack, 'OutId', {
                    value: key.keyId,
                });
                new core_1.CfnOutput(stack, 'OutArn', {
                    value: key.keyArn,
                });
            }
        }
        new MyConstruct(stack, 'MyConstruct', myAlias);
        const template = assert_1.SynthUtils.synthesize(stack).template.Outputs;
        test.deepEqual(template, {
            "OutId": {
                "Value": "alias/myAlias",
            },
            "OutArn": {
                "Value": {
                    "Fn::Join": ["", [
                            "arn:",
                            { Ref: "AWS::Partition" },
                            ":kms:",
                            { Ref: "AWS::Region" },
                            ":",
                            { Ref: "AWS::AccountId" },
                            ":alias/myAlias",
                        ]],
                },
            },
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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