#!/usr/bin/env python3
# -*- coding: utf-8 -*-

__author__ = ["Juliette Chabassier", "Augustin Ernoult", "Olivier Geber",
              "Alexis Thibault", "Tobias Van Baarsel"]
__copyright__ = "Copyright 2020, Inria"
__credits__ = ["Juliette Chabassier", "Augustin Ernoult", "Olivier Geber",
               "Alexis Thibault", "Tobias Van Baarsel"]
__license__ = "GPL 3.0"
__version__ = "0.4"
__email__ = "openwind-contact@inria.fr"
__status__ = "Dev"
"""
Base class for the shape of a pipe.
"""
from abc import ABC, abstractmethod

import numpy as np
import matplotlib.pyplot as plt

from openwind.design import eval_, diff_




class DesignShape(ABC):
    """
    Base class for the shape of a section of tube.

    Describe the radius profile r(x) of a length of pipe.
    """

    # --- ABSTRACT METHODS ---

    @abstractmethod
    def get_radius_at(self, x_norm):
        """
        Gives the value of the radius at the normalized positions.


        \(r(\\tilde{x})\)

        Parameters
        ---------
        x_norm : float, array of float
            The normalized position \( \\tilde{x}\) within 0 and 1 at which
            the radius is computed

        Return
        ------
        float, array of float
            The radii of the shape (same type of `x_norm`)

        """

    @abstractmethod
    def get_diff_radius_at(self, x_norm, diff_index):
        """
        Differentiate the radius w.r. to one parameter.

        .. math::
            \\frac{\\partial r(\\tilde{x})}{\\partial m_i}

        Gives the differentiate of the radius with respect to the
        optimization variable \(m_i\) at the normalized position
        \( \\tilde{x}\).



        Parameters
        ----------
        x_norm : float, array of float
            The normalized position \( \\tilde{x}\) within 0 and 1 at which the
            radius is computed
        diff_index : int
            The index at which the designated parameters is stored in the
            `openwind.design.design_parameter.OptimizationParameters`.

        Return
        -------
        float, array of float
            The value of the differentiate at each `x_norm`
        """

    def get_position_from_xnorm(self, x_norm):
        """
        Re-scaled the position.

        Give the position on the main bore pipe in meter from a scaled
        position.

        .. math::
            x = (x_{max} - x_{min}) \\tilde{x} + x_{min}

        with \(x_{max}, x_{min}\) the end points position of the considered
        pipe.

        Parameters
        ----------
        x_norm : float, array of float
            The normalized position \( \\tilde{x}\) within 0 and 1.

        Return
        ------
        float, array of float
            The physical position in meter.

        """
        Xmin, Xmax = eval_(self.get_endpoints_position())
        return x_norm*(Xmax - Xmin) + Xmin

    def get_xnorm_from_position(self, position):
        """
        Unscaled the position.

        Give the normalized position on the pipe from the physical position in
        meters.

        .. math::
            \\tilde{x} = \\frac{x - x_{min}}{x_{max} - x_{min}}

        with \(x_{max}, x_{min}\) the end points position of the considered
        pipe.

        Parameters
        ---------
        position : float, array of float
            The physical position in meter.

        Return
        ----------
        x_norm : float, array of float
            The normalized position \( \\tilde{x}\) within 0 and 1.


        """
        Xmin, Xmax = eval_(self.get_endpoints_position())
        return (position - Xmin) / (Xmax - Xmin)

    def get_diff_position_from_xnorm(self, x_norm, diff_index):
        """
        Differentiate the physical position w.r. to one parameter.

        .. math::
            \\frac{\\partial x(\\tilde{x})}{\\partial m_i} = \
                \\frac{\\partial x_{min}}{\\partial m_i} (1 - \\tilde{x}) + \
                    \\frac{\\partial x_{max}}{\\partial m_i}  \\tilde{x}

        Gives the differentiate of the physical position \(x\) with respect to
        the optimization variable \(m_i\) at the normalized position
        \( \\tilde{x}\).



        Parameters
        ----------
        x_norm : float, array of float
            The normalized position \( \\tilde{x}\) within 0 and 1
        diff_index : int
            The index at which the designated parameters is stored in the
            `openwind.design.design_parameter.OptimizationParameters`.

        Return
        -------
        float, array of float
            The value of the differentiate at each `x_norm`
        """
        dXmin, dXmax = diff_(self.get_endpoints_position(), diff_index)
        return dXmin*(1-x_norm) + dXmax*x_norm

    @abstractmethod
    def get_endpoints_position(self):
        """
        The design parameters corresponding to the endpoints position.

        Return
        ------
        openwind.design.design_parameter.DesignParameters
            The two paramaters corresponding to the endpoints position
            (abscissa in meters).
        """

    @abstractmethod
    def get_endpoints_radius(self):
        """
        The design parameters corresponding to the endpoints radii.

        Return
        ------
        openwind.design.design_parameter.DesignParameters
            The two paramaters corresponding to the endpoints radii of the
            shape (in meters).
        """

    def get_diff_shape_wr_x_norm(self, x_norm):
        """
        Differentiate the radius w.r. to the scaled position.

        .. math::
            \\frac{\\partial r(\\tilde{x})}{\\partial \\tilde{x}} \
        with \(r\) the radius and \(\\tilde{x}\) the normalized position.

        Parameters
        ----------
        x_norm : float, array of float
            The normalized position \( \\tilde{x}\) within 0 and 1.

        Return
        ------
        float, array of float
            The derivative
        """
        raise NotImplementedError

    def __repr__(self):
        """Allows readable printing of a list of shapes"""
        positions = self.get_endpoints_position()
        radii = self.get_endpoints_radius()
        return (('{class_}(length={length}cm, bounds positions=[{pos0}, {pos1}]'
                'm, bounds radii=[{rad0}, {rad1}]m)')
                .format(class_=type(self).__name__,
                        length=100*self.get_length(), pos0=positions[0],
                        pos1=positions[1], rad0=radii[0], rad1=radii[1]))

    # --- COMMON METHODS ---

    @staticmethod
    def check_bounds(x, bounds):
        """
        Ensure x is within the given bounds.

        Parameters
        ----------
        x : array
        bounds : list of two floats

        """
        tolerance = 1e-10
        if np.any(x < bounds[0] - tolerance) or np.any(x > bounds[-1] + tolerance):
            raise ValueError('A value is being estimated outside the shape!'
                             'x in [{:.2e},{:.2e}], whereas bounds are '
                             '[{:.2e},{:.2e}]'.format(np.min(x), np.max(x),
                                                      bounds[0], bounds[-1]))

    def get_length(self):
        """
        Physical length of the pipe in meters.
        """
        X_range = self.get_endpoints_position()
        Xmin, Xmax = eval_(list(X_range))
        return Xmax - Xmin

    def get_diff_length(self, diff_index):
        """
        Differentiate the length w.r.t. an optimization parameter.

        Parameters
        ----------
        diff_index : int
            The index at which the designated parameters is stored in the
            `openwind.design.design_parameter.OptimizationParameters`.

        """
        X_range = self.get_endpoints_position()
        dXmin, dXmax = diff_(list(X_range), diff_index)
        return dXmax - dXmin

    def plot_shape(self, **kwargs):
        """Display this DesignShape.

        Keyword arguments are passed directly to `plt.plot()`.
        """
        x_norm = np.linspace(0, 1, 100)
        x = self.get_position_from_xnorm(x_norm)
        r = self.get_radius_at(x_norm)
        plt.plot(x, r, **kwargs)
