#!/usr/bin/env python3
# -*- coding: utf-8 -*-

__author__ = ["Juliette Chabassier", "Augustin Ernoult", "Olivier Geber",
              "Alexis Thibault", "Tobias Van Baarsel"]
__copyright__ = "Copyright 2020, Inria"
__credits__ = ["Juliette Chabassier", "Augustin Ernoult", "Olivier Geber",
               "Alexis Thibault", "Tobias Van Baarsel"]
__license__ = "GPL 3.0"
__version__ = "0.4"
__email__ = "openwind-contact@inria.fr"
__status__ = "Dev"
"""
Pipe the radius of which follows a circle equation: constant curvature.
"""
import numpy as np

from openwind.design import DesignShape, eval_, diff_


def __compute_center_coor(x1, x2, y1, y2, R):
    """
    Determine the center coordinates of the circle.

    The circle is defined from two points and its radius.

    Parameters
    ----------
    x1, x2, y1, y2 : floats
        Coordinates of the two points.
    R : float
        Radius of the circle.

    Returns
    -------
    Xc, Yc : floats
        Coordinates of the center.

    """
    error_msg = ('The diameter of the circle must be larger than the length of'
                 ' the pipe. Here: {}<{}'.format(2*np.abs(R), np.abs(x2 - x1)))
    assert (2*np.abs(R) > np.abs(x2 - x1)), error_msg

    Deltax = x2 - x1
    Deltay = y2 - y1
    if Deltay != 0:
        u = x1**2 - x2**2 + y1**2 - y2**2

        a = 1 + (Deltax**2) / (Deltay**2)
        b = 2 * y1 * Deltax / Deltay - 2 * x1 + u * Deltax / (Deltay**2)
        c = u**2 / (4 * Deltay**2) + y1 * u / Deltay + y1**2 + x1**2 - R**2

        Xc1 = (-b - np.sqrt(b**2 - 4 * a * c)) / (2 * a)
        Xc2 = (-b + np.sqrt(b**2 - 4 * a * c)) / (2 * a)
        Yc1 = -(u + 2 * Deltax * Xc1) / (2 * Deltay)
        Yc2 = -(u + 2 * Deltax * Xc2) / (2 * Deltay)
        if np.sign(R)*Yc1 >= np.sign(R)*max(y1, y2):
            Xc = Xc1
            Yc = Yc1
        else:
            Xc = Xc2
            Yc = Yc2
    else:  # If Deltay = 0 it is a particular case not to divide by 0
        Xc = 0.5 * (x1 + x2)
        Yc = y1 + np.sign(R)*np.sqrt(R**2 - 0.25 * (x2 - x1)**2)
    return Xc, Yc


def circle(x, x1, x2, y1, y2, R):
    """Calculate images with a circular function between 2 points.

    Parameters
    ----------
    x : float
        the point at which the value of y is calculated
    x1, r1 : float
        the first point
    x2, r2 : float
        the second point
    R : float or list
        the curvature radius
    """
    Xc, Yc = __compute_center_coor(x1, x2, y1, y2, R)
    return Yc - np.sign(R)*np.sqrt(R**2 - (x - Xc)**2)


def dcircle_dx(x, x1, x2, y1, y2, R):
    Xc, _ = __compute_center_coor(x1, x2, y1, y2, R)
    return (x - Xc)*np.sign(R)/np.sqrt(R**2 - (x - Xc)**2)


def diff_circle(x, x1, x2, y1, y2, R, dx1, dx2, dy1, dy2, dR):
    """Differentitate the circular equation wrt the different parameters.

    Parameters
    ----------
    x : float
        the point at which the value of y is calculated
    x1, r1 : float
        the first point
    x2, r2 : float
        the second point
    R : float or list
        the curvature radius
    """
    if np.abs(R) < 0.5*np.abs(x2 - x1):
        raise ValueError('The diameter of the circle must be larger than the'
                         ' length of the pipe. Here: '
                         '{}<{}'.format(2*np.abs(R), np.abs(x2 - x1)))
    Deltax = x2 - x1
    Deltay = y2 - y1
    d_Deltax = dx2 - dx1
    d_Deltay = dy2 - dy1

    if Deltay != 0:
        u = x1**2 - x2**2 + y1**2 - y2**2
        du = 2*dx1*x1 - 2*dx2*x2 + 2*dy1*y1 - 2*dy2*y2

        a = 1 + (Deltax**2) / (Deltay**2)
        da = 2*d_Deltax*Deltax/(Deltay**2) - 2*d_Deltay*Deltax**2/(Deltay**3)

        b = 2*y1*Deltax/Deltay - 2*x1 + u*Deltax/(Deltay**2)
        db = (2*dy1*Deltax/Deltay - 2*dx1 + du*Deltax/(Deltay**2)
              + d_Deltax*(2*y1/Deltay + u/Deltay**2)
              - d_Deltay*(2*y1*Deltax/(Deltay**2) + 2*u*Deltax/(Deltay**3)))

        c = u**2/(4*Deltay**2) + y1*u/Deltay + y1**2 + x1**2 - R**2
        dc = (du*(u/(2*Deltay**2) + y1/Deltay) + dy1*(u/Deltay + 2*y1)
              + 2*dx1*x1 - 2*dR*R - d_Deltay*(u**2/(2*Deltay**3)
                                              + y1*u/Deltay**2))

        discr = np.sqrt(b**2 - 4 * a * c)
        d_discr = 0.5*(2*db*b - 4*da*c - 4*a*dc)/discr

        Xc1 = (-b - discr) / (2 * a)
        Xc2 = (-b + discr) / (2 * a)
        dXc1 = (-db - d_discr)/(2*a) - da/a*Xc1
        dXc2 = (-db + d_discr)/(2*a) - da/a*Xc2

        Yc1 = -(u + 2 * Deltax * Xc1) / (2 * Deltay)
        Yc2 = -(u + 2 * Deltax * Xc2) / (2 * Deltay)
        dYc1 = (-(du + 2*d_Deltax*Xc1 + 2*Deltax*dXc1)/(2*Deltay)
                - d_Deltay/Deltay*Yc1)
        dYc2 = (-(du + 2*d_Deltax*Xc2 + 2*Deltax*dXc2)/(2*Deltay)
                - d_Deltay/Deltay*Yc2)

        if np.sign(R)*Yc1 >= np.sign(R)*max(y1, y2):
            Xc = Xc1
            Yc = Yc1
            dXc = dXc1
            dYc = dYc1
        else:
            Xc = Xc2
            Yc = Yc2
            dXc = dXc2
            dYc = dYc2
    else:  # If Deltay = 0 it is a particular case not to divide by 0
        Xc = 0.5 * (x1 + x2)
        Yc = y1 + np.sign(R)*np.sqrt(R**2 - 0.25 * Deltax**2)
        dXc = 0.5*(dx1 + dx2)
        dYc = (dy1 + np.sign(R)*0.5*(2*dR*R - 0.5*d_Deltax*Deltax)
               / np.sqrt(R**2 - 0.25*Deltax**2))

    y = Yc - np.sign(R)*np.sqrt(R**2 - (x - Xc)**2)
    dy = (dYc - 0.5*np.sign(R)*(2*dR*R + 2*dXc*(x - Xc))
          / np.sqrt(R**2 - (x - Xc)**2))
    return dy

class Circle(DesignShape):
    """
    Pipe the radius of which follows a circle equation: constant curvature.

    The circle equation is obtained from the coordinates of two points
    \(x_1, r_1\) and \(x_2, r_2\) and the curvature radius \(R\).

    Parameters
    ----------
    *params: 5 openwind.design.design_parameter.DesignParameter
        The five parameters in this order: \(x_1, x_2, r_1, r_2, R\)

    """

    def __init__(self, *params):
        if len(params) != 5:
            raise ValueError("A circular shape need 5 parameters.")
        self.params = params

    def __str__(self):
        geom = ''
        for param in self.params[:-1]:
            geom += '{} '.format(param)
        return '{geom}{class_} {parameter}'.format(geom=geom,
                                                   parameter=self.params[-1],
                                                   class_=type(self).__name__)

    def get_position_from_xnorm(self, x_norm):
        Xmin = self.params[0].get_value()
        Xmax = self.params[1].get_value()
        return x_norm*(Xmax - Xmin) + Xmin

    def get_radius_at(self, x_norm):
        x1, x2, r1, r2, R = eval_(self.params)
        x = self.get_position_from_xnorm(x_norm)
        radius = circle(x, x1, x2, r1, r2, R)
        self.check_bounds(x, [x1, x2])
        return radius

    def get_diff_radius_at(self, x_norm, diff_index):
        x1, x2, r1, r2, R = eval_(self.params)
        dx1, dx2, dr1, dr2, dR = diff_(self.params, diff_index)
        dx_norm = self.get_diff_position_from_xnorm(x_norm, diff_index)
        x = self.get_position_from_xnorm(x_norm)
        self.check_bounds(x, [x1, x2])
        diff_radius = dcircle_dx(x, x1, x2, r1, r2, R)*dx_norm
        diff_radius += diff_circle(x, x1, x2, r1, r2, R, dx1, dx2, dr1, dr2, dR)
        return diff_radius

    def get_endpoints_position(self):
        return self.params[0], self.params[1]

    def get_endpoints_radius(self):
        return self.params[2], self.params[3]

    def get_diff_shape_wr_x_norm(self, x_norm):
        x1, x2, r1, r2, R = eval_(self.params)
        x = self.get_position_from_xnorm(x_norm)
        self.check_bounds(x, [x1, x2])
        dradius = dcircle_dx(x, x1, x2, r1, r2, R)
        return dradius * self.get_length()
