import argparse
import logging
import signal
import sys
import textwrap

from . import image
from . import transient
from . import utils
from . import __version__
from typing import List, Any, Optional


def parse_arguments() -> argparse.Namespace:
    # Parent parser for common arguments between the subcommands
    common_parser = argparse.ArgumentParser(add_help=False)
    common_parser.add_argument(
        '-image', metavar='IMG', nargs='+', action=ExtendAction,
        help='Disk image to use (this option can be repeated)')
    common_parser.add_argument(
        '-image-frontend',
        help='The location to place per-vm disk images')
    common_parser.add_argument(
        '-image-backend',
        help='The location to place the shared, read-only backing disk images')

    parser = argparse.ArgumentParser(prog='transient',
                                     description='A simple libvirt/vagrant alternative',
                                     allow_abbrev=False)

    parser.add_argument('-version', '--version', action='version',
                        version='%(prog)s {}'.format(__version__))

    parser.add_argument('-v', '--verbose', action='count', default=0,
                        help="Verbosity level (repeat to be more verbose)")

    subparsers = parser.add_subparsers(dest='command')

    # If we use the default 'usage' generated by argparse, it doesn't show
    # the '--' (though it does work). To avoid confusion, just use our own.
    run_parser = subparsers.add_parser('run',
                                       parents=[common_parser],
                                       usage="transient run [OPTIONS] -- [ARG [ARG ...]]",
                                       help='Run a transient virtual machine')
    run_parser.add_argument('-name', help='Set the vm name', required=True)

    run_parser.add_argument('-ssh-console', '-ssh', action='store_const',
                            default=False, const=True,
                            help='Use an ssh connection instead of the serial console')

    run_parser.add_argument('-ssh-with-serial', '-sshs',
                            default=False, const=True, action='store_const',
                            help='Show the serial output before SSH connects (implies -ssh)')

    run_parser.add_argument('-ssh-user', '-u', default="vagrant", help='User to pass to SSH')
    run_parser.add_argument('-ssh-bin-name', default="ssh", help='SSH binary to use')
    run_parser.add_argument('-ssh-timeout', default=90, type=int,
                            help='Time to wait for SSH connection before failing')
    run_parser.add_argument('-ssh-port', help='Local port the 22 of the guest is forwarded to')
    run_parser.add_argument('-ssh-command', '-cmd', help='Run an ssh command instead of a console')

    run_parser.add_argument('-sync-before', '-b', nargs='+', action=ExtendAction,
                            help='Sync a host path to a guest path before starting the guest')

    run_parser.add_argument('-sync-after', '-a', nargs='+', action=ExtendAction,
                            help='Sync a guest path to a host path after stopping the guest')

    run_parser.add_argument('-shutdown-timeout', type=int, default=20,
                            help='The time to wait for shutdown before terminating QEMU')

    run_parser.add_argument(
        '-shared-folder', '-s', nargs='+', action=ExtendAction, default=[],
        help='Share a host directory with the guest (/path/on/host:/path/on/guest)')

    run_parser.add_argument('-prepare-only', action='store_const', const=True, default=False,
                            help='Only download/create vm disks. Do not start the vm')

    run_parser.add_argument('qemu_args', metavar='ARG', nargs='*',
                            help='Arguments passed directly to QEMU')

    delete_parser = subparsers.add_parser('delete', help='Delete transient disks',
                                          parents=[common_parser])
    delete_parser.add_argument('-name', help='Delete images associated with the given vm name')
    delete_parser.add_argument('-force', '-f', help='Do not prompt before deletion',
                               action='store_const', const=True, default=False)

    list_parser = subparsers.add_parser('list', help='List transient disk information',
                                        parents=[common_parser])
    list_parser.add_argument('-name', help='List disks associated with the given vm name')

    return parser.parse_args()


class ExtendAction(argparse._AppendAction):
    def _copy_items(self, items: List[Any]) -> List[Any]:
        if items is None:
            return []
        import copy
        return copy.copy(items)

    # The 'extend' action is not available until 3.8. Backport that here
    # Mostly taken from https://github.com/python/cpython/blob/master/Lib/argparse.py#L1216
    def __call__(self, parser: argparse.ArgumentParser,
                 namespace: argparse.Namespace,
                 values: Any, option_string: Optional[str] = None) -> None:
        items = getattr(namespace, self.dest, None)
        items = self._copy_items(items)
        items.extend(values)
        setattr(namespace, self.dest, items)


def _find_requested_images(store: image.ImageStore,
                           args: argparse.Namespace) -> List[image.BaseImageInfo]:
    images: List[image.BaseImageInfo] = []
    if args.name is not None:
        if args.image is None:
            images = list(store.frontend_image_list(args.name))
        else:
            for image_identifier in args.image:
                images.extend(store.frontend_image_list(args.name, image_identifier))
    else:
        if args.image is None:
            images = list(store.backend_image_list())
            images.extend(store.frontend_image_list())
        else:
            for image_identifier in args.image:
                images.extend(store.backend_image_list(image_identifier))
                images.extend(store.frontend_image_list(image_identifier=image_identifier))
    return images


def _list_handler(store: image.ImageStore, args: argparse.Namespace) -> int:
    images = _find_requested_images(store, args)

    if len(images) == 0:
        print("No images match selection", file=sys.stderr)
        return 1

    frontend, backend = image.format_image_table(images)
    if len(frontend) > 0:
        print("Frontend Images:")
        print(frontend)
    if len(backend) > 0:
        print("\nBackend Images:")
        print(backend)
    return 0


def _delete_handler(store: image.ImageStore, args: argparse.Namespace) -> int:
    images = _find_requested_images(store, args)

    if len(images) == 0:
        print("No images match selection", file=sys.stderr)
        return 1

    print("The following images will be deleted:\n")
    frontend, backend = image.format_image_table(images)
    if len(frontend) > 0:
        print("Frontend Images:")
        print(frontend)
    if len(backend) > 0:
        print("\nBackend Images:")
        print(backend)

    if args.force is False:
        response = utils.prompt_yes_no("Proceed?", default=False)
    else:
        response = True

    if response is False:
        return 0

    for image_info in images:
        logging.info("Deleting image at {}".format(image_info.path))
        store.delete_image(image_info)
    return 0


def sigint_handler(sig, frame):
    logging.warning("transient process received SIGINT")
    sys.exit(1)


def main() -> None:
    signal.signal(signal.SIGINT, sigint_handler)
    args = parse_arguments()

    log_level = logging.ERROR
    if args.verbose == 1:
        log_level = logging.WARNING
    elif args.verbose == 2:
        log_level = logging.INFO
    elif args.verbose >= 3:
        log_level = logging.DEBUG
    logging.basicConfig(level=log_level, format='%(asctime)s:%(levelname)s:%(message)s',
                        datefmt='%Y-%m-%d %H:%M:%S')

    logging.debug("Parsed arguments: {}".format(args))

    store = image.ImageStore(backend_dir=args.image_backend,
                             frontend_dir=args.image_frontend)

    if args.command == "run":
        trans = transient.TransientVm(config=args, store=store)
        returncode = trans.run()
    elif args.command == "delete":
        returncode = _delete_handler(store, args)
    elif args.command == "list":
        returncode = _list_handler(store, args)
    sys.exit(returncode)
