#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
https://en.wikipedia.org/wiki/List_of_CIL_instructions
"""
from refinery.lib.dotnet.disassembler.model import (
    CallSiteDescr,
    Class,
    Ctor,
    Etype,
    Field,
    Float32,
    Float64,
    Int8,
    Int32,
    Int64,
    Method,
    Op,
    OpType,
    String,
    ThisType,
    Token,
    Type,
    TypeTok,
    UInt8,
    UInt16,
    UInt32,
    ValueType,
)


class OpRepository:
    INSTRUCTIONS = [
        Op(
            OpType.BASE,
            b'\x58',
            'add',
            [],
            'Add two values, returning a new value.',
        ),
        Op(
            OpType.BASE,
            b'\xd6',
            'add.ovf',
            [],
            'Add signed integer values with overflow check.',
        ),
        Op(
            OpType.BASE,
            b'\xd7',
            'add.ovf.un',
            [],
            'Add unsigned integer values with overflow check.',
        ),
        Op(
            OpType.BASE,
            b'\x5f',
            'and',
            [],
            'Bitwise AND of two integral values, returns an integral value.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x00',
            'arglist',
            [],
            'Return argument list handle for the current method.',
        ),
        Op(
            OpType.BASE,
            b'\x3b',
            'beq',
            [Int32(has_target=True)],
            'Branch to target if equal.',
        ),
        Op(
            OpType.BASE,
            b'\x2e',
            'beq.s',
            [Int8(has_target=True)],
            'Branch to target if equal, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x3c',
            'bge',
            [Int32(has_target=True)],
            'Branch to target if greater than or equal to.',
        ),
        Op(
            OpType.BASE,
            b'\x2f',
            'bge.s',
            [Int8(has_target=True)],
            'Branch to target if greater than or equal to, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x41',
            'bge.un',
            [Int32(has_target=True)],
            'Branch to target if greater than or equal to (unsigned or unordered).',
        ),
        Op(
            OpType.BASE,
            b'\x34',
            'bge.un.s',
            [Int8(has_target=True)],
            'Branch to target if greater than or equal to (unsigned or unordered), short form.',
        ),
        Op(
            OpType.BASE,
            b'\x3d',
            'bgt',
            [Int32(has_target=True)],
            'Branch to target if greater than.',
        ),
        Op(
            OpType.BASE,
            b'\x30',
            'bgt.s',
            [Int8(has_target=True)],
            'Branch to target if greater than, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x42',
            'bgt.un',
            [Int32(has_target=True)],
            'Branch to target if greater than (unsigned or unordered).',
        ),
        Op(
            OpType.BASE,
            b'\x35',
            'bgt.un.s',
            [Int8(has_target=True)],
            'Branch to target if greater than (unsigned or unordered), short form.',
        ),
        Op(
            OpType.BASE,
            b'\x3e',
            'ble',
            [Int32(has_target=True)],
            'Branch to target if less than or equal to.',
        ),
        Op(
            OpType.BASE,
            b'\x31',
            'ble.s',
            [Int8(has_target=True)],
            'Branch to target if less than or equal to, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x43',
            'ble.un',
            [Int32(has_target=True)],
            'Branch to target if less than or equal to (unsigned or unordered).',
        ),
        Op(
            OpType.BASE,
            b'\x36',
            'ble.un.s',
            [Int8(has_target=True)],
            'Branch to target if less than or equal to (unsigned or unordered), short form.',
        ),
        Op(
            OpType.BASE,
            b'\x3f',
            'blt',
            [Int32(has_target=True)],
            'Branch to target if less than.',
        ),
        Op(
            OpType.BASE,
            b'\x32',
            'blt.s',
            [Int8(has_target=True)],
            'Branch to target if less than, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x44',
            'blt.un',
            [Int32(has_target=True)],
            'Branch to target if less than (unsigned or unordered).',
        ),
        Op(
            OpType.BASE,
            b'\x37',
            'blt.un.s',
            [Int8(has_target=True)],
            'Branch to target if less than (unsigned or unordered), short form.',
        ),
        Op(
            OpType.BASE,
            b'\x40',
            'bne.un',
            [Int32(has_target=True)],
            'Branch to target if unequal or unordered.',
        ),
        Op(
            OpType.BASE,
            b'\x33',
            'bne.un.s',
            [Int8(has_target=True)],
            'Branch to target if unequal or unordered, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x38',
            'br',
            [Int32(has_target=True)],
            'Branch to target.',
        ),
        Op(
            OpType.BASE,
            b'\x2b',
            'br.s',
            [Int8(has_target=True)],
            'Branch to target, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x01',
            'break',
            [],
            'Inform a debugger that a breakpoint has been reached.',
        ),
        Op(
            OpType.BASE,
            b'\x39',
            'brfalse',
            [Int32(has_target=True)],
            'Branch to target if value is zero (false).',
        ),
        Op(
            OpType.BASE,
            b'\x2c',
            'brfalse.s',
            [Int8(has_target=True)],
            'Branch to target if value is zero (false), short form.',
        ),
        Op(
            OpType.BASE,
            b'\x3a',
            'brinst',
            [Int32(has_target=True)],
            'Branch to target if value is a non-null object reference (alias for brtrue).',
            is_alias=True,
        ),
        Op(
            OpType.BASE,
            b'\x2d',
            'brinst.s',
            [Int8(has_target=True)],
            'Branch to target if value is a non-null object reference, short form (alias for brtrue.s).',
            is_alias=True,
        ),
        Op(
            OpType.BASE,
            b'\x39',
            'brnull',
            [Int32(has_target=True)],
            'Branch to target if value is null (alias for brfalse).',
            is_alias=True,
        ),
        Op(
            OpType.BASE,
            b'\x2c',
            'brnull.s',
            [Int8(has_target=True)],
            'Branch to target if value is null (alias for brfalse.s), short form.',
            is_alias=True,
        ),
        Op(
            OpType.BASE,
            b'\x3a',
            'brtrue',
            [Int32(has_target=True)],
            'Branch to target if value is non-zero (true).',
        ),
        Op(
            OpType.BASE,
            b'\x2d',
            'brtrue.s',
            [Int8(has_target=True)],
            'Branch to target if value is non-zero (true), short form.',
        ),
        Op(
            OpType.BASE,
            b'\x39',
            'brzero',
            [Int32(has_target=True)],
            'Branch to target if value is zero (alias for brfalse).',
            is_alias=True,
        ),
        Op(
            OpType.BASE,
            b'\x2c',
            'brzero.s',
            [Int8(has_target=True)],
            'Branch to target if value is zero (alias for brfalse.s), short form.',
            is_alias=True,
        ),
        Op(
            OpType.BASE,
            b'\x28',
            'call',
            [Method()],
            'Call method described by method.',
        ),
        Op(
            OpType.BASE,
            b'\x29',
            'calli',
            [CallSiteDescr()],
            'Call method indicated on the stack with arguments described by callsitedescr.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x01',
            'ceq',
            [],
            'Push 1 (of type int32) if value1 equals value2, else push 0.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x02',
            'cgt',
            [],
            'Push 1 (of type int32) if value1 greater than value2, else push 0.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x03',
            'cgt.un',
            [],
            'Push 1 (of type int32) if value1 greater than value2, unsigned or unordered, else push 0.',
        ),
        Op(
            OpType.BASE,
            b'\xc3',
            'ckfinite',
            [],
            'Throw ArithmeticException if value is not a finite number.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x04',
            'clt',
            [],
            'Push 1 (of type int32) if value1 lower than value2, else push 0.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x05',
            'clt.un',
            [],
            'Push 1 (of type int32) if value1 lower than value2, unsigned or unordered, else push 0.',
        ),
        Op(
            OpType.BASE,
            b'\xd3',
            'conv.i',
            [],
            'Convert to native int, pushing native int on stack.',
        ),
        Op(
            OpType.BASE,
            b'\x67',
            'conv.i1',
            [],
            'Convert to int8, pushing int32 on stack.',
        ),
        Op(
            OpType.BASE,
            b'\x68',
            'conv.i2',
            [],
            'Convert to int16, pushing int32 on stack.',
        ),
        Op(
            OpType.BASE,
            b'\x69',
            'conv.i4',
            [],
            'Convert to int32, pushing int32 on stack.',
        ),
        Op(
            OpType.BASE,
            b'\x6a',
            'conv.i8',
            [],
            'Convert to int64, pushing int64 on stack.',
        ),
        Op(
            OpType.BASE,
            b'\xd4',
            'conv.ovf.i',
            [],
            'Convert to a native int (on the stack as native int) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x8a',
            'conv.ovf.i.un',
            [],
            'Convert unsigned to a native int (on the stack as native int) and throw an exception on '
            'overflow.',
        ),
        Op(
            OpType.BASE,
            b'\xb3',
            'conv.ovf.i1',
            [],
            'Convert to an int8 (on the stack as int32) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x82',
            'conv.ovf.i1.un',
            [],
            'Convert unsigned to an int8 (on the stack as int32) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\xb5',
            'conv.ovf.i2',
            [],
            'Convert to an int16 (on the stack as int32) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x83',
            'conv.ovf.i2.un',
            [],
            'Convert unsigned to an int16 (on the stack as int32) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\xb7',
            'conv.ovf.i4',
            [],
            'Convert to an int32 (on the stack as int32) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x84',
            'conv.ovf.i4.un',
            [],
            'Convert unsigned to an int32 (on the stack as int32) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\xb9',
            'conv.ovf.i8',
            [],
            'Convert to an int64 (on the stack as int64) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x85',
            'conv.ovf.i8.un',
            [],
            'Convert unsigned to an int64 (on the stack as int64) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\xd5',
            'conv.ovf.u',
            [],
            'Convert to a native unsigned int (on the stack as native int) and throw an exception on '
            'overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x8b',
            'conv.ovf.u.un',
            [],
            'Convert unsigned to a native unsigned int (on the stack as native int) and throw an exception '
            'on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\xb4',
            'conv.ovf.u1',
            [],
            'Convert to an unsigned int8 (on the stack as int32) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x86',
            'conv.ovf.u1.un',
            [],
            'Convert unsigned to an unsigned int8 (on the stack as int32) and throw an exception on '
            'overflow.',
        ),
        Op(
            OpType.BASE,
            b'\xb6',
            'conv.ovf.u2',
            [],
            'Convert to an unsigned int16 (on the stack as int32) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x87',
            'conv.ovf.u2.un',
            [],
            'Convert unsigned to an unsigned int16 (on the stack as int32) and throw an exception on '
            'overflow.',
        ),
        Op(
            OpType.BASE,
            b'\xb8',
            'conv.ovf.u4',
            [],
            'Convert to an unsigned int32 (on the stack as int32) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x88',
            'conv.ovf.u4.un',
            [],
            'Convert unsigned to an unsigned int32 (on the stack as int32) and throw an exception on '
            'overflow.',
        ),
        Op(
            OpType.BASE,
            b'\xba',
            'conv.ovf.u8',
            [],
            'Convert to an unsigned int64 (on the stack as int64) and throw an exception on overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x89',
            'conv.ovf.u8.un',
            [],
            'Convert unsigned to an unsigned int64 (on the stack as int64) and throw an exception on '
            'overflow.',
        ),
        Op(
            OpType.BASE,
            b'\x76',
            'conv.r.un',
            [],
            'Convert unsigned integer to floating-point, pushing F on stack.',
        ),
        Op(
            OpType.BASE,
            b'\x6b',
            'conv.r4',
            [],
            'Convert to float32, pushing F on stack.',
        ),
        Op(
            OpType.BASE,
            b'\x6c',
            'conv.r8',
            [],
            'Convert to float64, pushing F on stack.',
        ),
        Op(
            OpType.BASE,
            b'\xe0',
            'conv.u',
            [],
            'Convert to native unsigned int, pushing native int on stack.',
        ),
        Op(
            OpType.BASE,
            b'\xd2',
            'conv.u1',
            [],
            'Convert to unsigned int8, pushing int32 on stack.',
        ),
        Op(
            OpType.BASE,
            b'\xd1',
            'conv.u2',
            [],
            'Convert to unsigned int16, pushing int32 on stack.',
        ),
        Op(
            OpType.BASE,
            b'\x6d',
            'conv.u4',
            [],
            'Convert to unsigned int32, pushing int32 on stack.',
        ),
        Op(
            OpType.BASE,
            b'\x6e',
            'conv.u8',
            [],
            'Convert to unsigned int64, pushing int64 on stack.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x17',
            'cpblk',
            [],
            'Copy data from memory to memory.',
        ),
        Op(
            OpType.BASE,
            b'\x5b',
            'div',
            [],
            'Divide two values to return a quotient or floating-point result.',
        ),
        Op(
            OpType.BASE,
            b'\x5c',
            'div.un',
            [],
            'Divide two values, unsigned, returning a quotient.',
        ),
        Op(
            OpType.BASE,
            b'\x25',
            'dup',
            [],
            'Duplicate the value on the top of the stack.',
        ),
        Op(
            OpType.BASE,
            b'\xdc',
            'endfault',
            [],
            'End fault clause of an exception block.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x11',
            'endfilter',
            [],
            'End an exception handling filter clause.',
        ),
        Op(
            OpType.BASE,
            b'\xdc',
            'endfinally',
            [],
            'End finally clause of an exception block.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x18',
            'initblk',
            [],
            'Set all bytes in a block of memory to a given byte value.',
        ),
        Op(
            OpType.BASE,
            b'\x27',
            'jmp',
            [Method()],
            'Exit current method and jump to the specified method.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x09',
            'ldarg',
            [UInt16(is_num=True)],
            'Load argument numbered num onto the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x02',
            'ldarg.0',
            [],
            'Load argument 0 onto the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x03',
            'ldarg.1',
            [],
            'Load argument 1 onto the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x04',
            'ldarg.2',
            [],
            'Load argument 2 onto the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x05',
            'ldarg.3',
            [],
            'Load argument 3 onto the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x0e',
            'ldarg.s',
            [UInt8(is_num=True)],
            'Load argument numbered num onto the stack, short form.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x0a',
            'ldarga',
            [UInt16(is_arg_num=True)],
            'Fetch the address of argument argNum.',
        ),
        Op(
            OpType.BASE,
            b'\x0f',
            'ldarga.s',
            [UInt8(is_arg_num=True)],
            'Fetch the address of argument argNum, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x20',
            'ldc.i4',
            [Int32(is_num=True)],
            'Push num of type int32 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x16',
            'ldc.i4.0',
            [],
            'Push 0 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x17',
            'ldc.i4.1',
            [],
            'Push 1 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x18',
            'ldc.i4.2',
            [],
            'Push 2 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x19',
            'ldc.i4.3',
            [],
            'Push 3 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x1a',
            'ldc.i4.4',
            [],
            'Push 4 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x1b',
            'ldc.i4.5',
            [],
            'Push 5 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x1c',
            'ldc.i4.6',
            [],
            'Push 6 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x1d',
            'ldc.i4.7',
            [],
            'Push 7 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x1e',
            'ldc.i4.8',
            [],
            'Push 8 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x15',
            'ldc.i4.m1',
            [],
            'Push -1 onto the stack as int32.',
        ),
        Op(
            OpType.BASE,
            b'\x15',
            'ldc.i4.M1',
            [],
            'Push -1 onto the stack as int32 (alias for ldc.i4.m1).',
            is_alias=True,
        ),
        Op(
            OpType.BASE,
            b'\x1f',
            'ldc.i4.s',
            [Int8(is_num=True)],
            'Push num onto the stack as int32, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x21',
            'ldc.i8',
            [Int64(is_num=True)],
            'Push num of type int64 onto the stack as int64.',
        ),
        Op(
            OpType.BASE,
            b'\x22',
            'ldc.r4',
            [Float32(is_num=True)],
            'Push num of type float32 onto the stack as F.',
        ),
        Op(
            OpType.BASE,
            b'\x23',
            'ldc.r8',
            [Float64(is_num=True)],
            'Push num of type float64 onto the stack as F.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x06',
            'ldftn',
            [Method()],
            'Push a pointer to a method referenced by method, on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x4d',
            'ldind.i',
            [],
            'Indirect load value of type native int as native int on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x46',
            'ldind.i1',
            [],
            'Indirect load value of type int8 as int32 on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x48',
            'ldind.i2',
            [],
            'Indirect load value of type int16 as int32 on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x4a',
            'ldind.i4',
            [],
            'Indirect load value of type int32 as int32 on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x4c',
            'ldind.i8',
            [],
            'Indirect load value of type int64 as int64 on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x4e',
            'ldind.r4',
            [],
            'Indirect load value of type float32 as F on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x4f',
            'ldind.r8',
            [],
            'Indirect load value of type float64 as F on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x50',
            'ldind.ref',
            [],
            'Indirect load value of type object ref as O on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x47',
            'ldind.u1',
            [],
            'Indirect load value of type unsigned int8 as int32 on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x49',
            'ldind.u2',
            [],
            'Indirect load value of type unsigned int16 as int32 on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x4b',
            'ldind.u4',
            [],
            'Indirect load value of type unsigned int32 as int32 on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x4c',
            'ldind.u8',
            [],
            'Indirect load value of type unsigned int64 as int64 on the stack (alias for ldind.i8).',
            is_alias=True,
        ),
        Op(
            OpType.BASE,
            b'\xfe\x0c',
            'ldloc',
            [UInt16(is_idx=True)],
            'Load local variable of index indx onto stack.',
        ),
        Op(
            OpType.BASE,
            b'\x06',
            'ldloc.0',
            [],
            'Load local variable 0 onto stack.',
        ),
        Op(
            OpType.BASE,
            b'\x07',
            'ldloc.1',
            [],
            'Load local variable 1 onto stack.',
        ),
        Op(
            OpType.BASE,
            b'\x08',
            'ldloc.2',
            [],
            'Load local variable 2 onto stack.',
        ),
        Op(
            OpType.BASE,
            b'\x09',
            'ldloc.3',
            [],
            'Load local variable 3 onto stack.',
        ),
        Op(
            OpType.BASE,
            b'\x11',
            'ldloc.s',
            [UInt8(is_idx=True)],
            'Load local variable of index indx onto stack, short form.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x0d',
            'ldloca',
            [UInt16(is_idx=True)],
            'Load address of local variable with index indx.',
        ),
        Op(
            OpType.BASE,
            b'\x12',
            'ldloca.s',
            [UInt8(is_idx=True)],
            'Load address of local variable with index indx, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x14',
            'ldnull',
            [],
            'Push a null reference on the stack.',
        ),
        Op(
            OpType.BASE,
            b'\xdd',
            'leave',
            [Int32(has_target=True)],
            'Exit a protected region of code.',
        ),
        Op(
            OpType.BASE,
            b'\xde',
            'leave.s',
            [Int8(has_target=True)],
            'Exit a protected region of code, short form.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x0f',
            'localloc',
            [],
            'Allocate space from the local memory pool.',
        ),
        Op(
            OpType.BASE,
            b'\x5a',
            'mul',
            [],
            'Multiply values.',
        ),
        Op(
            OpType.BASE,
            b'\xd8',
            'mul.ovf',
            [],
            'Multiply signed integer values. Signed result shall fit in same size.',
        ),
        Op(
            OpType.BASE,
            b'\xd9',
            'mul.ovf.un',
            [],
            'Multiply unsigned integer values. Unsigned result shall fit in same size.',
        ),
        Op(
            OpType.BASE,
            b'\x65',
            'neg',
            [],
            'Negate value.',
        ),
        Op(
            OpType.BASE,
            b'\x00',
            'nop',
            [],
            'Do nothing (No operation).',
        ),
        Op(
            OpType.BASE,
            b'\x66',
            'not',
            [],
            'Bitwise complement.',
        ),
        Op(
            OpType.BASE,
            b'\x60',
            'or',
            [],
            'Bitwise OR of two integer values, returns an integer.',
        ),
        Op(
            OpType.BASE,
            b'\x26',
            'pop',
            [],
            'Pop value from the stack.',
        ),
        Op(
            OpType.BASE,
            b'\x5d',
            'rem',
            [],
            'Remainder when dividing one value by another.',
        ),
        Op(
            OpType.BASE,
            b'\x5e',
            'rem.un',
            [],
            'Remainder when dividing one unsigned value by another.',
        ),
        Op(
            OpType.BASE,
            b'\x2a',
            'ret',
            [],
            'Return from method, possibly with a value.',
        ),
        Op(
            OpType.BASE,
            b'\x62',
            'shl',
            [],
            'Shift an integer left (shifting in zeros), return an integer.',
        ),
        Op(
            OpType.BASE,
            b'\x63',
            'shr',
            [],
            'Shift an integer right (shift in sign), return an integer.',
        ),
        Op(
            OpType.BASE,
            b'\x64',
            'shr.un',
            [],
            'Shift an integer right (shift in zero), return an integer.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x0b',
            'starg',
            [UInt16(is_num=True)],
            'Store value to the argument numbered num.',
        ),
        Op(
            OpType.BASE,
            b'\x10',
            'starg.s',
            [UInt8(is_num=True)],
            'Store value to the argument numbered num, short form.',
        ),
        Op(
            OpType.BASE,
            b'\xdf',
            'stind.i',
            [],
            'Store value of type native int into memory at address.',
        ),
        Op(
            OpType.BASE,
            b'\x52',
            'stind.i1',
            [],
            'Store value of type int8 into memory at address.',
        ),
        Op(
            OpType.BASE,
            b'\x53',
            'stind.i2',
            [],
            'Store value of type int16 into memory at address.',
        ),
        Op(
            OpType.BASE,
            b'\x54',
            'stind.i4',
            [],
            'Store value of type int32 into memory at address.',
        ),
        Op(
            OpType.BASE,
            b'\x55',
            'stind.i8',
            [],
            'Store value of type int64 into memory at address.',
        ),
        Op(
            OpType.BASE,
            b'\x56',
            'stind.r4',
            [],
            'Store value of type float32 into memory at address.',
        ),
        Op(
            OpType.BASE,
            b'\x57',
            'stind.r8',
            [],
            'Store value of type float64 into memory at address.',
        ),
        Op(
            OpType.BASE,
            b'\x51',
            'stind.ref',
            [],
            'Store value of type object ref (type O) into memory at address.',
        ),
        Op(
            OpType.BASE,
            b'\xfe\x0e',
            'stloc',
            [UInt16(is_idx=True)],
            'Pop a value from stack into local variable indx.',
        ),
        Op(
            OpType.BASE,
            b'\x0a',
            'stloc.0',
            [],
            'Pop a value from stack into local variable 0.',
        ),
        Op(
            OpType.BASE,
            b'\x0b',
            'stloc.1',
            [],
            'Pop a value from stack into local variable 1.',
        ),
        Op(
            OpType.BASE,
            b'\x0c',
            'stloc.2',
            [],
            'Pop a value from stack into local variable 2.',
        ),
        Op(
            OpType.BASE,
            b'\x0d',
            'stloc.3',
            [],
            'Pop a value from stack into local variable 3.',
        ),
        Op(
            OpType.BASE,
            b'\x13',
            'stloc.s',
            [UInt8(is_idx=True)],
            'Pop a value from stack into local variable indx, short form.',
        ),
        Op(
            OpType.BASE,
            b'\x59',
            'sub',
            [],
            'Subtract value2 from value1, returning a new value.',
        ),
        Op(
            OpType.BASE,
            b'\xda',
            'sub.ovf',
            [],
            'Subtract native int from a native int. Signed result shall fit in same size.',
        ),
        Op(
            OpType.BASE,
            b'\xdb',
            'sub.ovf.un',
            [],
            'Subtract native unsigned int from a native unsigned int. Unsigned result shall fit in same '
            'size.',
        ),
        Op(
            OpType.BASE,
            b'\x45',
            'switch',
            [UInt32(is_casecnt=True), Int32(is_caseargs=True), ...],
            'Jump to one of n values.',
            is_switch=True,
        ),
        Op(
            OpType.BASE,
            b'\x61',
            'xor',
            [],
            'Bitwise XOR of integer values, returns an integer.',
        ),
        Op(
            OpType.OBJECT_MODEL,
            b'\xfe\x16',
            'constrained.',
            [ThisType()],
            'Call a virtual method on a type constrained to be type T.',
        ),
        Op(
            OpType.OBJECT_MODEL,
            b'\xfe\x1e',
            'readonly.',
            [],
            'Specify that the subsequent array address operation performs no type check at runtime, and that '
            'it returns a controlled-mutability managed pointer.',
        ),
        Op(
            OpType.OBJECT_MODEL,
            b'\xfe\x14',
            'tail.',
            [],
            'Subsequent call terminates current method.',
        ),
        Op(
            OpType.OBJECT_MODEL,
            b'\xfe\x12',
            'unaligned. (alignment)',
            [],
            'Subsequent pointer instruction might be unaligned.',
        ),
        Op(
            OpType.OBJECT_MODEL,
            b'\xfe\x13',
            'volatile.',
            [],
            'Subsequent pointer reference is volatile.',
        ),
        Op(
            OpType.OBJECT_MODEL,
            b'\xfe\x19',
            'no. {typecheck, rangecheck, nullcheck }',
            [],
            'The specified fault check(s) normally performed as part of the execution of the subsequent '
            'instruction can/shall be skipped.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x8c',
            'box',
            [TypeTok()],
            'Convert a boxable value to its boxed form.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x6f',
            'callvirt',
            [Method()],
            'Call a method associated with an object.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x74',
            'castclass',
            [Class()],
            'Cast obj to class.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x70',
            'cpobj',
            [TypeTok()],
            'Copy a value type from src to dest.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xfe\x15',
            'initobj',
            [TypeTok()],
            'Initialize the value at address dest.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x75',
            'isinst',
            [Class()],
            'Test if obj is an instance of class, returning null or an instance of that class or interface.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xa3',
            'ldelem',
            [TypeTok()],
            'Load the element at index onto the top of the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x97',
            'ldelem.i',
            [],
            'Load the element with type native int at index onto the top of the stack as a native int.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x90',
            'ldelem.i1',
            [],
            'Load the element with type int8 at index onto the top of the stack as an int32.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x92',
            'ldelem.i2',
            [],
            'Load the element with type int16 at index onto the top of the stack as an int32.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x94',
            'ldelem.i4',
            [],
            'Load the element with type int32 at index onto the top of the stack as an int32.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x96',
            'ldelem.i8',
            [],
            'Load the element with type int64 at index onto the top of the stack as an int64.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x98',
            'ldelem.r4',
            [],
            'Load the element with type float32 at index onto the top of the stack as an F.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x99',
            'ldelem.r8',
            [],
            'Load the element with type float64 at index onto the top of the stack as an F.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x9a',
            'ldelem.ref',
            [],
            'Load the element at index onto the top of the stack as an O. The type of the O is the same as '
            'the element type of the array pushed on the CIL stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x91',
            'ldelem.u1',
            [],
            'Load the element with type unsigned int8 at index onto the top of the stack as an int32.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x93',
            'ldelem.u2',
            [],
            'Load the element with type unsigned int16 at index onto the top of the stack as an int32.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x95',
            'ldelem.u4',
            [],
            'Load the element with type unsigned int32 at index onto the top of the stack as an int32.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x96',
            'ldelem.u8',
            [],
            'Load the element with type unsigned int64 at index onto the top of the stack as an int64 '
            '(alias for ldelem.i8).',
            is_alias=True,
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x8f',
            'ldelema',
            [Class()],
            'Load the address of element at index onto the top of the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x7b',
            'ldfld',
            [Field()],
            'Push the value of field of object (or value type) obj, onto the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x7c',
            'ldflda',
            [Field()],
            'Push the address of field of object obj on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x8e',
            'ldlen',
            [],
            'Push the length (of type native unsigned int) of array on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x71',
            'ldobj',
            [TypeTok()],
            'Copy the value stored at address src to the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x7e',
            'ldsfld',
            [Field()],
            'Push the value of the static field on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x7f',
            'ldsflda',
            [Field()],
            'Push the address of the static field, field, on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x72',
            'ldstr',
            [String()],
            'Push a string object for the literal string.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xd0',
            'ldtoken',
            [Token()],
            'Convert metadata token to its runtime representation.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xfe\x07',
            'ldvirtftn',
            [Method()],
            'Push address of virtual method on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xc6',
            'mkrefany',
            [Class()],
            'Push a typed reference to ptr of type class onto the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x8d',
            'newarr',
            [Etype()],
            'Create a new array with elements of type etype.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x73',
            'newobj',
            [Ctor()],
            'Allocate an uninitialized object or value type and call ctor.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xfe\x1d',
            'refanytype',
            [],
            'Push the type token stored in a typed reference.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xc2',
            'refanyval',
            [Type()],
            'Push the address stored in a typed reference.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xfe\x1a',
            'rethrow',
            [],
            'Rethrow the current exception.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xfe\x1c',
            'sizeof',
            [TypeTok()],
            'Push the size, in bytes, of a type as an unsigned int32.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xa4',
            'stelem',
            [TypeTok()],
            'Replace array element at index with the value on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x9b',
            'stelem.i',
            [],
            'Replace array element at index with the native int value on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x9c',
            'stelem.i1',
            [],
            'Replace array element at index with the int8 value on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x9d',
            'stelem.i2',
            [],
            'Replace array element at index with the int16 value on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x9e',
            'stelem.i4',
            [],
            'Replace array element at index with the int32 value on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x9f',
            'stelem.i8',
            [],
            'Replace array element at index with the int64 value on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xa0',
            'stelem.r4',
            [],
            'Replace array element at index with the float32 value on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xa1',
            'stelem.r8',
            [],
            'Replace array element at index with the float64 value on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xa2',
            'stelem.ref',
            [],
            'Replace array element at index with the ref value on the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x7d',
            'stfld',
            [Field()],
            'Replace the value of field of the object obj with value.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x81',
            'stobj',
            [TypeTok()],
            'Store a value of type typeTok at an address.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x80',
            'stsfld',
            [Field()],
            'Replace the value of the static field with val.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x7a',
            'throw',
            [],
            'Throw an exception.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\x79',
            'unbox',
            [ValueType()],
            'Extract a value-type from obj, its boxed representation, and push a controlled-mutability '
            'managed pointer to it to the top of the stack.',
        ),
        Op(
            OpType.PREFIX_TO,
            b'\xa5',
            'unbox.any',
            [TypeTok()],
            'Extract a value-type from obj, its boxed representation, and copy to the top of the stack.',
        ),
    ]

    def lookup(self, data: bytes, include_alias: bool = False) -> Op:
        for op in self.INSTRUCTIONS:
            if not include_alias and op.is_alias:
                continue
            if data[: len(op.code)] == op.code:
                return op
        raise ValueError(f'Cannot find Op for data={data.hex()}')
