# -*- coding: utf-8 -*-

'''
This module extends the default output formatting to include HTML.
'''

from __future__ import print_function
import sys
import datetime


def html_output(result, options, *_):
    try:
        from jinja2 import Template
    except ImportError:
        sys.stderr.write(
                "HTML Output depends on jinja2. `pip install jinja2` first")
        sys.exit(2)

    file_list = []
    for source_file in result:
        if source_file:
            source_file_dict = {"filename": source_file.filename}
            func_list = []
            for source_function in source_file.function_list:
                if source_function:
                    source_function_dict = _create_dict(source_function)
                    func_list.append(source_function_dict)
                    source_file_dict["functions"] = func_list
        file_list.append(source_file_dict)
    output = Template(TEMPLATE).render(
            title='Lizard code complexity report',
            date=datetime.datetime.now().strftime('%Y-%m-%d %H:%M'),
            thresholds=options.thresholds, files=file_list)
    print(output)
    return 0


def _create_dict(obj):

    return obj.__dict__


TEMPLATE = '''<!DOCTYPE HTML PUBLIC
"-//W3C//DTD HTML 4.01 Transitional//EN"
"http://www.w3.org/TR/html4/loose.dtd">
<html>
 <head>
  <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
  <title>Code complexity report</title>
  <style>
    h2
    {
          text-align: center;
    }
        td.footer
        {
          text-align: center;
          padding-top: 3px;
          font-family: sans-serif;
        }
    td.greater-value
        {
          padding-top: 2px;
          text-align: center;
          padding-left: 10px;
          padding-right: 10px;
          font-family: sans-serif;
          white-space: nowrap;
      background-color: LightPink;
        }
    td.lesser-value
        {
          padding-top: 2px;
          text-align: center;
          padding-left: 10px;
          padding-right: 10px;
          font-family: sans-serif;
          white-space: nowrap;
      background-color: LightGreen;
        }
    td.value
        {
          padding-top: 2px;
          text-align: center;
          padding-left: 10px;
          padding-right: 10px;
          font-family: sans-serif;
          white-space: nowrap;
        }
  </style>
 </head>
 <body>
<h2>Code Complexity Report</h2>

<center>

<table>
{% for file in files %}
  <tr><td colspan="7" style="background-color:LightBlue;">
  Source file: <b>{{ file.filename }}</b></td></tr>
{% if file.functions|length > 0 %}
<tr><th>Function name</th><th></th><th>
Cyclomatic complexity
({{ thresholds["cyclomatic_complexity"] }})
</th><th>LOC ({{ thresholds["nloc"] }})</th><th>
    {% if thresholds["token_count"] %}
       Token count ({{ thresholds["token_count"] }})
    {% else %}
       Token count
    {% endif %}
</th><th>Parameter count ({{ thresholds["parameter_count"] }})</th></tr>
{% endif %}
  {% for func in file.functions %}
  <tr>
    <td style="background-color:LightSteelBlue">{{ func.name }}</td>
    <td></td>
    {% if func.cyclomatic_complexity > thresholds["cyclomatic_complexity"] %}
       <td class="greater-value">{{ func.cyclomatic_complexity }}</td>
    {% else %}
       <td class="lesser-value">{{ func.cyclomatic_complexity }}</td>
    {% endif %}

    {% if func.nloc > thresholds["nloc"] %}
       <td class="greater-value">{{ func.nloc }}</td>
    {% else %}
       <td class="lesser-value">{{ func.nloc}}</td>
    {% endif %}

    {% if thresholds["token_count"] %}
       {% if func.token_count > thresholds["token_count"] %}
          <td class="greater-value">{{ func.token_count }}</td>
       {% else %}
          <td class="lesser-value">{{ func.token_count }}</td>
       {% endif %}
    {% else %}
       <td class="value">{{ func.token_count }}</td>
    {% endif %}

    {% if func.parameters|length > thresholds["parameter_count"] %}
       <td class="greater-value">{{ func.parameters|length }}</td>
    {% else %}
       <td class="lesser-value">{{ func.parameters|length }}</td>
    {% endif %}
  </tr>
  {% endfor %}
{% endfor %}
</table>
<center>

<br>
<table width="100%" border=0 cellspacing=0 cellpadding=0>
    <tr><td class="footer">Generated by
    <a href="http://www.lizard.ws/">Lizard</a> on {{ date }}
    </td></tr>
</table>
 </body>
</html>

'''
