use gap_statistic;
use kmeans;
use ndarray::{arr2, Array1, Array2};
use ndarray_rand::RandomExt;
use rand::distributions::ChiSquared;
use test::Bencher;

#[test]
fn test_ref_dataset() {
    let data = arr2(&[[10.0, -10.0], [20.0, -20.0], [30.0, -30.0]]);

    let ref_data = gap_statistic::ref_dataset(&data.view());

    // First column all vals between 10.0 and 30.0
    assert!(ref_data
        .slice(s![.., 0])
        .iter()
        .all(|v| *v >= 10.0 && *v <= 30.0));

    // Second column all vals between -10.0 and -30.0
    assert!(ref_data
        .slice(s![.., 1])
        .iter()
        .all(|v| *v <= -10.0 && *v >= -30.0));
}

#[bench]
fn bench_kmeans(b: &mut Bencher) {
    let x = Array2::random((1000, 4), ChiSquared::new(0.1));
    let mut clf = kmeans::KMeans::new(3, 0.001, 100, 32);

    b.iter(|| {
        clf.fit(&x.view());
    });
}

#[bench]
fn bench_optimalk(b: &mut Bencher) {
    // Data with 3 clusters
    let data = [
        [-3.48416673, 2.21690696, -3.78160285],
        [-2.83352934, 1.50781001, -6.82835687],
        [-2.22449957, 0.34561257, 7.44993481],
        [-2.04892321, 2.64002175, -5.35244919],
        [4.11177911, -4.61402543, 4.17635304],
        [-3.07992021, 1.05891225, -4.97868865],
        [2.21709291, -3.98294669, 4.61313336],
        [2.56762214, -1.97643083, 3.84779264],
        [4.04101035, -4.36593097, 4.61291282],
        [-3.67133312, 0.80469347, -3.7794995],
        [3.77890239, -5.68432387, 5.2920889],
        [-4.83981245, -0.96379555, 9.6228935],
        [-6.5893229, -0.37604103, 8.27621906],
        [4.31980713, -3.87033642, 4.276133],
        [3.68529082, -6.07091476, 3.17753506],
        [-4.54024048, 2.53162379, -5.00838413],
        [-5.21432436, -1.52978635, 8.99165553],
        [-4.01864279, 3.67221166, -6.66421425],
        [-3.89002123, -2.01032378, 8.6846035],
        [1.85841071, -5.20801588, 4.98592912],
        [3.6018484, -2.61579024, 4.15515938],
        [-3.74165366, -2.75083534, 7.37785508],
        [-4.76825175, -0.64980349, 7.92086471],
        [-1.46962343, -0.85216119, 8.62895149],
        [4.91082622, -6.14576776, 5.97590565],
        [-1.76218703, 4.68553731, -4.46515943],
        [3.21106323, -4.88987441, 4.03593456],
        [-4.52169357, 0.18010008, 8.20794263],
        [2.47509328, -4.25350973, 5.74560096],
        [-2.82032967, 2.60518543, -6.11594444],
        [-3.63788606, 0.66960709, -6.08787357],
        [-3.65687425, 1.79450933, -4.41384015],
        [-2.70119706, 2.64227073, -4.81772391],
        [-4.17923336, 0.94405477, -5.65460075],
        [3.10150784, -8.08733012, 6.27913381],
        [-2.4149846, 1.04203391, -7.35178234],
        [-3.59651777, -0.65732748, 8.38521341],
        [-3.30741855, -0.05189384, 9.05717065],
        [-2.80180911, -1.78118279, 10.62745558],
        [-5.10236292, 0.4020601, 8.05814104],
        [-2.93642214, 0.0939677, -3.98925247],
        [-5.74964083, -0.73548004, 10.398722],
        [2.24462146, -6.32365701, 4.01348798],
        [-2.97285699, 2.15987212, -6.43230844],
        [2.66838814, -3.50122198, 4.17690656],
        [-3.68269469, -0.38556751, 8.48469807],
        [-2.43592565, 3.25715637, -5.32284976],
        [1.1630839, -3.18341328, 3.91517739],
        [-3.93168076, 0.37462374, 6.76977806],
        [-4.69817173, -0.18782651, 7.71200999],
    ];

    let x = Array1::from_iter(data.to_vec().iter().flat_map(|v| v.iter().map(|i| *i)))
        .into_shape((50, 3))
        .unwrap();
    let cluster_range = Array1::from_iter(1..6);

    b.iter(|| {
        let _ = gap_statistic::optimal_k(&x.view(), cluster_range.view(), 10, 10);
    });
}

#[test]
fn test_optimalk() {
    // Data with 3 clusters
    let data = [
        [-3.48416673, 2.21690696, -3.78160285],
        [-2.83352934, 1.50781001, -6.82835687],
        [-2.22449957, 0.34561257, 7.44993481],
        [-2.04892321, 2.64002175, -5.35244919],
        [4.11177911, -4.61402543, 4.17635304],
        [-3.07992021, 1.05891225, -4.97868865],
        [2.21709291, -3.98294669, 4.61313336],
        [2.56762214, -1.97643083, 3.84779264],
        [4.04101035, -4.36593097, 4.61291282],
        [-3.67133312, 0.80469347, -3.7794995],
        [3.77890239, -5.68432387, 5.2920889],
        [-4.83981245, -0.96379555, 9.6228935],
        [-6.5893229, -0.37604103, 8.27621906],
        [4.31980713, -3.87033642, 4.276133],
        [3.68529082, -6.07091476, 3.17753506],
        [-4.54024048, 2.53162379, -5.00838413],
        [-5.21432436, -1.52978635, 8.99165553],
        [-4.01864279, 3.67221166, -6.66421425],
        [-3.89002123, -2.01032378, 8.6846035],
        [1.85841071, -5.20801588, 4.98592912],
        [3.6018484, -2.61579024, 4.15515938],
        [-3.74165366, -2.75083534, 7.37785508],
        [-4.76825175, -0.64980349, 7.92086471],
        [-1.46962343, -0.85216119, 8.62895149],
        [4.91082622, -6.14576776, 5.97590565],
        [-1.76218703, 4.68553731, -4.46515943],
        [3.21106323, -4.88987441, 4.03593456],
        [-4.52169357, 0.18010008, 8.20794263],
        [2.47509328, -4.25350973, 5.74560096],
        [-2.82032967, 2.60518543, -6.11594444],
        [-3.63788606, 0.66960709, -6.08787357],
        [-3.65687425, 1.79450933, -4.41384015],
        [-2.70119706, 2.64227073, -4.81772391],
        [-4.17923336, 0.94405477, -5.65460075],
        [3.10150784, -8.08733012, 6.27913381],
        [-2.4149846, 1.04203391, -7.35178234],
        [-3.59651777, -0.65732748, 8.38521341],
        [-3.30741855, -0.05189384, 9.05717065],
        [-2.80180911, -1.78118279, 10.62745558],
        [-5.10236292, 0.4020601, 8.05814104],
        [-2.93642214, 0.0939677, -3.98925247],
        [-5.74964083, -0.73548004, 10.398722],
        [2.24462146, -6.32365701, 4.01348798],
        [-2.97285699, 2.15987212, -6.43230844],
        [2.66838814, -3.50122198, 4.17690656],
        [-3.68269469, -0.38556751, 8.48469807],
        [-2.43592565, 3.25715637, -5.32284976],
        [1.1630839, -3.18341328, 3.91517739],
        [-3.93168076, 0.37462374, 6.76977806],
        [-4.69817173, -0.18782651, 7.71200999],
    ];

    let x = Array1::from_iter(data.to_vec().iter().flat_map(|v| v.iter().map(|i| *i)))
        .into_shape((50, 3))
        .unwrap();
    let cluster_range = Array1::from_iter(1..6);
    let _result = gap_statistic::optimal_k(&x.view(), cluster_range.view(), 10, 10);
    // println!("Got {:?}", result)
}
