import logging
import os
import subprocess
from pathlib import Path

import yaml

from pyopenapi_gen.context.render_context import RenderContext
from pyopenapi_gen.core.loader.loader import load_ir_from_spec
from pyopenapi_gen.emitters.client_emitter import ClientEmitter
from pyopenapi_gen.emitters.core_emitter import CoreEmitter
from pyopenapi_gen.emitters.endpoints_emitter import EndpointsEmitter
from pyopenapi_gen.emitters.exceptions_emitter import ExceptionsEmitter
from pyopenapi_gen.emitters.models_emitter import ModelsEmitter


def test_business_swagger_generation(tmp_path: Path) -> None:
    """
    Scenario:
        - Generate the client for the business_swagger.json spec using direct emitter calls.
        - Verify all expected files are generated.
    Expected Outcome:
        - config.py, client.py, and endpoint modules are present and correct.
    """
    # Configure logging to capture DEBUG from specific modules
    logging.basicConfig(level=logging.WARNING)  # Default for others
    logging.getLogger("pyopenapi_gen").setLevel(logging.WARNING)  # Changed from DEBUG to WARNING

    # Arrange
    # Correctly locate spec_source relative to the project root
    project_root_dir = Path(
        __file__
    ).parent.parent.parent  # Go up three levels from tests/integrations/test.py to project root
    spec_source = project_root_dir / "input" / "business_swagger.json"

    spec_file = tmp_path / "spec.json"
    spec_file.write_text(spec_source.read_text())
    out_dir = tmp_path / "out"
    out_dir.mkdir()

    # Define the output package name corresponding to out_dir
    output_package = "out"

    # Act
    spec_dict = yaml.safe_load(spec_file.read_text())
    ir = load_ir_from_spec(spec_dict)

    # Define core and output paths for this test's direct emitter setup
    project_root = tmp_path
    output_package = "out"  # Treat 'out' as the top-level package dir
    core_package_name = f"{output_package}.core"  # e.g., out.core
    core_dir = project_root / Path(*core_package_name.split("."))

    # Create directories if they don't exist
    out_dir = project_root / Path(*output_package.split("."))
    out_dir.mkdir(exist_ok=True)
    core_dir.mkdir(exist_ok=True)

    # Create RenderContext for ModelsEmitter
    render_context = RenderContext(
        core_package_name=core_package_name,
        package_root_for_generated_code=str(out_dir),
        overall_project_root=str(project_root),
        parsed_schemas=ir.schemas,
    )

    # Run ExceptionsEmitter first to get alias names
    exceptions_emitter = ExceptionsEmitter(core_package_name=core_package_name, overall_project_root=str(project_root))
    exception_files_list, exception_alias_names = exceptions_emitter.emit(ir, str(core_dir))  # Emit to core_dir

    core_emitter = CoreEmitter(
        core_package=core_package_name, exception_alias_names=exception_alias_names
    )  # Pass names
    models_emitter = ModelsEmitter(context=render_context, parsed_schemas=ir.schemas)
    endpoints_emitter = EndpointsEmitter(context=render_context)
    client_emitter = ClientEmitter(context=render_context)

    # Run emitters
    # Exceptions were already emitted to get names
    core_emitter.emit(str(out_dir))  # Generate core files first (takes output dir)
    models_emitter.emit(ir, str(out_dir))
    endpoints_emitter.emit(ir.operations, str(out_dir))
    client_emitter.emit(ir, str(out_dir))

    # Assert
    # Core files should now be generated by CoreEmitter
    assert (core_dir / "__init__.py").exists(), "core/__init__.py not generated by CoreEmitter"
    assert (core_dir / "py.typed").exists(), "core/py.typed not generated by CoreEmitter"
    assert (core_dir / "config.py").exists(), "core/config.py not generated by CoreEmitter"
    assert (core_dir / "http_transport.py").exists(), "core/http_transport.py not generated by CoreEmitter"
    assert (core_dir / "exceptions.py").exists(), "core/exceptions.py not generated by CoreEmitter"
    assert (core_dir / "auth" / "base.py").exists(), "core/auth/base.py not generated by CoreEmitter"

    assert (out_dir / "client.py").exists(), "client.py not generated"
    endpoints_dir = out_dir / "endpoints"
    assert endpoints_dir.exists(), "endpoints directory not generated"
    assert (endpoints_dir / "__init__.py").exists(), "__init__.py missing in endpoints"
    py_files = [p for p in endpoints_dir.glob("*.py") if p.name != "__init__.py"]
    assert py_files, "no endpoint modules generated"

    # Run mypy on the generated code to ensure type correctness
    env = os.environ.copy()
    src_dir = os.path.abspath(os.path.join(os.path.dirname(__file__), "..", "..", "src"))
    env["PYTHONPATH"] = os.pathsep.join([str(out_dir.parent.resolve()), src_dir, env.get("PYTHONPATH", "")])

    # Check the defined top-level package name
    packages_to_check = [output_package.split(".")[0]]

    # Define a fixed path for the mypy output log in the project root
    project_root_dir = Path(__file__).parent.parent.parent  # This should be the project root
    test_outputs_dir = project_root_dir / "test_outputs"
    test_outputs_dir.mkdir(parents=True, exist_ok=True)
    mypy_output_filename = test_outputs_dir / "mypy_business_swagger_errors.txt"

    mypy_command = ["mypy", "--strict"] + packages_to_check

    # Run mypy capturing output to stdout/stderr attributes of the result object
    mypy_result = subprocess.run(
        mypy_command,
        capture_output=True,
        text=False,
        env=env,
        cwd=out_dir.parent,  # text=False to handle bytes manually
    )

    # Initialize with placeholder for mypy_output_content
    mypy_output_content = "Mypy did not run or produce output."

    if mypy_result.returncode != 0:
        # print(f"DEBUG: mypy_result object: {mypy_result}")

        # Decode stdout and stderr, then combine for full output
        mypy_stdout_decoded = (
            mypy_result.stdout.decode("utf-8", errors="replace") if mypy_result.stdout else "(empty stdout)"
        )
        mypy_stderr_decoded = (
            mypy_result.stderr.decode("utf-8", errors="replace") if mypy_result.stderr else "(empty stderr)"
        )

        combined_output = f"Mypy STDOUT:\n{mypy_stdout_decoded}\n\nMypy STDERR:\n{mypy_stderr_decoded}"
        # print("--- Mypy Full Captured Output ---")
        # print(combined_output)
        # print("--- End Mypy Full Captured Output ---")

        # Write to the debug file as well
        Path(mypy_output_filename).write_text(combined_output, encoding="utf-8")
        mypy_output_content = combined_output  # For the assertion message
    else:
        # If successful, still good to log that it passed and where, but less verbose
        mypy_output_content = "Mypy checks passed."
        Path(mypy_output_filename).write_text(
            mypy_output_content
            + f"\nSTDOUT:\n{mypy_result.stdout.decode('utf-8', errors='replace') if mypy_result.stdout else ''}\nSTDERR:\n{mypy_result.stderr.decode('utf-8', errors='replace') if mypy_result.stderr else ''}",
            encoding="utf-8",
        )

    assert (
        mypy_result.returncode == 0
    ), f"mypy errors (see full output in logs or {mypy_output_filename.relative_to(project_root_dir)}):\n{mypy_output_content}"


def test_generated_agent_datasources_imports_are_valid(tmp_path: Path) -> None:
    """
    Scenario:
        - Generate the business_swagger client as in the main test.
        - Read the generated agent_datasources.py file.
    Expected Outcome:
        - The first import line is a valid Python import (no slashes, starts with 'from ..models.' or 'from .').
    """
    # Copy the provided business_swagger.json into a temporary spec file
    project_root_dir = Path(__file__).parent.parent.parent  # Go up three levels
    spec_source = project_root_dir / "input" / "business_swagger.json"
    spec_file = tmp_path / "spec.json"
    spec_file.write_text(spec_source.read_text())

    out_dir = tmp_path / "out"
    out_dir.mkdir()

    spec_dict = yaml.safe_load(spec_file.read_text())
    ir = load_ir_from_spec(spec_dict)

    # Define core and output paths for this test's direct emitter setup
    project_root = tmp_path
    output_package = "out"  # Treat 'out' as the top-level package dir
    core_package_name = f"{output_package}.core"  # e.g., out.core
    core_dir = project_root / Path(*core_package_name.split("."))

    # Create directories if they don't exist
    out_dir = project_root / Path(*output_package.split("."))
    out_dir.mkdir(exist_ok=True)
    core_dir.mkdir(exist_ok=True)

    # Create RenderContext for ModelsEmitter
    render_context = RenderContext(
        core_package_name=core_package_name,
        package_root_for_generated_code=str(out_dir),
        overall_project_root=str(project_root),
        parsed_schemas=ir.schemas,
    )

    # Instantiate emitters with correct core package name where needed
    core_emitter = CoreEmitter(core_package=core_package_name)
    models_emitter = ModelsEmitter(context=render_context, parsed_schemas=ir.schemas)
    endpoints_emitter = EndpointsEmitter(context=render_context)
    client_emitter = ClientEmitter(context=render_context)

    # Run emitters
    core_emitter.emit(str(out_dir))
    models_emitter.emit(ir, str(out_dir))
    endpoints_emitter.emit(ir.operations, str(out_dir))
    client_emitter.emit(ir, str(out_dir))

    assert (out_dir / "endpoints" / "agent_datasources.py").exists(), "agent_datasources.py not generated"
    content = (out_dir / "endpoints" / "agent_datasources.py").read_text().splitlines()
    # Find the first non-empty, non-comment line that is a relative import
    first_relative_import = next(
        (line for line in content if line.strip().startswith("from ..models.") or line.strip().startswith("from .")),
        "",
    )
    assert first_relative_import, "No relative import found in agent_datasources.py"
    assert "/" not in first_relative_import, f"Relative import line contains a slash: {first_relative_import}"
