"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiExecuteAutomationHook = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const __1 = require("../..");
/**
 * ExecuteAutomation implementation using AWS API
 */
class ApiExecuteAutomationHook {
    constructor(awsInvoker, sleepHook) {
        this.props = { awsInvoker, sleepHook };
    }
    execute(props) {
        console.log(`ExecuteAutomation: Checking execution inputs for ${props.documentName}`);
        this.parameterValidation(props);
        console.log(`ExecuteAutomation: Starting execution of ${props.documentName}`);
        const executionId = this.startExecution(props);
        console.log(`ExecuteAutomation: Waiting for ${executionId} to complete`);
        const result = this.waitForExecution(executionId);
        console.log(`ExecuteAutomation: ${executionId} finished`);
        return result;
    }
    parameterValidation(props) {
        if (props.targetLocations && (props.targetLocations.length < 1 || props.targetLocations.length > 100)) {
            throw new Error('TargetLocations must contain atleast 1 and no more than 100 items.');
        }
        if (props.targetMaps && props.targets) {
            throw new Error('TargetMaps and Targets cannot both be specified.');
        }
        if ((props.targets && !props.targetParameterName) || (!props.targets && props.targetParameterName)) {
            throw new Error('TargetParameterName and Targets must both be specified together.');
        }
    }
    startExecution(props) {
        const result = new __1.AutomationStepSimulation(new __1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'execute', {
            service: 'SSM',
            pascalCaseApi: 'StartAutomationExecution',
            apiParams: this.getExecuteProps(props),
            outputs: [{
                    outputType: __1.DataTypeEnum.STRING,
                    name: 'executionId',
                    selector: '$.AutomationExecutionId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== __1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to start execution: ${result.stackTrace}`);
        }
        return result.outputs?.['execute.executionId'];
    }
    getExecuteProps(props) {
        const apiParamMap = {
            DocumentName: props.documentName,
            DocumentVersion: props.documentVersion,
            TargetParameterName: props.targetParameterName,
            MaxConcurrency: props.maxConcurrency,
            MaxErrors: props.maxErrors,
            Targets: props.targets,
            Tags: props.tags,
            TargetLocations: props.targetLocations,
        };
        const apiParams = {};
        for (const key of Object.keys(apiParamMap)) {
            const value = apiParamMap[key];
            if (value === undefined) {
                continue;
            }
            apiParams[key] = value;
        }
        if (props.runtimeParameters) {
            apiParams.Parameters = this.adaptExecuteParameters(props.runtimeParameters);
        }
        if (props.targetMaps) {
            apiParams.TargetMaps = this.adaptTargetMaps(props.targetMaps);
        }
        return apiParams;
    }
    adaptExecuteParameters(parameters) {
        const result = {};
        for (const key of Object.keys(parameters)) {
            const value = parameters[key];
            if (__1.isString(value)) {
                result[key] = [value];
                continue;
            }
            if (__1.isBoolean(value) || __1.isNumber(value)) {
                result[key] = [`${value}`];
                continue;
            }
            if (__1.isStringList(value)) {
                result[key] = value;
                continue;
            }
            if (__1.isMapList(value)) {
                result[key] = value.map(x => JSON.stringify(x));
                continue;
            }
            if (__1.isStringMap(value)) {
                result[key] = [JSON.stringify(value)];
                continue;
            }
            throw new Error(`Parameter ${key} must be a Boolean, Number, String, StringList, StringMap, or MapList.`);
        }
        return result;
    }
    adaptTargetMaps(targetMaps) {
        return targetMaps.map(this.adaptExecuteParameters);
    }
    waitForExecution(executionId) {
        let execution = this.getExecution(executionId);
        while (ApiExecuteAutomationHook.RunningStatuses.includes(execution.AutomationExecutionStatus)) {
            console.log(`Automation ${executionId} is not complete. Status: ${execution.AutomationExecutionStatus}.`);
            new __1.AutomationStepSimulation(new __1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
            execution = this.getExecution(executionId);
        }
        if (execution.AutomationExecutionStatus !== 'Success') {
            throw new Error(`${executionId} ended in status ${execution.AutomationExecutionStatus}`);
        }
        return {
            executionId: executionId,
            status: execution.AutomationExecutionStatus,
            output: Object.values(execution.Outputs).flat(),
        };
    }
    getExecution(executionId) {
        const result = new __1.AutomationStepSimulation(new __1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'getExecution', {
            service: 'SSM',
            pascalCaseApi: 'GetAutomationExecution',
            apiParams: {
                AutomationExecutionId: executionId,
            },
            outputs: [{
                    outputType: __1.DataTypeEnum.STRING_MAP,
                    name: 'execution',
                    selector: '$.AutomationExecution',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== __1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to get execution ${executionId} status: ${result.stackTrace}`);
        }
        return result.outputs?.['getExecution.execution'];
    }
}
exports.ApiExecuteAutomationHook = ApiExecuteAutomationHook;
_a = JSII_RTTI_SYMBOL_1;
ApiExecuteAutomationHook[_a] = { fqn: "@cdklabs/cdk-ssm-documents.ApiExecuteAutomationHook", version: "0.0.23" };
ApiExecuteAutomationHook.RunningStatuses = ['Pending', 'InProgress', 'Cancelling', 'Waiting'];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpLWV4ZWN1dGUtYXV0b21hdGlvbi1ob29rLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2ludGVyZmFjZS9leGVjdXRlLWF1dG9tYXRpb24taG9vay9hcGktZXhlY3V0ZS1hdXRvbWF0aW9uLWhvb2sudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2Q0FBb0M7QUFDcEMsNkJBQTBOO0FBaUMxTjs7R0FFRztBQUNILE1BQWEsd0JBQXdCO0lBS25DLFlBQVksVUFBdUIsRUFBRSxTQUFxQjtRQUN4RCxJQUFJLENBQUMsS0FBSyxHQUFHLEVBQUUsVUFBVSxFQUFFLFNBQVMsRUFBRSxDQUFDO0lBQ3pDLENBQUM7SUFFTSxPQUFPLENBQUMsS0FBNkI7UUFDMUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxvREFBb0QsS0FBSyxDQUFDLFlBQVksRUFBRSxDQUFDLENBQUM7UUFDdEYsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sQ0FBQyxHQUFHLENBQUMsNENBQTRDLEtBQUssQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDO1FBQzlFLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDL0MsT0FBTyxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsV0FBVyxjQUFjLENBQUMsQ0FBQztRQUN6RSxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDbEQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsV0FBVyxXQUFXLENBQUMsQ0FBQztRQUUxRCxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDO0lBRU8sbUJBQW1CLENBQUMsS0FBNkI7UUFDdkQsSUFBSSxLQUFLLENBQUMsZUFBZSxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxJQUFJLEtBQUssQ0FBQyxlQUFlLENBQUMsTUFBTSxHQUFHLEdBQUcsQ0FBQyxFQUFFO1lBQ3JHLE1BQU0sSUFBSSxLQUFLLENBQUMsb0VBQW9FLENBQUMsQ0FBQztTQUN2RjtRQUNELElBQUksS0FBSyxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFO1lBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELENBQUMsQ0FBQztTQUNyRTtRQUNELElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLG1CQUFtQixDQUFDLEVBQUU7WUFDbEcsTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO1NBQ3JGO0lBQ0gsQ0FBQztJQUVPLGNBQWMsQ0FBQyxLQUE2QjtRQUNsRCxNQUFNLE1BQU0sR0FBRyxJQUFJLDRCQUF3QixDQUFDLElBQUksY0FBVSxDQUFDLElBQUksbUJBQUssRUFBRSxFQUFFLFNBQVMsRUFBRTtZQUNqRixPQUFPLEVBQUUsS0FBSztZQUNkLGFBQWEsRUFBRSwwQkFBMEI7WUFDekMsU0FBUyxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDO1lBQ3RDLE9BQU8sRUFBRSxDQUFDO29CQUNSLFVBQVUsRUFBRSxnQkFBWSxDQUFDLE1BQU07b0JBQy9CLElBQUksRUFBRSxhQUFhO29CQUNuQixRQUFRLEVBQUUseUJBQXlCO2lCQUNwQyxDQUFDO1NBQ0gsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDM0IsSUFBSSxNQUFNLENBQUMsWUFBWSxLQUFLLGdCQUFZLENBQUMsT0FBTyxFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsOEJBQThCLE1BQU0sQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1NBQ3BFO1FBQ0QsT0FBTyxNQUFNLENBQUMsT0FBTyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRU8sZUFBZSxDQUFDLEtBQTZCO1FBQ25ELE1BQU0sV0FBVyxHQUF3QjtZQUN2QyxZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7WUFDaEMsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO1lBQ3RDLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxtQkFBbUI7WUFDOUMsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1lBQ3BDLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztZQUMxQixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZTtTQUN2QyxDQUFDO1FBQ0YsTUFBTSxTQUFTLEdBQXdCLEVBQUUsQ0FBQztRQUMxQyxLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUU7WUFDMUMsTUFBTSxLQUFLLEdBQUcsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQy9CLElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtnQkFBRSxTQUFTO2FBQUU7WUFDdEMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztTQUN4QjtRQUVELElBQUksS0FBSyxDQUFDLGlCQUFpQixFQUFFO1lBQzNCLFNBQVMsQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLHNCQUFzQixDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1NBQzdFO1FBQ0QsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ3BCLFNBQVMsQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDL0Q7UUFFRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRU8sc0JBQXNCLENBQUMsVUFBK0I7UUFDNUQsTUFBTSxNQUFNLEdBQTZCLEVBQUUsQ0FBQztRQUM1QyxLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDekMsTUFBTSxLQUFLLEdBQUcsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQzlCLElBQUksWUFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFO2dCQUNuQixNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDdEIsU0FBUzthQUNWO1lBQ0QsSUFBSSxhQUFTLENBQUMsS0FBSyxDQUFDLElBQUksWUFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFO2dCQUN2QyxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUM7Z0JBQzNCLFNBQVM7YUFDVjtZQUNELElBQUksZ0JBQVksQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDdkIsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztnQkFDcEIsU0FBUzthQUNWO1lBQ0QsSUFBSSxhQUFTLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQ3BCLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNoRCxTQUFTO2FBQ1Y7WUFDRCxJQUFJLGVBQVcsQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDdEIsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO2dCQUN0QyxTQUFTO2FBQ1Y7WUFDRCxNQUFNLElBQUksS0FBSyxDQUFDLGFBQWEsR0FBRyx3RUFBd0UsQ0FBQyxDQUFDO1NBQzNHO1FBQ0QsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQztJQUVPLGVBQWUsQ0FBQyxVQUFpQztRQUN2RCxPQUFPLFVBQVUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDckQsQ0FBQztJQUVPLGdCQUFnQixDQUFDLFdBQW1CO1FBQzFDLElBQUksU0FBUyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDL0MsT0FBTyx3QkFBd0IsQ0FBQyxlQUFlLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyx5QkFBeUIsQ0FBQyxFQUFFO1lBQzdGLE9BQU8sQ0FBQyxHQUFHLENBQUMsY0FBYyxXQUFXLDZCQUE2QixTQUFTLENBQUMseUJBQXlCLEdBQUcsQ0FBQyxDQUFDO1lBQzFHLElBQUksNEJBQXdCLENBQUMsSUFBSSxhQUFTLENBQUMsSUFBSSxtQkFBSyxFQUFFLEVBQUUsT0FBTyxFQUFFO2dCQUMvRCxZQUFZLEVBQUUsQ0FBQzthQUNoQixDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUMzQixTQUFTLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUM1QztRQUNELElBQUksU0FBUyxDQUFDLHlCQUF5QixLQUFLLFNBQVMsRUFBRTtZQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLEdBQUcsV0FBVyxvQkFBb0IsU0FBUyxDQUFDLHlCQUF5QixFQUFFLENBQUMsQ0FBQztTQUMxRjtRQUVELE9BQU87WUFDTCxXQUFXLEVBQUUsV0FBVztZQUN4QixNQUFNLEVBQUUsU0FBUyxDQUFDLHlCQUF5QjtZQUMzQyxNQUFNLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsSUFBSSxFQUFFO1NBQ2hELENBQUM7SUFDSixDQUFDO0lBRU8sWUFBWSxDQUFDLFdBQW1CO1FBQ3RDLE1BQU0sTUFBTSxHQUFHLElBQUksNEJBQXdCLENBQUMsSUFBSSxjQUFVLENBQUMsSUFBSSxtQkFBSyxFQUFFLEVBQUUsY0FBYyxFQUFFO1lBQ3RGLE9BQU8sRUFBRSxLQUFLO1lBQ2QsYUFBYSxFQUFFLHdCQUF3QjtZQUN2QyxTQUFTLEVBQUU7Z0JBQ1QscUJBQXFCLEVBQUUsV0FBVzthQUNuQztZQUNELE9BQU8sRUFBRSxDQUFDO29CQUNSLFVBQVUsRUFBRSxnQkFBWSxDQUFDLFVBQVU7b0JBQ25DLElBQUksRUFBRSxXQUFXO29CQUNqQixRQUFRLEVBQUUsdUJBQXVCO2lCQUNsQyxDQUFDO1NBQ0gsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDM0IsSUFBSSxNQUFNLENBQUMsWUFBWSxLQUFLLGdCQUFZLENBQUMsT0FBTyxFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLFdBQVcsWUFBWSxNQUFNLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUN4RjtRQUNELE9BQU8sTUFBTSxDQUFDLE9BQU8sRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDcEQsQ0FBQzs7QUFwSkgsNERBcUpDOzs7QUFwSnlCLHdDQUFlLEdBQXNCLENBQUMsU0FBUyxFQUFFLFlBQVksRUFBRSxZQUFZLEVBQUUsU0FBUyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBTdGFjayB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbiwgQXdzQXBpU3RlcCwgRGF0YVR5cGVFbnVtLCBJQXdzSW52b2tlciwgSUV4ZWN1dGVBdXRvbWF0aW9uSG9vaywgaXNCb29sZWFuLCBJU2xlZXBIb29rLCBpc01hcExpc3QsIGlzTnVtYmVyLCBpc1N0cmluZywgaXNTdHJpbmdMaXN0LCBpc1N0cmluZ01hcCwgUmVzcG9uc2VDb2RlLCBTbGVlcFN0ZXAgfSBmcm9tICcuLi8uLic7XG5pbXBvcnQgeyBFeGVjdXRlQXV0b21hdGlvblByb3BzLCBFeGVjdXRlQXV0b21hdGlvbk91dHB1dHMgfSBmcm9tICcuLi9leGVjdXRlLWF1dG9tYXRpb24taG9vayc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQXBpRXhlY3V0ZUF1dG9tYXRpb25Qcm9wcyB7XG4gIHJlYWRvbmx5IGF3c0ludm9rZXI6IElBd3NJbnZva2VyO1xuICByZWFkb25seSBzbGVlcEhvb2s6IElTbGVlcEhvb2s7XG59XG5cbmludGVyZmFjZSBBdXRvbWF0aW9uRXhlY3V0aW9uIHtcbiAgQXV0b21hdGlvbkV4ZWN1dGlvblN0YXR1czogRXhlY3V0aW9uU3RhdHVzO1xuICBPdXRwdXRzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmdbXT47XG59XG5cbnR5cGUgRXhlY3V0aW9uU3RhdHVzID1cbidQZW5kaW5nJ3xcbidJblByb2dyZXNzJ3xcbidXYWl0aW5nJ3xcbidTdWNjZXNzJ3xcbidUaW1lZE91dCd8XG4nQ2FuY2VsbGluZyd8XG4nQ2FuY2VsbGVkJ3xcbidGYWlsZWQnfFxuJ1BlbmRpbmdBcHByb3ZhbCd8XG4nQXBwcm92ZWQnfFxuJ1JlamVjdGVkJ3xcbidTY2hlZHVsZWQnfFxuJ1J1bmJvb2tJblByb2dyZXNzJ3xcbidQZW5kaW5nQ2hhbmdlQ2FsZW5kYXJPdmVycmlkZSd8XG4nQ2hhbmdlQ2FsZW5kYXJPdmVycmlkZUFwcHJvdmVkJ3xcbidDaGFuZ2VDYWxlbmRhck92ZXJyaWRlUmVqZWN0ZWQnfFxuJ0NvbXBsZXRlZFdpdGhTdWNjZXNzJ3xcbidDb21wbGV0ZWRXaXRoRmFpbHVyZSc7XG5cbi8qKlxuICogRXhlY3V0ZUF1dG9tYXRpb24gaW1wbGVtZW50YXRpb24gdXNpbmcgQVdTIEFQSVxuICovXG5leHBvcnQgY2xhc3MgQXBpRXhlY3V0ZUF1dG9tYXRpb25Ib29rIGltcGxlbWVudHMgSUV4ZWN1dGVBdXRvbWF0aW9uSG9vayB7XG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IFJ1bm5pbmdTdGF0dXNlczogRXhlY3V0aW9uU3RhdHVzW10gPSBbJ1BlbmRpbmcnLCAnSW5Qcm9ncmVzcycsICdDYW5jZWxsaW5nJywgJ1dhaXRpbmcnXTtcblxuICByZWFkb25seSBwcm9wczogQXBpRXhlY3V0ZUF1dG9tYXRpb25Qcm9wcztcblxuICBjb25zdHJ1Y3Rvcihhd3NJbnZva2VyOiBJQXdzSW52b2tlciwgc2xlZXBIb29rOiBJU2xlZXBIb29rKSB7XG4gICAgdGhpcy5wcm9wcyA9IHsgYXdzSW52b2tlciwgc2xlZXBIb29rIH07XG4gIH1cblxuICBwdWJsaWMgZXhlY3V0ZShwcm9wczogRXhlY3V0ZUF1dG9tYXRpb25Qcm9wcyk6IEV4ZWN1dGVBdXRvbWF0aW9uT3V0cHV0cyB7XG4gICAgY29uc29sZS5sb2coYEV4ZWN1dGVBdXRvbWF0aW9uOiBDaGVja2luZyBleGVjdXRpb24gaW5wdXRzIGZvciAke3Byb3BzLmRvY3VtZW50TmFtZX1gKTtcbiAgICB0aGlzLnBhcmFtZXRlclZhbGlkYXRpb24ocHJvcHMpO1xuICAgIGNvbnNvbGUubG9nKGBFeGVjdXRlQXV0b21hdGlvbjogU3RhcnRpbmcgZXhlY3V0aW9uIG9mICR7cHJvcHMuZG9jdW1lbnROYW1lfWApO1xuICAgIGNvbnN0IGV4ZWN1dGlvbklkID0gdGhpcy5zdGFydEV4ZWN1dGlvbihwcm9wcyk7XG4gICAgY29uc29sZS5sb2coYEV4ZWN1dGVBdXRvbWF0aW9uOiBXYWl0aW5nIGZvciAke2V4ZWN1dGlvbklkfSB0byBjb21wbGV0ZWApO1xuICAgIGNvbnN0IHJlc3VsdCA9IHRoaXMud2FpdEZvckV4ZWN1dGlvbihleGVjdXRpb25JZCk7XG4gICAgY29uc29sZS5sb2coYEV4ZWN1dGVBdXRvbWF0aW9uOiAke2V4ZWN1dGlvbklkfSBmaW5pc2hlZGApO1xuXG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfVxuXG4gIHByaXZhdGUgcGFyYW1ldGVyVmFsaWRhdGlvbihwcm9wczogRXhlY3V0ZUF1dG9tYXRpb25Qcm9wcyk6IHZvaWQge1xuICAgIGlmIChwcm9wcy50YXJnZXRMb2NhdGlvbnMgJiYgKHByb3BzLnRhcmdldExvY2F0aW9ucy5sZW5ndGggPCAxIHx8IHByb3BzLnRhcmdldExvY2F0aW9ucy5sZW5ndGggPiAxMDApKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RhcmdldExvY2F0aW9ucyBtdXN0IGNvbnRhaW4gYXRsZWFzdCAxIGFuZCBubyBtb3JlIHRoYW4gMTAwIGl0ZW1zLicpO1xuICAgIH1cbiAgICBpZiAocHJvcHMudGFyZ2V0TWFwcyAmJiBwcm9wcy50YXJnZXRzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RhcmdldE1hcHMgYW5kIFRhcmdldHMgY2Fubm90IGJvdGggYmUgc3BlY2lmaWVkLicpO1xuICAgIH1cbiAgICBpZiAoKHByb3BzLnRhcmdldHMgJiYgIXByb3BzLnRhcmdldFBhcmFtZXRlck5hbWUpIHx8ICghcHJvcHMudGFyZ2V0cyAmJiBwcm9wcy50YXJnZXRQYXJhbWV0ZXJOYW1lKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUYXJnZXRQYXJhbWV0ZXJOYW1lIGFuZCBUYXJnZXRzIG11c3QgYm90aCBiZSBzcGVjaWZpZWQgdG9nZXRoZXIuJyk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBzdGFydEV4ZWN1dGlvbihwcm9wczogRXhlY3V0ZUF1dG9tYXRpb25Qcm9wcyk6IHN0cmluZyB7XG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbihuZXcgQXdzQXBpU3RlcChuZXcgU3RhY2soKSwgJ2V4ZWN1dGUnLCB7XG4gICAgICBzZXJ2aWNlOiAnU1NNJyxcbiAgICAgIHBhc2NhbENhc2VBcGk6ICdTdGFydEF1dG9tYXRpb25FeGVjdXRpb24nLFxuICAgICAgYXBpUGFyYW1zOiB0aGlzLmdldEV4ZWN1dGVQcm9wcyhwcm9wcyksXG4gICAgICBvdXRwdXRzOiBbe1xuICAgICAgICBvdXRwdXRUeXBlOiBEYXRhVHlwZUVudW0uU1RSSU5HLFxuICAgICAgICBuYW1lOiAnZXhlY3V0aW9uSWQnLFxuICAgICAgICBzZWxlY3RvcjogJyQuQXV0b21hdGlvbkV4ZWN1dGlvbklkJyxcbiAgICAgIH1dLFxuICAgIH0pLCB0aGlzLnByb3BzKS5pbnZva2Uoe30pO1xuICAgIGlmIChyZXN1bHQucmVzcG9uc2VDb2RlICE9PSBSZXNwb25zZUNvZGUuU1VDQ0VTUykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBGYWlsZWQgdG8gc3RhcnQgZXhlY3V0aW9uOiAke3Jlc3VsdC5zdGFja1RyYWNlfWApO1xuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0Lm91dHB1dHM/LlsnZXhlY3V0ZS5leGVjdXRpb25JZCddO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRFeGVjdXRlUHJvcHMocHJvcHM6IEV4ZWN1dGVBdXRvbWF0aW9uUHJvcHMpOiBSZWNvcmQ8c3RyaW5nLCBhbnk+IHtcbiAgICBjb25zdCBhcGlQYXJhbU1hcDogUmVjb3JkPHN0cmluZywgYW55PiA9IHtcbiAgICAgIERvY3VtZW50TmFtZTogcHJvcHMuZG9jdW1lbnROYW1lLFxuICAgICAgRG9jdW1lbnRWZXJzaW9uOiBwcm9wcy5kb2N1bWVudFZlcnNpb24sXG4gICAgICBUYXJnZXRQYXJhbWV0ZXJOYW1lOiBwcm9wcy50YXJnZXRQYXJhbWV0ZXJOYW1lLFxuICAgICAgTWF4Q29uY3VycmVuY3k6IHByb3BzLm1heENvbmN1cnJlbmN5LFxuICAgICAgTWF4RXJyb3JzOiBwcm9wcy5tYXhFcnJvcnMsXG4gICAgICBUYXJnZXRzOiBwcm9wcy50YXJnZXRzLFxuICAgICAgVGFnczogcHJvcHMudGFncyxcbiAgICAgIFRhcmdldExvY2F0aW9uczogcHJvcHMudGFyZ2V0TG9jYXRpb25zLFxuICAgIH07XG4gICAgY29uc3QgYXBpUGFyYW1zOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge307XG4gICAgZm9yIChjb25zdCBrZXkgb2YgT2JqZWN0LmtleXMoYXBpUGFyYW1NYXApKSB7XG4gICAgICBjb25zdCB2YWx1ZSA9IGFwaVBhcmFtTWFwW2tleV07XG4gICAgICBpZiAodmFsdWUgPT09IHVuZGVmaW5lZCkgeyBjb250aW51ZTsgfVxuICAgICAgYXBpUGFyYW1zW2tleV0gPSB2YWx1ZTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMucnVudGltZVBhcmFtZXRlcnMpIHtcbiAgICAgIGFwaVBhcmFtcy5QYXJhbWV0ZXJzID0gdGhpcy5hZGFwdEV4ZWN1dGVQYXJhbWV0ZXJzKHByb3BzLnJ1bnRpbWVQYXJhbWV0ZXJzKTtcbiAgICB9XG4gICAgaWYgKHByb3BzLnRhcmdldE1hcHMpIHtcbiAgICAgIGFwaVBhcmFtcy5UYXJnZXRNYXBzID0gdGhpcy5hZGFwdFRhcmdldE1hcHMocHJvcHMudGFyZ2V0TWFwcyk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGFwaVBhcmFtcztcbiAgfVxuXG4gIHByaXZhdGUgYWRhcHRFeGVjdXRlUGFyYW1ldGVycyhwYXJhbWV0ZXJzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+KTogUmVjb3JkPHN0cmluZywgc3RyaW5nW10+IHtcbiAgICBjb25zdCByZXN1bHQ6IFJlY29yZDxzdHJpbmcsIHN0cmluZ1tdPiA9IHt9O1xuICAgIGZvciAoY29uc3Qga2V5IG9mIE9iamVjdC5rZXlzKHBhcmFtZXRlcnMpKSB7XG4gICAgICBjb25zdCB2YWx1ZSA9IHBhcmFtZXRlcnNba2V5XTtcbiAgICAgIGlmIChpc1N0cmluZyh2YWx1ZSkpIHtcbiAgICAgICAgcmVzdWx0W2tleV0gPSBbdmFsdWVdO1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cbiAgICAgIGlmIChpc0Jvb2xlYW4odmFsdWUpIHx8IGlzTnVtYmVyKHZhbHVlKSkge1xuICAgICAgICByZXN1bHRba2V5XSA9IFtgJHt2YWx1ZX1gXTtcbiAgICAgICAgY29udGludWU7XG4gICAgICB9XG4gICAgICBpZiAoaXNTdHJpbmdMaXN0KHZhbHVlKSkge1xuICAgICAgICByZXN1bHRba2V5XSA9IHZhbHVlO1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cbiAgICAgIGlmIChpc01hcExpc3QodmFsdWUpKSB7XG4gICAgICAgIHJlc3VsdFtrZXldID0gdmFsdWUubWFwKHggPT4gSlNPTi5zdHJpbmdpZnkoeCkpO1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cbiAgICAgIGlmIChpc1N0cmluZ01hcCh2YWx1ZSkpIHtcbiAgICAgICAgcmVzdWx0W2tleV0gPSBbSlNPTi5zdHJpbmdpZnkodmFsdWUpXTtcbiAgICAgICAgY29udGludWU7XG4gICAgICB9XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFBhcmFtZXRlciAke2tleX0gbXVzdCBiZSBhIEJvb2xlYW4sIE51bWJlciwgU3RyaW5nLCBTdHJpbmdMaXN0LCBTdHJpbmdNYXAsIG9yIE1hcExpc3QuYCk7XG4gICAgfVxuICAgIHJldHVybiByZXN1bHQ7XG4gIH1cblxuICBwcml2YXRlIGFkYXB0VGFyZ2V0TWFwcyh0YXJnZXRNYXBzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+W10pOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmdbXT5bXSB7XG4gICAgcmV0dXJuIHRhcmdldE1hcHMubWFwKHRoaXMuYWRhcHRFeGVjdXRlUGFyYW1ldGVycyk7XG4gIH1cblxuICBwcml2YXRlIHdhaXRGb3JFeGVjdXRpb24oZXhlY3V0aW9uSWQ6IHN0cmluZyk6IEV4ZWN1dGVBdXRvbWF0aW9uT3V0cHV0cyB7XG4gICAgbGV0IGV4ZWN1dGlvbiA9IHRoaXMuZ2V0RXhlY3V0aW9uKGV4ZWN1dGlvbklkKTtcbiAgICB3aGlsZSAoQXBpRXhlY3V0ZUF1dG9tYXRpb25Ib29rLlJ1bm5pbmdTdGF0dXNlcy5pbmNsdWRlcyhleGVjdXRpb24uQXV0b21hdGlvbkV4ZWN1dGlvblN0YXR1cykpIHtcbiAgICAgIGNvbnNvbGUubG9nKGBBdXRvbWF0aW9uICR7ZXhlY3V0aW9uSWR9IGlzIG5vdCBjb21wbGV0ZS4gU3RhdHVzOiAke2V4ZWN1dGlvbi5BdXRvbWF0aW9uRXhlY3V0aW9uU3RhdHVzfS5gKTtcbiAgICAgIG5ldyBBdXRvbWF0aW9uU3RlcFNpbXVsYXRpb24obmV3IFNsZWVwU3RlcChuZXcgU3RhY2soKSwgJ3NsZWVwJywge1xuICAgICAgICBzbGVlcFNlY29uZHM6IDIsXG4gICAgICB9KSwgdGhpcy5wcm9wcykuaW52b2tlKHt9KTtcbiAgICAgIGV4ZWN1dGlvbiA9IHRoaXMuZ2V0RXhlY3V0aW9uKGV4ZWN1dGlvbklkKTtcbiAgICB9XG4gICAgaWYgKGV4ZWN1dGlvbi5BdXRvbWF0aW9uRXhlY3V0aW9uU3RhdHVzICE9PSAnU3VjY2VzcycpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJHtleGVjdXRpb25JZH0gZW5kZWQgaW4gc3RhdHVzICR7ZXhlY3V0aW9uLkF1dG9tYXRpb25FeGVjdXRpb25TdGF0dXN9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHtcbiAgICAgIGV4ZWN1dGlvbklkOiBleGVjdXRpb25JZCxcbiAgICAgIHN0YXR1czogZXhlY3V0aW9uLkF1dG9tYXRpb25FeGVjdXRpb25TdGF0dXMsXG4gICAgICBvdXRwdXQ6IE9iamVjdC52YWx1ZXMoZXhlY3V0aW9uLk91dHB1dHMpLmZsYXQoKSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRFeGVjdXRpb24oZXhlY3V0aW9uSWQ6IHN0cmluZyk6IEF1dG9tYXRpb25FeGVjdXRpb24ge1xuICAgIGNvbnN0IHJlc3VsdCA9IG5ldyBBdXRvbWF0aW9uU3RlcFNpbXVsYXRpb24obmV3IEF3c0FwaVN0ZXAobmV3IFN0YWNrKCksICdnZXRFeGVjdXRpb24nLCB7XG4gICAgICBzZXJ2aWNlOiAnU1NNJyxcbiAgICAgIHBhc2NhbENhc2VBcGk6ICdHZXRBdXRvbWF0aW9uRXhlY3V0aW9uJyxcbiAgICAgIGFwaVBhcmFtczoge1xuICAgICAgICBBdXRvbWF0aW9uRXhlY3V0aW9uSWQ6IGV4ZWN1dGlvbklkLFxuICAgICAgfSxcbiAgICAgIG91dHB1dHM6IFt7XG4gICAgICAgIG91dHB1dFR5cGU6IERhdGFUeXBlRW51bS5TVFJJTkdfTUFQLFxuICAgICAgICBuYW1lOiAnZXhlY3V0aW9uJyxcbiAgICAgICAgc2VsZWN0b3I6ICckLkF1dG9tYXRpb25FeGVjdXRpb24nLFxuICAgICAgfV0sXG4gICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgaWYgKHJlc3VsdC5yZXNwb25zZUNvZGUgIT09IFJlc3BvbnNlQ29kZS5TVUNDRVNTKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEZhaWxlZCB0byBnZXQgZXhlY3V0aW9uICR7ZXhlY3V0aW9uSWR9IHN0YXR1czogJHtyZXN1bHQuc3RhY2tUcmFjZX1gKTtcbiAgICB9XG4gICAgcmV0dXJuIHJlc3VsdC5vdXRwdXRzPy5bJ2dldEV4ZWN1dGlvbi5leGVjdXRpb24nXTtcbiAgfVxufSJdfQ==