"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const scalable_target_1 = require("./scalable-target");
/**
 * Represent an attribute for which autoscaling can be configured
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *   Step Scaling, so the Dynamo subclass won't expose this method).
 */
class BaseScalableAttribute extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity,
        });
    }
    /**
     * Scale out or in based on time
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,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