from __future__ import annotations

from copy import deepcopy
from typing import Any, Dict, Generic, Optional, TypeVar, Union

from pydantic import BaseModel, Field, model_validator
from pydantic.json import pydantic_encoder

from rss_parser.models import XMLBaseModel
from rss_parser.models.utils import snake_case

T = TypeVar("T")


class Tag(BaseModel, Generic[T]):
    """
    >>> from rss_parser.models import XMLBaseModel
    >>> from rss_parser.models.types.tag import Tag
    >>> class Model(XMLBaseModel):
    ...     width: Tag[int]
    ...     category: Tag[str]
    >>> m = Model(
    ...     width=48,
    ...     category={"@someAttribute": "https://example.com", "#text": "valid string"},
    ... )
    >>> # Content value is an integer, as per the generic type
    >>> m.width.content
    48
    >>> type(m.width), type(m.width.content)
    (<class 'rss_parser.models.rss.image.Tag[int]'>, <class 'int'>)
    >>> # The attributes are empty by default
    >>> m.width.attributes
    {}
    >>> # But are populated when provided.
    >>> # Note that the @ symbol is trimmed from the beggining and name is convert to snake_case
    >>> m.category.attributes
    {'some_attribute': 'https://example.com'}
    >>> # Generic argument types are handled by pydantic - let's try to provide a string for a Tag[int] number
    >>> m = Model(width="not_a_number", category="valid_string")  # doctest: +IGNORE_EXCEPTION_DETAIL
    Traceback (most recent call last):
        ...
    ValidationError: 1 validation error for Model
    width -> content
      value is not a valid integer (type=type_error.integer)
    """

    # Optional in case of self-closing tags
    content: Optional[T] = None
    attributes: Dict[str, Any] = Field(default_factory=dict)

    def __getattr__(self, item):
        """Forward default getattr for content for simplicity."""
        return getattr(self.content, item)

    def __getitem__(self, key):
        return self.content[key]

    def __setitem__(self, key, value):
        self.content[key] = value

    @model_validator(mode="before")
    @classmethod
    def pre_convert(cls, value: Union[T, dict, "Tag[T]"]) -> Union["Tag[T]", Dict[str, Any]]:
        """Used to split tag's text with other xml attributes."""
        if isinstance(value, cls):
            return value

        if isinstance(value, dict):
            data = deepcopy(value)
            attributes = {snake_case(k.lstrip("@")): v for k, v in data.items() if k.startswith("@")}
            content = data.pop("#text", data) if len(attributes) != len(data) else None
            return {"content": content, "attributes": attributes}

        return {"content": value, "attributes": {}}

    @classmethod
    def flatten_tag_encoder(cls, value):
        """Encoder that translates Tag objects (dict) to plain .content values (T)."""
        if isinstance(value, XMLBaseModel):
            return value.dict_plain()
        if isinstance(value, Tag):
            return value.content

        return pydantic_encoder(value)
