# Author: Miasnenko Dmitry
# Github: https://github.com/YoungMeatBoy
# Made for labs in asm (masm)
# It was impossible to find comment aligner for
# asm so I decided to implement by myself
#
# The idea of an implementation:
#       Comment aligner uses FileManager class from alcom.filemanager.py file
#       Input from command line by user
#       Examples:
#           1) alcom
#           2) alcom -f filename.py
#       First will align all .asm files in dir and subdirs
#       Second will just align comments in one file
#
#       CommentAligner constructor params:
#           comment_splitter - splitter between comment and line
#               default = ';'
#           comment_offset   - space between comment and comment_splitter
#               default = 4
#           splitter_offset  - ofsset of comment_splitter from line
#               default = 10
#
#           mov ax, bx----------;----moves bx into ax
#                         ▲     ▲  ▲
#       splitter_offset = 10    ▲  comment_offset = 4
#                               ▲
#                      comment_splitter
#
#       CommentAligner class has two public method:
#           CommentAligner.align_comments(filpath:pathlib.Path, encoding) -> None
#           CommentAligner.align_directory(dirpath:pathlib.Path) -> None
#
#
#       How comment aligner works:
#           1) Finds longest line in file with method self.fileworker.find_longest_line
#              instead of len key for max uses self.__line_len__
#              It splits line by comment_splitter and return s len of code part
#           2) Then it creates temp unique file with uuid.uuid().hex
#           3) for each line in file:
#                   write line generated by self.__align_comments_in_line__ to a new temp file
#                   params:
#                       line:str
#                       longest_line:int
#           4) Remove old file and rename temp to its original name
#
#
#        How line is aligned:
#           Params given:
#           longest_line_len = 20
#           line = '    mov ax, bx ;   move ax to bx ; test second splitter'
#
#           first split line in line and comment
#               line, comment = self.__parse_line__(line) -> Tuple[str, str]
#               line    = '    mov ax, bx'
#               comment = 'move ax to bx ; test second splitter'
#
#           if comment is empty and line is empty:
#               the line is empty and we return ''
#
#           if line is empty and ther is only comment:
#              we return ';' + self.comment_offset + comment
#
#           if line and comment:
#               calculate len of the line
#               line_len = len(line) = len('    mov ax, bx') = 14
#               claculate offset of the splitter from line
#               splitter_offset = ' ' * (longest_line_len - line_len)
#               return line + splitter_offset + self.comment_splitter + self.comment_offset + comment


import argparse
import pathlib
import uuid
from typing import Callable, Generator, Tuple
from accessify import private, implements
try:
	from alcom.filemanager import FileManager
except:
	from filemanager import FileManager


class CommentsAlignerInterface:
	def align_comments(self, filepath:pathlib.Path, encoding:str='utf-8') -> None:
		pass
	def align_directories(self, dirpath:pathlib.Path) -> None:
		pass
	def align_no_blank_comments(self, line:str, longest_line:int) -> None:
		pass
	def align_comments_in_line(self, line:str, longest_line:int) -> str:
		pass

@implements(CommentsAlignerInterface)
class CommentsAligner:
	def __init__(self, align_function:Callable, comment_splitter:str=';', comment_offset:int=4, splitter_offset=10) -> None:
		self.comment_splitter:str         = comment_splitter
		self.comment_offset  :int         = " " * comment_offset
		self.splitter_offset :str         = " " * splitter_offset
		self.filemanager     :FileManager = FileManager()
		self.align_function = align_function

	# @brief Aligns comments in file
	# @param[in] filepath pathlib.Path - file to be aligned
	# @param[in] encoding str          - encoding of the file
	# @return None
	def align_comments(self, filepath:pathlib.Path, encoding:str='utf-8') -> None:
		longest_line:int          = self.filemanager.find_longest_line(filepath, self.__line_len__) + 10
		temp_filepath    :pathlib.Path = self.filemanager.generate_temp_file(filepath)
		with open(filepath, encoding = encoding) as f:
			with open(temp_filepath, 'w',encoding = encoding) as new:
				for line in f:
					line = line.expandtabs(tabsize=4)
					new.write(self.align_function(self, line, longest_line) + '\n')
		filepath.unlink()
		temp_filepath.rename(filepath)

	# @brief Aligns all files in directory
	# param[in] directory pathlib.Path - directory with files to be align
	# @return None
	def align_directories(self, dirpath:pathlib.Path) -> None:
		for file in self.filemanager.find_files_with_extension(dirpath, '*.asm'):
			self.align_comments(file)

	# @brief returns len of the line with code (no comment)
	# param[in] line str line to be parsed
	# @return int
	def __line_len__(self, line:str) -> int:
		_line_, comment = self.__parse_line__(line)
		return len(_line_.rstrip())

	# @brief parses line with code and comment
	# param[in] line str line to be parsed
	# @return Tuple[str, str]
	# Method is private
	@private
	def __parse_line__(self, line:str) -> Tuple[str, str]:
		line, *comments = line.split(self.comment_splitter)
		comment:str = self.comment_splitter.join(comments)
		return line, comment.strip()

	# @brief Align comments in given line
	# param[in] line str line to be aligned
	# param[in] longest_line int len of the longest line in th file
	# @return aligned line str
	# Method is private
	def align_comments_in_line(self, line:str, longest_line:int) -> str:
		line, comment = self.__parse_line__(line)
		line = line.rstrip()
		offset = " " * (longest_line - len(line)) + self.splitter_offset
		if not line and not comment:
			return ''
		elif not line:
			return f"{offset};{self.comment_offset}{comment.strip()}"
		comment = f'{self.comment_splitter}{self.comment_offset}{comment}'
		line = self.set_tabs(line)
		line = f"{line}{offset}{comment}"
		return line

	def align_no_blank_comments(self, line:str, longest_line:int) -> str:
		line, comment = self.__parse_line__(line)
		line = line.rstrip()
		offset = " " * (longest_line - len(line)) + self.splitter_offset
		if not line and not comment:
			return ''
		elif not line:
			return f"{offset};{self.comment_offset}{comment.strip()}"
		elif line and not comment:
			return self.set_tabs(line)
		else:
			comment = f'{self.comment_splitter}{self.comment_offset}{comment}'
			line = self.set_tabs(line)
			line = f"{line}{offset}{comment}"
			return line

	def set_tabs(self, line) -> str:
		return line.replace('    ', '\t')