# fflogsapi

fflogsapi is a lazy Python 3 client for [fflogs](https://www.fflogs.com/)' v2 API with query caching functionality.

[![Tests](https://github.com/halworsen/fflogsapi/actions/workflows/test.yml/badge.svg?branch=master)](https://github.com/halworsen/fflogsapi/actions/workflows/test.yml)
[![Linting](https://github.com/halworsen/fflogsapi/actions/workflows/lint.yml/badge.svg?branch=master)](https://github.com/halworsen/fflogsapi/actions/workflows/lint.yml)
[![codecov](https://codecov.io/gh/halworsen/fflogsapi/branch/master/graph/badge.svg?token=YTEGMDJOGL)](https://codecov.io/gh/halworsen/fflogsapi)
[![pypi](https://shields.io/pypi/v/fflogsapi)](https://pypi.org/project/fflogsapi/)

---

## Features

* Retrieve information from FF Logs' v2 GraphQL API
  * Including private information only accessible through the user API
* Lazy evaluation
  * Queries for data are not executed until it is explicitly needed
* Query caching
  * Requesting the same data twice will instead fetch the result from cache
  * Customizable cache lifetime and options to ignore cached results
* Sensible interfaces to parts of the API that aren't well defined in the schema

## Example usage

```python
from config import CLIENT_ID, CLIENT_SECRET

from fflogsapi.client import FFLogsClient

client = FFLogsClient(CLIENT_ID, CLIENT_SECRET)
report = client.get_report('rGARYmQwTKbahXz9')

for fight in report:
    print(f'Fight #{fight.fight_id}:', fight.name(), f'- Kill: {fight.is_kill()}')
    pot_table = fight.table(filters={'sourceAurasPresent': 'Medicated'})
    potted_damage = 0
    for damage in pot_table['damageDone']:
        potted_damage += damage['total']
    print(f'Damage done under pots: {potted_damage}')
    if not fight.is_kill():
        print(f'Percentage reached: {fight.percentage()}')

client.close()
client.save_cache()
```

```python
from config import CLIENT_ID, CLIENT_SECRET

from fflogsapi.client import FFLogsClient

client = FFLogsClient(CLIENT_ID, CLIENT_SECRET)
for page in client.report_pages(filters={ 'guildID': 80551 }):
    print(f'Reports in page: {page.count()}')
    for report in page:
        print(report.title(), f'Duration: {report.duration()}')

client.close()
client.save_cache()
```

## User mode

The default mode of the client is 'client' mode, which uses the public API. This is by far the most
convenient method to use the client, and usually provides enough functionality for the majority of
use cases.

If you need access to private information, however, it is possible to use the client in user mode,
granting access to extra information such as private reports provided by the user API.

To use user mode, you must first specify `https://localhost:4433` as the redirect URL in your API
client on FF Logs. Then, provide the `mode='user'` kwarg to the client when instantiating it:
```python
client = FFLogsClient(CLIENT_ID, CLIENT_SECRET, mode='user')
```

This will have the client popup a browser window for the user for login, after which the client has access to the
user API. Note that the client will generate a self-signed certificate to serve the redirect.
Your browser will likely produce a warning about this, although it is safe to ignore.
