"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsSvcs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const apigwy = require("@aws-cdk/aws-apigatewayv2-alpha");
const apigwyAuth = require("@aws-cdk/aws-apigatewayv2-authorizers-alpha");
const apigwyint = require("@aws-cdk/aws-apigatewayv2-integrations-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps Services construct, including the Deployer
 * and Router Lambda Functions, and the DynamoDB Table used by both.
 */
class MicroAppsSvcs extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        const { bucketApps, bucketAppsOAI, bucketAppsStaging, s3PolicyBypassAROAs = [], s3PolicyBypassPrincipalARNs = [], s3StrictBucketPolicy = false, appEnv, httpApi, removalPolicy, assetNameRoot, assetNameSuffix, rootPathPrefix = '', requireIAMAuthorization = true, } = props;
        if (s3StrictBucketPolicy === true) {
            if (s3PolicyBypassAROAs.length === 0 && s3PolicyBypassPrincipalARNs.length === 0) {
                throw new Error('s3StrictBucketPolicy cannot be true without specifying at least one s3PolicyBypassAROAs or s3PolicyBypassPrincipalARNs');
            }
        }
        //
        // DynamoDB Table
        //
        if (props.table === undefined) {
            // Create able if none passed
            this._ownedTable = new dynamodb.Table(this, 'table', {
                tableName: assetNameRoot
                    ? `${assetNameRoot}${assetNameSuffix}`
                    : aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
                billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
                partitionKey: {
                    name: 'PK',
                    type: dynamodb.AttributeType.STRING,
                },
                sortKey: {
                    name: 'SK',
                    type: dynamodb.AttributeType.STRING,
                },
                removalPolicy,
            });
            this._table = this._ownedTable;
        }
        else {
            this._table = props.table;
        }
        //
        // Deployer Lambda Function
        //
        // Create Deployer Lambda Function
        const iamRoleUploadName = assetNameRoot
            ? `${assetNameRoot}-deployer-upload${assetNameSuffix}`
            : undefined;
        const deployerFuncName = assetNameRoot
            ? `${assetNameRoot}-deployer${assetNameSuffix}`
            : undefined;
        const deployerFuncProps = {
            functionName: deployerFuncName,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_16_X,
            timeout: aws_cdk_lib_1.Duration.seconds(15),
            environment: {
                NODE_ENV: appEnv,
                ...(httpApi ? { APIGWY_ID: httpApi.httpApiId } : {}),
                DATABASE_TABLE_NAME: this._table.tableName,
                FILESTORE_STAGING_BUCKET: bucketAppsStaging.bucketName,
                FILESTORE_DEST_BUCKET: bucketApps.bucketName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
                REQUIRE_IAM_AUTHORIZATION: requireIAMAuthorization ? 'true' : 'false',
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist', 'index.js'))) {
            // This is for local dev
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-deployer', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-deployer')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else {
            this._deployerFunc = new lambdaNodejs.NodejsFunction(this, 'deployer-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-deployer', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                ...deployerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._deployerFunc.applyRemovalPolicy(removalPolicy);
        }
        // Give the Deployer access to DynamoDB table
        this._table.grantReadWriteData(this._deployerFunc);
        this._table.grant(this._deployerFunc, 'dynamodb:DescribeTable');
        //
        // Deloyer upload temp role
        // Deployer assumes this role with a limited policy to generate
        // an STS temp token to return to microapps-publish for the upload.
        //
        const iamRoleUpload = new iam.Role(this, 'deployer-upload-role', {
            roleName: iamRoleUploadName,
            inlinePolicies: {
                uploadPolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['s3:ListBucket'],
                            resources: [bucketAppsStaging.bucketArn],
                        }),
                        new iam.PolicyStatement({
                            actions: ['s3:PutObject', 's3:GetObject', 's3:AbortMultipartUpload'],
                            resources: [`${bucketAppsStaging.bucketArn}/*`],
                        }),
                    ],
                }),
            },
            assumedBy: this._deployerFunc.grantPrincipal,
        });
        this._deployerFunc.addEnvironment('UPLOAD_ROLE_NAME', iamRoleUpload.roleName);
        //
        // Update S3 permissions
        //
        // Create PrincipalARN List
        const s3PolicyBypassArnPrincipals = [];
        for (const arnPrincipal of s3PolicyBypassPrincipalARNs) {
            s3PolicyBypassArnPrincipals.push(new iam.ArnPrincipal(arnPrincipal));
        }
        // Create AROA List that matches assumed sessions
        const s3PolicyBypassAROAMatches = [];
        for (const aroa of s3PolicyBypassAROAs) {
            s3PolicyBypassAROAMatches.push(`${aroa}:*`);
        }
        // Deny apps from reading:
        // - If they are missing the microapp-name tag
        // - Anything outside of the folder that matches their microapp-name tag
        const policyDenyPrefixOutsideTag = new iam.PolicyStatement({
            sid: 'deny-prefix-outside-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                ...s3PolicyBypassArnPrincipals,
                this._deployerFunc.grantPrincipal,
            ],
            notResources: [
                `${bucketApps.bucketArn}/\${aws:PrincipalTag/microapp-name}/*`,
                bucketApps.bucketArn,
            ],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'false' },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyPrefixOutsideTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyDenyMissingTag = new iam.PolicyStatement({
            sid: 'deny-missing-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                // Exclude the Deployer Function directly
                this._deployerFunc.grantPrincipal,
                // 2021-12-04 - Not 100% sure that this is actually needed...
                // Let's test this and remove if actually not necessary
                new iam.ArnPrincipal(`arn:aws:sts::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:assumed-role/${(_b = this._deployerFunc.role) === null || _b === void 0 ? void 0 : _b.roleName}/${this._deployerFunc.functionName}`),
                ...s3PolicyBypassArnPrincipals,
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'true' },
                // Note: This AROA must be specified to prevent this policy from locking
                // out non-root sessions that have assumed the admin role.
                // The notPrincipals will only match the role name exactly and will not match
                // any session that has assumed the role since notPrincipals does not allow
                // wildcard matches and does not do them implicitly either.
                // The AROA must be used because there are only 3 Principal variables:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_variables.html#principaltable
                //  aws:username, aws:userid, aws:PrincipalTag
                // For an assumed role, aws:username is blank, aws:userid is:
                //  [unique id AKA AROA for Role]:[session name]
                // Table of unique ID prefixes such as AROA:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-prefixes
                // The name of the role is simply not available and if it was
                // we'd need to write a complicated comparison to make sure
                // that we didn't exclude the Deny tag from roles in other accounts.
                //
                // To get the AROA with the AWS CLI:
                //   aws iam get-role --role-name ROLE-NAME
                //   aws iam get-user -–user-name USER-NAME
                StringNotLike: { 'aws:userid': [aws_cdk_lib_1.Aws.ACCOUNT_ID, ...s3PolicyBypassAROAMatches] },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyMissingTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyCloudFrontAccess = new iam.PolicyStatement({
            sid: 'cloudfront-oai-access',
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            principals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        if (bucketApps.policy === undefined) {
            const document = new s3.BucketPolicy(this, 's3-policy', {
                bucket: bucketApps,
            }).document;
            document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                document.addStatements(policyDenyPrefixOutsideTag);
                document.addStatements(policyDenyMissingTag);
            }
        }
        else {
            bucketApps.policy.document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                bucketApps.policy.document.addStatements(policyDenyPrefixOutsideTag);
                bucketApps.policy.document.addStatements(policyDenyMissingTag);
            }
        }
        // Allow the Lambda to read from the staging bucket
        const policyReadListStaging = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            // FIXME: Allow Deployer to delete from Staging bucket
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:ListBucket'],
            resources: [`${bucketAppsStaging.bucketArn}/*`, bucketAppsStaging.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadListStaging);
        // Allow the Lambda to write to the target bucket and delete
        const policyReadWriteListTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:PutObject', 's3:ListBucket'],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadWriteListTarget);
        // Allow the deployer to get a temporary STS token
        const policyGetSTSToken = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:GetFederationToken'],
            resources: ['*'],
        });
        this._deployerFunc.addToRolePolicy(policyGetSTSToken);
        // Allow the deployer to assume the upload role
        const policyAssumeUpload = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:AssumeRole'],
            resources: [iamRoleUpload.roleArn],
        });
        this._deployerFunc.addToRolePolicy(policyAssumeUpload);
        //
        // Give Deployer permissions to create routes and integrations
        // on the API Gateway API.
        //
        // Grant the ability to List all APIs (we have to find it)
        const policyAPIList = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:GET'],
            resources: [`arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis`],
        });
        this._deployerFunc.addToRolePolicy(policyAPIList);
        if (httpApi) {
            // Grant full control over the API we created
            const policyAPIManage = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['apigateway:*'],
                resources: [
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:${httpApi.httpApiId}/*`,
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations/*`,
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations`,
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes`,
                    `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes/*`,
                ],
            });
            this._deployerFunc.addToRolePolicy(policyAPIManage);
        }
        // Grant full control over lambdas that indicate they are microapps
        const policyAPIManageLambdas = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['lambda:*'],
            resources: [
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*`,
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*:*`,
            ],
            conditions: {
                StringEqualsIfExists: { 'aws:ResourceTag/microapp-managed': 'true' },
            },
        });
        this._deployerFunc.addToRolePolicy(policyAPIManageLambdas);
        if (httpApi) {
            //
            // Router Lambda Function
            //
            // Create Router Lambda Function
            const routerFuncProps = {
                functionName: assetNameRoot ? `${assetNameRoot}-router${assetNameSuffix}` : undefined,
                memorySize: 1769,
                logRetention: logs.RetentionDays.ONE_MONTH,
                runtime: lambda.Runtime.NODEJS_16_X,
                timeout: aws_cdk_lib_1.Duration.seconds(15),
                environment: {
                    NODE_ENV: appEnv,
                    DATABASE_TABLE_NAME: this._table.tableName,
                    AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                    ROOT_PATH_PREFIX: rootPathPrefix,
                },
            };
            if (process.env.NODE_ENV === 'test' &&
                fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-router', 'dist', 'index.js'))) {
                // This is for local dev
                this._routerFunc = new lambda.Function(this, 'router-func', {
                    code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'dist')),
                    handler: 'index.handler',
                    ...routerFuncProps,
                });
            }
            else if (fs_1.existsSync(path.join(__dirname, 'microapps-router', 'index.js'))) {
                // This is for built apps packaged with the CDK construct
                this._routerFunc = new lambda.Function(this, 'router-func', {
                    code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-router')),
                    handler: 'index.handler',
                    ...routerFuncProps,
                });
            }
            else {
                // Create Router Lambda Layer
                const routerDataFiles = new lambda.LayerVersion(this, 'router-templates', {
                    code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'templates')),
                    removalPolicy,
                });
                this._routerFunc = new lambdaNodejs.NodejsFunction(this, 'router-func', {
                    entry: path.join(__dirname, '..', '..', 'microapps-router', 'src', 'index.ts'),
                    handler: 'handler',
                    bundling: {
                        minify: true,
                        sourceMap: true,
                    },
                    layers: [routerDataFiles],
                    ...routerFuncProps,
                });
            }
            if (removalPolicy !== undefined) {
                this._routerFunc.applyRemovalPolicy(removalPolicy);
            }
            const policyReadTarget = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:GetObject'],
                resources: [`${bucketApps.bucketArn}/*`],
            });
            for (const router of [this._routerFunc]) {
                router.addToRolePolicy(policyReadTarget);
                // Give the Router access to DynamoDB table
                this._table.grantReadData(router);
                this._table.grant(router, 'dynamodb:DescribeTable');
            }
            // Create alias for Router
            const routerAlias = this._routerFunc.addAlias('CurrentVersion');
            // This creates an integration and a router
            const route = new apigwy.HttpRoute(this, 'route-default', {
                httpApi,
                routeKey: apigwy.HttpRouteKey.DEFAULT,
                integration: new apigwyint.HttpLambdaIntegration('router-integration', routerAlias),
                authorizer: requireIAMAuthorization ? new apigwyAuth.HttpIamAuthorizer() : undefined,
            });
            let routeArn = route.routeArn;
            // Remove the trailing `/` on the ARN, which is not correct
            if (routeArn.endsWith('/')) {
                routeArn = routeArn.slice(0, routeArn.length - 1);
            }
            // Grant API Gateway permission to invoke the Lambda
            new lambda.CfnPermission(this, 'router-invoke', {
                action: 'lambda:InvokeFunction',
                functionName: this._routerFunc.functionName,
                principal: 'apigateway.amazonaws.com',
                sourceArn: routeArn,
            });
        }
    }
    get table() {
        return this._table;
    }
    get deployerFunc() {
        return this._deployerFunc;
    }
    get routerFunc() {
        return this._routerFunc;
    }
}
exports.MicroAppsSvcs = MicroAppsSvcs;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsSvcs[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsSvcs", version: "0.3.5-alpha.2" };
//# sourceMappingURL=data:application/json;base64,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