"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsCF = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cf = require("aws-cdk-lib/aws-cloudfront");
const cforigins = require("aws-cdk-lib/aws-cloudfront-origins");
const r53 = require("aws-cdk-lib/aws-route53");
const r53targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const ReverseDomain_1 = require("./utils/ReverseDomain");
/**
 * Create a new MicroApps CloudFront Distribution.
 */
class MicroAppsCF extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props must be set');
        }
        if ((props.r53Zone === undefined && props.domainNameEdge !== undefined) ||
            (props.r53Zone !== undefined && props.domainNameEdge === undefined)) {
            throw new Error('If either of r53Zone or domainNameEdge are set then the other must be set');
        }
        const { domainNameEdge, domainNameOrigin, httpApi, removalPolicy, certEdge, assetNameRoot, assetNameSuffix, r53Zone, bucketLogs, bucketAppsOrigin, rootPathPrefix, createAPIPathRoute = !!props.httpApi, createNextDataPathRoute = !!props.httpApi, edgeLambdas, originShieldRegion, } = props;
        const appOriginRequestPolicy = MicroAppsCF.createAPIOriginPolicy(this, {
            assetNameRoot,
            assetNameSuffix,
            domainNameEdge,
        });
        //
        // Determine URL of the origin FQDN
        //
        let httpOriginFQDN = 'invalid.pwrdrvr.com';
        if (domainNameOrigin !== undefined) {
            httpOriginFQDN = domainNameOrigin;
        }
        else if (httpApi) {
            httpOriginFQDN = `${httpApi.apiId}.execute-api.${aws_cdk_lib_1.Aws.REGION}.amazonaws.com`;
        }
        //
        // Get the Edge to Origin Lambdas
        //
        //
        // CloudFront Distro
        //
        const appOrigin = httpApi
            ? new cforigins.HttpOrigin(httpOriginFQDN, {
                protocolPolicy: cf.OriginProtocolPolicy.HTTPS_ONLY,
                originSslProtocols: [cf.OriginSslPolicy.TLS_V1_2],
                originShieldRegion,
            })
            : bucketAppsOrigin;
        this._cloudFrontDistro = new cf.Distribution(this, 'cft', {
            comment: assetNameRoot ? `${assetNameRoot}${assetNameSuffix}` : domainNameEdge,
            domainNames: domainNameEdge !== undefined ? [domainNameEdge] : undefined,
            certificate: certEdge,
            httpVersion: cf.HttpVersion.HTTP2,
            defaultBehavior: {
                allowedMethods: cf.AllowedMethods.ALLOW_ALL,
                cachePolicy: cf.CachePolicy.CACHING_DISABLED,
                compress: true,
                originRequestPolicy: appOriginRequestPolicy,
                origin: appOrigin,
                viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas,
            },
            enableIpv6: true,
            priceClass: cf.PriceClass.PRICE_CLASS_100,
            logBucket: bucketLogs,
            logFilePrefix: props.domainNameEdge
                ? `${ReverseDomain_1.reverseDomain(props.domainNameEdge)}/cloudfront-raw/`
                : undefined,
        });
        if (removalPolicy !== undefined) {
            this._cloudFrontDistro.applyRemovalPolicy(removalPolicy);
        }
        // Add routes to the CloudFront Distribution
        MicroAppsCF.addRoutes(scope, {
            appOrigin,
            bucketAppsOrigin,
            distro: this._cloudFrontDistro,
            appOriginRequestPolicy,
            rootPathPrefix,
            createAPIPathRoute,
            createNextDataPathRoute,
            edgeLambdas,
        });
        //
        // Create the edge name for the CloudFront distro
        //
        if (r53Zone !== undefined) {
            const rrAppsEdge = new r53.RecordSet(this, 'edge-arecord', {
                recordName: domainNameEdge,
                recordType: r53.RecordType.A,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this._cloudFrontDistro)),
                zone: r53Zone,
            });
            if (removalPolicy !== undefined) {
                rrAppsEdge.applyRemovalPolicy(removalPolicy);
            }
        }
    }
    /**
     * Create or get the origin request policy
     *
     * If a custom domain name is NOT used for the origin then a policy
     * will be created.
     *
     * If a custom domain name IS used for the origin then the ALL_VIEWER
     * policy will be returned.  This policy passes the Host header to the
     * origin, which is fine when using a custom domain name on the origin.
     *
     * @param _scope
     * @param _props
     */
    static createAPIOriginPolicy(_scope, _props) {
        // const { assetNameRoot, assetNameSuffix, domainNameEdge } = props;
        // let apigwyOriginRequestPolicy: cf.IOriginRequestPolicy = cf.OriginRequestPolicy.ALL_VIEWER;
        // if (domainNameEdge === undefined) {
        //   // When not using a custom domain name we must limit down the origin policy to
        //   // prevent it from passing the Host header (distribution_id.cloudfront.net) to
        //   // apigwy which will then reject it with a 403 because it does not match the
        //   // execute-api name that apigwy is expecting.
        //   //
        //   // 2021-12-28 - There is a bug in the name generation that causes the same asset
        //   // in different stacks to have the same generated name.  We have to make the id
        //   // in all cases to ensure the generated name is unique.
        //   apigwyOriginRequestPolicy = new cf.OriginRequestPolicy(
        //     scope,
        //     `apigwy-origin-policy-${Stack.of(scope).stackName}`,
        //     {
        //       comment: assetNameRoot ? `${assetNameRoot}-apigwy${assetNameSuffix}` : undefined,
        //       originRequestPolicyName: assetNameRoot
        //         ? `${assetNameRoot}-apigwy${assetNameSuffix}`
        //         : undefined,
        //       cookieBehavior: cf.OriginRequestCookieBehavior.all(),
        //       queryStringBehavior: cf.OriginRequestQueryStringBehavior.all(),
        //       // TODO: If signing is enabled this should forward all signature headers
        //       // TODO: If set to "cfront.OriginRequestHeaderBehavior.all()" then
        //       // `replaceHostHeader` must be set to true to prevent API Gateway from rejecting
        //       // the request
        //       // headerBehavior: cf.OriginRequestHeaderBehavior.allowList('user-agent', 'referer'),
        //       headerBehavior: cf.OriginRequestHeaderBehavior.all(),
        //     },
        //   );
        // }
        return cf.OriginRequestPolicy.ALL_VIEWER;
    }
    /**
     * Add API Gateway and S3 routes to an existing CloudFront Distribution
     * @param _scope
     * @param props
     */
    static addRoutes(_scope, props) {
        const { appOrigin: defaultOrigin, bucketAppsOrigin, distro, appOriginRequestPolicy, rootPathPrefix = '', createAPIPathRoute = false, createNextDataPathRoute = false, } = props;
        //
        // Add Behaviors
        //
        const s3BehaviorOptions = {
            allowedMethods: cf.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
            cachePolicy: cf.CachePolicy.CACHING_OPTIMIZED,
            compress: true,
            originRequestPolicy: cf.OriginRequestPolicy.CORS_S3_ORIGIN,
            viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        const appBehaviorOptions = {
            allowedMethods: cf.AllowedMethods.ALLOW_ALL,
            // TODO: Caching needs to be set by the app response
            cachePolicy: cf.CachePolicy.CACHING_DISABLED,
            compress: true,
            originRequestPolicy: appOriginRequestPolicy,
            viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            edgeLambdas: props.edgeLambdas,
        };
        //
        // If a route specifically has `/api/` in it, send it to API Gateway
        // This is needed to catch routes that have periods in the API path data,
        // such as: /release/0.0.0/api/update/default/release/0.0.0
        //
        if (createAPIPathRoute) {
            distro.addBehavior(path_1.posix.join(rootPathPrefix, '*/api/*'), defaultOrigin, appBehaviorOptions);
            distro.addBehavior(path_1.posix.join(rootPathPrefix, 'api/*'), defaultOrigin, appBehaviorOptions);
        }
        //
        // If a route specifically has `/_next/data/` in it, send it to API Gateway
        // This is needed to catch routes that have periods in the API path data,
        // such as: /release/0.0.0/_next/data/app.json
        //
        if (createNextDataPathRoute) {
            distro.addBehavior(
            // Note: send anything with _next/data after the appName (and optional version)
            // to the app origin as iframe-less will have no version before _next/data
            // in the path
            path_1.posix.join(rootPathPrefix, '*/_next/data/*'), defaultOrigin, appBehaviorOptions);
            distro.addBehavior(
            // Note: send anything with _next/data after the appName (and optional version)
            // to the app origin as iframe-less will have no version before _next/data
            // in the path
            path_1.posix.join(rootPathPrefix, '_next/data/*'), defaultOrigin, appBehaviorOptions);
        }
        //
        // All static assets are assumed to have a dot in them
        //
        distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*/*/*.*'), bucketAppsOrigin, s3BehaviorOptions);
        //
        // Root app static resources
        //
        distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*.*.*/*.*'), bucketAppsOrigin, s3BehaviorOptions);
        //
        // Everything that isn't a static asset is going to API Gateway
        // There is no trailing slash because Serverless Next.js wants
        // go load pages at /release/0.0.3 (with no trailing slash).
        //
        distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*'), defaultOrigin, appBehaviorOptions);
    }
    get cloudFrontDistro() {
        return this._cloudFrontDistro;
    }
}
exports.MicroAppsCF = MicroAppsCF;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsCF[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsCF", version: "0.3.5-alpha.2" };
//# sourceMappingURL=data:application/json;base64,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