"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsSvcs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const apigwy = require("@aws-cdk/aws-apigatewayv2-alpha");
const apigwyAuth = require("@aws-cdk/aws-apigatewayv2-authorizers-alpha");
const apigwyint = require("@aws-cdk/aws-apigatewayv2-integrations-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps Services construct, including the Deployer
 * and Router Lambda Functions, and the DynamoDB Table used by both.
 */
class MicroAppsSvcs extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        const { bucketApps, bucketAppsOAI, bucketAppsStaging, s3PolicyBypassAROAs = [], s3PolicyBypassPrincipalARNs = [], s3StrictBucketPolicy = false, appEnv, httpApi, removalPolicy, assetNameRoot, assetNameSuffix, rootPathPrefix = '', requireIAMAuthorization = true, } = props;
        if (s3StrictBucketPolicy === true) {
            if (s3PolicyBypassAROAs.length === 0 && s3PolicyBypassPrincipalARNs.length === 0) {
                throw new Error('s3StrictBucketPolicy cannot be true without specifying at least one s3PolicyBypassAROAs or s3PolicyBypassPrincipalARNs');
            }
        }
        //
        // DynamoDB Table
        //
        if (props.table === undefined) {
            // Create able if none passed
            this._ownedTable = new dynamodb.Table(this, 'table', {
                tableName: assetNameRoot ? `${assetNameRoot}${assetNameSuffix}` : undefined,
                billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
                partitionKey: {
                    name: 'PK',
                    type: dynamodb.AttributeType.STRING,
                },
                sortKey: {
                    name: 'SK',
                    type: dynamodb.AttributeType.STRING,
                },
                removalPolicy,
            });
            this._table = this._ownedTable;
        }
        else {
            this._table = props.table;
        }
        //
        // Router Lambda Function
        //
        // Create Router Lambda Function
        const routerFuncProps = {
            functionName: assetNameRoot ? `${assetNameRoot}-router${assetNameSuffix}` : undefined,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: aws_cdk_lib_1.Duration.seconds(15),
            environment: {
                NODE_ENV: appEnv,
                DATABASE_TABLE_NAME: this._table.tableName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-router', 'dist', 'index.js'))) {
            // This is for local dev
            this._routerFunc = new lambda.Function(this, 'router-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'dist')),
                handler: 'index.handler',
                ...routerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-router', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._routerFunc = new lambda.Function(this, 'router-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-router')),
                handler: 'index.handler',
                ...routerFuncProps,
            });
        }
        else {
            // Create Router Lambda Layer
            const routerDataFiles = new lambda.LayerVersion(this, 'router-templates', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'templates')),
                removalPolicy,
            });
            this._routerFunc = new lambdaNodejs.NodejsFunction(this, 'router-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-router', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                layers: [routerDataFiles],
                ...routerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._routerFunc.applyRemovalPolicy(removalPolicy);
        }
        const policyReadTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject'],
            resources: [`${bucketApps.bucketArn}/*`],
        });
        for (const router of [this._routerFunc]) {
            router.addToRolePolicy(policyReadTarget);
            // Give the Router access to DynamoDB table
            this._table.grantReadData(router);
            this._table.grant(router, 'dynamodb:DescribeTable');
        }
        // Create alias for Router
        const routerAlias = this._routerFunc.addAlias('CurrentVersion');
        //
        // Deployer Lambda Function
        //
        // Create Deployer Lambda Function
        const iamRoleUploadName = assetNameRoot
            ? `${assetNameRoot}-deployer-upload${assetNameSuffix}`
            : undefined;
        const deployerFuncName = assetNameRoot
            ? `${assetNameRoot}-deployer${assetNameSuffix}`
            : undefined;
        const deployerFuncProps = {
            functionName: deployerFuncName,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: aws_cdk_lib_1.Duration.seconds(15),
            environment: {
                NODE_ENV: appEnv,
                APIGWY_ID: httpApi.httpApiId,
                DATABASE_TABLE_NAME: this._table.tableName,
                FILESTORE_STAGING_BUCKET: bucketAppsStaging.bucketName,
                FILESTORE_DEST_BUCKET: bucketApps.bucketName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
                REQUIRE_IAM_AUTHORIZATION: requireIAMAuthorization ? 'true' : 'false',
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist', 'index.js'))) {
            // This is for local dev
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-deployer', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-deployer')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else {
            this._deployerFunc = new lambdaNodejs.NodejsFunction(this, 'deployer-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-deployer', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                ...deployerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._deployerFunc.applyRemovalPolicy(removalPolicy);
        }
        // Give the Deployer access to DynamoDB table
        this._table.grantReadWriteData(this._deployerFunc);
        this._table.grant(this._deployerFunc, 'dynamodb:DescribeTable');
        //
        // Deloyer upload temp role
        // Deployer assumes this role with a limited policy to generate
        // an STS temp token to return to microapps-publish for the upload.
        //
        const iamRoleUpload = new iam.Role(this, 'deployer-upload-role', {
            roleName: iamRoleUploadName,
            inlinePolicies: {
                uploadPolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['s3:ListBucket'],
                            resources: [bucketAppsStaging.bucketArn],
                        }),
                        new iam.PolicyStatement({
                            actions: ['s3:PutObject', 's3:GetObject', 's3:AbortMultipartUpload'],
                            resources: [`${bucketAppsStaging.bucketArn}/*`],
                        }),
                    ],
                }),
            },
            assumedBy: this._deployerFunc.grantPrincipal,
        });
        this._deployerFunc.addEnvironment('UPLOAD_ROLE_NAME', iamRoleUpload.roleName);
        //
        // Update S3 permissions
        //
        // Create PrincipalARN List
        const s3PolicyBypassArnPrincipals = [];
        for (const arnPrincipal of s3PolicyBypassPrincipalARNs) {
            s3PolicyBypassArnPrincipals.push(new iam.ArnPrincipal(arnPrincipal));
        }
        // Create AROA List that matches assumed sessions
        const s3PolicyBypassAROAMatches = [];
        for (const aroa of s3PolicyBypassAROAs) {
            s3PolicyBypassAROAMatches.push(`${aroa}:*`);
        }
        // Deny apps from reading:
        // - If they are missing the microapp-name tag
        // - Anything outside of the folder that matches their microapp-name tag
        const policyDenyPrefixOutsideTag = new iam.PolicyStatement({
            sid: 'deny-prefix-outside-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                ...s3PolicyBypassArnPrincipals,
                this._deployerFunc.grantPrincipal,
            ],
            notResources: [
                `${bucketApps.bucketArn}/\${aws:PrincipalTag/microapp-name}/*`,
                bucketApps.bucketArn,
            ],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'false' },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyPrefixOutsideTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyDenyMissingTag = new iam.PolicyStatement({
            sid: 'deny-missing-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                // Exclude the Deployer Function directly
                this._deployerFunc.grantPrincipal,
                // 2021-12-04 - Not 100% sure that this is actually needed...
                // Let's test this and remove if actually not necessary
                new iam.ArnPrincipal(`arn:aws:sts::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:assumed-role/${(_b = this._deployerFunc.role) === null || _b === void 0 ? void 0 : _b.roleName}/${this._deployerFunc.functionName}`),
                ...s3PolicyBypassArnPrincipals,
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'true' },
                // Note: This AROA must be specified to prevent this policy from locking
                // out non-root sessions that have assumed the admin role.
                // The notPrincipals will only match the role name exactly and will not match
                // any session that has assumed the role since notPrincipals does not allow
                // wildcard matches and does not do them implicitly either.
                // The AROA must be used because there are only 3 Principal variables:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_variables.html#principaltable
                //  aws:username, aws:userid, aws:PrincipalTag
                // For an assumed role, aws:username is blank, aws:userid is:
                //  [unique id AKA AROA for Role]:[session name]
                // Table of unique ID prefixes such as AROA:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-prefixes
                // The name of the role is simply not available and if it was
                // we'd need to write a complicated comparison to make sure
                // that we didn't exclude the Deny tag from roles in other accounts.
                //
                // To get the AROA with the AWS CLI:
                //   aws iam get-role --role-name ROLE-NAME
                //   aws iam get-user -–user-name USER-NAME
                StringNotLike: { 'aws:userid': [aws_cdk_lib_1.Aws.ACCOUNT_ID, ...s3PolicyBypassAROAMatches] },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyMissingTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyCloudFrontAccess = new iam.PolicyStatement({
            sid: 'cloudfront-oai-access',
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            principals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        if (bucketApps.policy === undefined) {
            const document = new s3.BucketPolicy(this, 's3-policy', {
                bucket: bucketApps,
            }).document;
            document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                document.addStatements(policyDenyPrefixOutsideTag);
                document.addStatements(policyDenyMissingTag);
            }
        }
        else {
            bucketApps.policy.document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                bucketApps.policy.document.addStatements(policyDenyPrefixOutsideTag);
                bucketApps.policy.document.addStatements(policyDenyMissingTag);
            }
        }
        // Allow the Lambda to read from the staging bucket
        const policyReadListStaging = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            // FIXME: Allow Deployer to delete from Staging bucket
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:ListBucket'],
            resources: [`${bucketAppsStaging.bucketArn}/*`, bucketAppsStaging.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadListStaging);
        // Allow the Lambda to write to the target bucket and delete
        const policyReadWriteListTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:PutObject', 's3:ListBucket'],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadWriteListTarget);
        // Allow the deployer to get a temporary STS token
        const policyGetSTSToken = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:GetFederationToken'],
            resources: ['*'],
        });
        this._deployerFunc.addToRolePolicy(policyGetSTSToken);
        // Allow the deployer to assume the upload role
        const policyAssumeUpload = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:AssumeRole'],
            resources: [iamRoleUpload.roleArn],
        });
        this._deployerFunc.addToRolePolicy(policyAssumeUpload);
        //
        // Give Deployer permissions to create routes and integrations
        // on the API Gateway API.
        //
        // Grant the ability to List all APIs (we have to find it)
        const policyAPIList = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:GET'],
            resources: [`arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis`],
        });
        this._deployerFunc.addToRolePolicy(policyAPIList);
        // Grant full control over the API we created
        const policyAPIManage = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:*'],
            resources: [
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:${httpApi.httpApiId}/*`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations/*`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes/*`,
            ],
        });
        this._deployerFunc.addToRolePolicy(policyAPIManage);
        // Grant full control over lambdas that indicate they are microapps
        const policyAPIManageLambdas = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['lambda:*'],
            resources: [
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*`,
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*:*`,
            ],
            conditions: {
                StringEqualsIfExists: { 'aws:ResourceTag/microapp-managed': 'true' },
            },
        });
        this._deployerFunc.addToRolePolicy(policyAPIManageLambdas);
        // This creates an integration and a router
        const route = new apigwy.HttpRoute(this, 'route-default', {
            httpApi,
            routeKey: apigwy.HttpRouteKey.DEFAULT,
            integration: new apigwyint.HttpLambdaIntegration('router-integration', routerAlias),
            authorizer: requireIAMAuthorization ? new apigwyAuth.HttpIamAuthorizer() : undefined,
        });
        let routeArn = route.routeArn;
        // Remove the trailing `/` on the ARN, which is not correct
        if (routeArn.endsWith('/')) {
            routeArn = routeArn.slice(0, routeArn.length - 1);
        }
        // Grant API Gateway permission to invoke the Lambda
        new lambda.CfnPermission(this, 'router-invoke', {
            action: 'lambda:InvokeFunction',
            functionName: this._routerFunc.functionName,
            principal: 'apigateway.amazonaws.com',
            sourceArn: routeArn,
        });
    }
    get table() {
        return this._table;
    }
    get deployerFunc() {
        return this._deployerFunc;
    }
    get routerFunc() {
        return this._routerFunc;
    }
}
exports.MicroAppsSvcs = MicroAppsSvcs;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsSvcs[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsSvcs", version: "0.2.12" };
//# sourceMappingURL=data:application/json;base64,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