"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsEdgeToOrigin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os = require("os");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cf = require("aws-cdk-lib/aws-cloudfront");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps Edge to Origin Function w/ `config.yml`
 */
class MicroAppsEdgeToOrigin extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props must be set');
        }
        const { removalPolicy, assetNameRoot, assetNameSuffix, signingMode = 'sign', addXForwardedHostHeader = true, replaceHostHeader = true, originRegion, } = props;
        // Create the edge function config file from the construct options
        const edgeToOriginConfigYaml = MicroAppsEdgeToOrigin.generateEdgeToOriginConfig({
            originRegion: originRegion || aws_cdk_lib_1.Aws.REGION,
            addXForwardedHostHeader,
            replaceHostHeader,
            signingMode: signingMode === 'none' ? '' : signingMode,
        });
        //
        // Create the Edge to Origin Function
        //
        const edgeToOriginFuncProps = {
            functionName: assetNameRoot ? `${assetNameRoot}-edge-to-origin${assetNameSuffix}` : undefined,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: aws_cdk_lib_1.Duration.seconds(5),
            initialPolicy: [
                // This can't have a reference to the httpApi because it would mean
                // the parent stack (this stack) has to be created before the us-east-1
                // child stack for the Edge Lambda Function.
                // That's why we use a tag-based policy to allow the Edge Function
                // to invoke any API Gateway API that we apply a tag to
                // We allow the edge function to sign for all regions since
                // we may use custom closest region in the future.
                new iam.PolicyStatement({
                    actions: ['execute-api:Invoke'],
                    resources: [`arn:aws:execute-api:*:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*/*/*/*`],
                }),
            ],
            ...(removalPolicy ? { removalPolicy } : {}),
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'dist', 'index.js'))) {
            // Emit the config file from the construct options
            fs_1.writeFileSync(path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'dist', 'config.yml'), edgeToOriginConfigYaml);
            // copyFileSync(
            //   path.join(__dirname, '..', '..', '..', 'configs', 'microapps-edge-to-origin', 'config.yml'),
            //   path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'dist', 'config.yml'),
            // );
            // This is for tests run under jest
            // This is also for anytime when the edge function has already been bundled
            this._edgeToOriginFunction = new cf.experimental.EdgeFunction(this, 'edge-to-apigwy-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'dist')),
                handler: 'index.handler',
                ...edgeToOriginFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-edge-to-origin', 'index.js'))) {
            // Emit the config file from the construct options
            fs_1.writeFileSync(path.join(__dirname, 'microapps-edge-to-origin', 'config.yml'), edgeToOriginConfigYaml);
            // This is for built apps packaged with the CDK construct
            this._edgeToOriginFunction = new cf.experimental.EdgeFunction(this, 'edge-to-apigwy-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-edge-to-origin')),
                handler: 'index.handler',
                ...edgeToOriginFuncProps,
            });
        }
        else {
            // Emit the config file from the construct options
            fs_1.writeFileSync(path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'config.yml'), edgeToOriginConfigYaml);
            // This builds the function for distribution with the CDK Construct
            // and will be used during local builds and PR builds of microapps-core
            // if the microapps-edge-to-origin function is not already bundled.
            // This will fail to deploy in any region other than us-east-1
            // We cannot use NodejsFunction because it will not create in us-east-1
            this._edgeToOriginFunction = new lambdaNodejs.NodejsFunction(this, 'edge-to-apigwy-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                    commandHooks: {
                        beforeInstall: () => [],
                        beforeBundling: () => [],
                        afterBundling: (_inputDir, outputDir) => {
                            return [
                                `${os.platform() === 'win32' ? 'copy' : 'cp'} ${path.join(__dirname, '..', '..', '..', 'configs', 'microapps-edge-to-origin', 'config.yml')} ${outputDir}`,
                            ];
                        },
                    },
                },
                ...edgeToOriginFuncProps,
            });
        }
        this._edgeToOriginLambdas = [
            {
                eventType: cf.LambdaEdgeEventType.ORIGIN_REQUEST,
                functionVersion: this._edgeToOriginFunction.currentVersion,
                includeBody: true,
            },
        ];
    }
    /**
     * Generate the yaml config for the edge lambda
     * @param props
     * @returns
     */
    static generateEdgeToOriginConfig(props) {
        return `originRegion: ${props.originRegion}
${props.signingMode === '' ? '' : `signingMode: ${props.signingMode}`}
addXForwardedHostHeader: ${props.addXForwardedHostHeader}
replaceHostHeader: ${props.replaceHostHeader}`;
    }
    get edgeToOriginFunction() {
        return this._edgeToOriginFunction;
    }
    get edgeToOriginLambdas() {
        return this._edgeToOriginLambdas;
    }
}
exports.MicroAppsEdgeToOrigin = MicroAppsEdgeToOrigin;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsEdgeToOrigin[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsEdgeToOrigin", version: "0.2.12" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiTWljcm9BcHBzRWRnZVRvT3JpZ2luLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL01pY3JvQXBwc0VkZ2VUb09yaWdpbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDJCQUErQztBQUMvQyx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLDZDQUEyRDtBQUMzRCxpREFBaUQ7QUFDakQsMkNBQTJDO0FBQzNDLGlEQUFpRDtBQUNqRCw4REFBOEQ7QUFDOUQsNkNBQTZDO0FBQzdDLDJDQUF1QztBQW1HdkM7O0dBRUc7QUFDSCxNQUFhLHFCQUFzQixTQUFRLHNCQUFTO0lBdUJsRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWlDO1FBQ3pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQztTQUN0QztRQUVELE1BQU0sRUFDSixhQUFhLEVBQ2IsYUFBYSxFQUNiLGVBQWUsRUFDZixXQUFXLEdBQUcsTUFBTSxFQUNwQix1QkFBdUIsR0FBRyxJQUFJLEVBQzlCLGlCQUFpQixHQUFHLElBQUksRUFDeEIsWUFBWSxHQUNiLEdBQUcsS0FBSyxDQUFDO1FBRVYsa0VBQWtFO1FBQ2xFLE1BQU0sc0JBQXNCLEdBQUcscUJBQXFCLENBQUMsMEJBQTBCLENBQUM7WUFDOUUsWUFBWSxFQUFFLFlBQVksSUFBSSxpQkFBRyxDQUFDLE1BQU07WUFDeEMsdUJBQXVCO1lBQ3ZCLGlCQUFpQjtZQUNqQixXQUFXLEVBQUUsV0FBVyxLQUFLLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxXQUFXO1NBQ3ZELENBQUMsQ0FBQztRQUVILEVBQUU7UUFDRixxQ0FBcUM7UUFDckMsRUFBRTtRQUNGLE1BQU0scUJBQXFCLEdBQW1EO1lBQzVFLFlBQVksRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDLEdBQUcsYUFBYSxrQkFBa0IsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDN0YsVUFBVSxFQUFFLElBQUk7WUFDaEIsWUFBWSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUztZQUMxQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDNUIsYUFBYSxFQUFFO2dCQUNiLG1FQUFtRTtnQkFDbkUsdUVBQXVFO2dCQUN2RSw0Q0FBNEM7Z0JBQzVDLGtFQUFrRTtnQkFDbEUsdURBQXVEO2dCQUN2RCwyREFBMkQ7Z0JBQzNELGtEQUFrRDtnQkFDbEQsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixPQUFPLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQztvQkFDL0IsU0FBUyxFQUFFLENBQUMseUJBQXlCLGlCQUFHLENBQUMsVUFBVSxVQUFVLENBQUM7aUJBUy9ELENBQUM7YUFDSDtZQUNELEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztTQUM1QyxDQUFDO1FBQ0YsSUFDRSxPQUFPLENBQUMsR0FBRyxDQUFDLFFBQVEsS0FBSyxNQUFNO1lBQy9CLGVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLDBCQUEwQixFQUFFLE1BQU0sRUFBRSxVQUFVLENBQUMsQ0FBQyxFQUM1RjtZQUNBLGtEQUFrRDtZQUNsRCxrQkFBYSxDQUNYLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsMEJBQTBCLEVBQUUsTUFBTSxFQUFFLFlBQVksQ0FBQyxFQUNsRixzQkFBc0IsQ0FDdkIsQ0FBQztZQUNGLGdCQUFnQjtZQUNoQixpR0FBaUc7WUFDakcsd0ZBQXdGO1lBQ3hGLEtBQUs7WUFDTCxtQ0FBbUM7WUFDbkMsMkVBQTJFO1lBQzNFLElBQUksQ0FBQyxxQkFBcUIsR0FBRyxJQUFJLEVBQUUsQ0FBQyxZQUFZLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtnQkFDekYsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUN6QixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLDBCQUEwQixFQUFFLE1BQU0sQ0FBQyxDQUNyRTtnQkFDRCxPQUFPLEVBQUUsZUFBZTtnQkFDeEIsR0FBRyxxQkFBcUI7YUFDekIsQ0FBQyxDQUFDO1NBQ0o7YUFBTSxJQUFJLGVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwwQkFBMEIsRUFBRSxVQUFVLENBQUMsQ0FBQyxFQUFFO1lBQ25GLGtEQUFrRDtZQUNsRCxrQkFBYSxDQUNYLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLDBCQUEwQixFQUFFLFlBQVksQ0FBQyxFQUM5RCxzQkFBc0IsQ0FDdkIsQ0FBQztZQUVGLHlEQUF5RDtZQUN6RCxJQUFJLENBQUMscUJBQXFCLEdBQUcsSUFBSSxFQUFFLENBQUMsWUFBWSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUscUJBQXFCLEVBQUU7Z0JBQ3pGLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO2dCQUM3RSxPQUFPLEVBQUUsZUFBZTtnQkFDeEIsR0FBRyxxQkFBcUI7YUFDekIsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLGtEQUFrRDtZQUNsRCxrQkFBYSxDQUNYLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsMEJBQTBCLEVBQUUsWUFBWSxDQUFDLEVBQzFFLHNCQUFzQixDQUN2QixDQUFDO1lBRUYsbUVBQW1FO1lBQ25FLHVFQUF1RTtZQUN2RSxtRUFBbUU7WUFDbkUsOERBQThEO1lBQzlELHVFQUF1RTtZQUN2RSxJQUFJLENBQUMscUJBQXFCLEdBQUcsSUFBSSxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtnQkFDeEYsS0FBSyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsMEJBQTBCLEVBQUUsS0FBSyxFQUFFLFVBQVUsQ0FBQztnQkFDdEYsT0FBTyxFQUFFLFNBQVM7Z0JBQ2xCLFFBQVEsRUFBRTtvQkFDUixNQUFNLEVBQUUsSUFBSTtvQkFDWixTQUFTLEVBQUUsSUFBSTtvQkFDZixZQUFZLEVBQUU7d0JBQ1osYUFBYSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7d0JBQ3ZCLGNBQWMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO3dCQUN4QixhQUFhLEVBQUUsQ0FBQyxTQUFpQixFQUFFLFNBQWlCLEVBQUUsRUFBRTs0QkFDdEQsT0FBTztnQ0FDTCxHQUFHLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxPQUFPLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxJQUFJLENBQ3ZELFNBQVMsRUFDVCxJQUFJLEVBQ0osSUFBSSxFQUNKLElBQUksRUFDSixTQUFTLEVBQ1QsMEJBQTBCLEVBQzFCLFlBQVksQ0FDYixJQUFJLFNBQVMsRUFBRTs2QkFDakIsQ0FBQzt3QkFDSixDQUFDO3FCQUNGO2lCQUNGO2dCQUNELEdBQUcscUJBQXFCO2FBQ3pCLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxDQUFDLG9CQUFvQixHQUFHO1lBQzFCO2dCQUNFLFNBQVMsRUFBRSxFQUFFLENBQUMsbUJBQW1CLENBQUMsY0FBYztnQkFDaEQsZUFBZSxFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxjQUFjO2dCQUMxRCxXQUFXLEVBQUUsSUFBSTthQUNsQjtTQUNGLENBQUM7SUFDSixDQUFDO0lBaktEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsMEJBQTBCLENBQUMsS0FBd0M7UUFDL0UsT0FBTyxpQkFBaUIsS0FBSyxDQUFDLFlBQVk7RUFDNUMsS0FBSyxDQUFDLFdBQVcsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLEtBQUssQ0FBQyxXQUFXLEVBQUU7MkJBQzFDLEtBQUssQ0FBQyx1QkFBdUI7cUJBQ25DLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO0lBQzdDLENBQUM7SUFHRCxJQUFXLG9CQUFvQjtRQUM3QixPQUFPLElBQUksQ0FBQyxxQkFBcUIsQ0FBQztJQUNwQyxDQUFDO0lBR0QsSUFBVyxtQkFBbUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsb0JBQW9CLENBQUM7SUFDbkMsQ0FBQzs7QUFyQkgsc0RBbUtDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhpc3RzU3luYywgd3JpdGVGaWxlU3luYyB9IGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIG9zIGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBBd3MsIER1cmF0aW9uLCBSZW1vdmFsUG9saWN5IH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgY2YgZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3VkZnJvbnQnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgbGFtYmRhTm9kZWpzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEtbm9kZWpzJztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIE1pY3JvQXBwcyBFZGdlIHRvIE9yaWdpbiBGdW5jdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIElNaWNyb0FwcHNFZGdlVG9PcmlnaW4ge1xuICAvKipcbiAgICogVGhlIGVkZ2UgdG8gb3JpZ2luIGZ1bmN0aW9uIGZvciBBUEkgR2F0ZXdheSBSZXF1ZXN0IE9yaWdpbiBFZGdlIExhbWJkYVxuICAgKlxuICAgKiBUaGUgZ2VuZXJhdGVkIGBjb25maWcueW1sYCBpcyBpbmNsdWRlZCBpbiB0aGUgTGFtYmRhJ3MgY29kZS5cbiAgICovXG4gIHJlYWRvbmx5IGVkZ2VUb09yaWdpbkZ1bmN0aW9uOiBsYW1iZGEuRnVuY3Rpb24gfCBjZi5leHBlcmltZW50YWwuRWRnZUZ1bmN0aW9uO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIG9mIHRoZSBlZGdlIHRvIG9yaWdpbiBsYW1iZGEgZnVuY3Rpb25zXG4gICAqL1xuICByZWFkb25seSBlZGdlVG9PcmlnaW5MYW1iZGFzOiBjZi5FZGdlTGFtYmRhW107XG59XG5cbi8qKlxuICogUHJvcGVydGllcyB0byBpbml0aWFsaXplIGFuIGluc3RhbmNlIG9mIGBNaWNyb0FwcHNFZGdlVG9PcmlnaW5gLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE1pY3JvQXBwc0VkZ2VUb09yaWdpblByb3BzIHtcbiAgLyoqXG4gICAqIFJlbW92YWxQb2xpY3kgb3ZlcnJpZGUgZm9yIGNoaWxkIHJlc291cmNlc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIHBlciByZXNvdXJjZSBkZWZhdWx0XG4gICAqL1xuICByZWFkb25seSByZW1vdmFsUG9saWN5PzogUmVtb3ZhbFBvbGljeTtcblxuICAvKipcbiAgICogT3B0aW9uYWwgYXNzZXQgbmFtZSByb290XG4gICAqXG4gICAqIEBleGFtcGxlIG1pY3JvYXBwc1xuICAgKiBAZGVmYXVsdCAtIHJlc291cmNlIG5hbWVzIGF1dG8gYXNzaWduZWRcbiAgICovXG4gIHJlYWRvbmx5IGFzc2V0TmFtZVJvb3Q/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIGFzc2V0IG5hbWUgc3VmZml4XG4gICAqXG4gICAqIEBleGFtcGxlIC1kZXYtcHItMTJcbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgYXNzZXROYW1lU3VmZml4Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBZGRzIGFuIFgtRm9yd2FyZGVkLUhvc3QtSGVhZGVyIHdoZW4gY2FsbGluZyBBUEkgR2F0ZXdheVxuICAgKlxuICAgKiBDYW4gb25seSBiZSB0cnVzdGVkIGlmIGBzaWduaW5nTW9kZWAgaXMgZW5hYmxlZCwgd2hpY2ggcmVzdHJpY3RzXG4gICAqIGFjY2VzcyB0byBBUEkgR2F0ZXdheSB0byBvbmx5IElBTSBzaWduZWQgcmVxdWVzdHMuXG4gICAqXG4gICAqIE5vdGU6IGlmIHRydWUsIGNyZWF0ZXMgT3JpZ2luUmVxdWVzdCBMYW1iZGEgQCBFZGdlIGZ1bmN0aW9uIGZvciBBUEkgR2F0ZXdheSBPcmlnaW5cbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgYWRkWEZvcndhcmRlZEhvc3RIZWFkZXI/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBSZXBsYWNlcyBIb3N0IGhlYWRlciAod2hpY2ggd2lsbCBiZSB0aGUgRWRnZSBkb21haW4gbmFtZSkgd2l0aCB0aGUgT3JpZ2luIGRvbWFpbiBuYW1lXG4gICAqIHdoZW4gZW5hYmxlZC4gIFRoaXMgaXMgbmVjZXNzYXJ5IHdoZW4gQVBJIEdhdGV3YXkgaGFzIG5vdCBiZWVuIGNvbmZpZ3VyZWRcbiAgICogd2l0aCBhIGN1c3RvbSBkb21haW4gbmFtZSB0aGF0IG1hdGNoZXMgdGhlIGV4YWN0IGRvbWFpbiBuYW1lIHVzZWQgYnkgdGhlIENsb3VkRnJvbnRcbiAgICogRGlzdHJpYnV0aW9uIEFORCB3aGVuIHRoZSBPcmlnaW5SZXF1ZXN0UG9saWN5LkhlYWRlcnNCZWhhdmlvciBpcyBzZXRcbiAgICogdG8gcGFzcyBhbGwgaGVhZGVycyB0byB0aGUgb3JpZ2luLlxuICAgKlxuICAgKiBOb3RlOiBpZiB0cnVlLCBjcmVhdGVzIE9yaWdpblJlcXVlc3QgTGFtYmRhIEAgRWRnZSBmdW5jdGlvbiBmb3IgQVBJIEdhdGV3YXkgT3JpZ2luXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHJlcGxhY2VIb3N0SGVhZGVyPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogUmVxdWlyZXMgSUFNIGF1dGggb24gdGhlIEFQSSBHYXRld2F5IG9yaWdpbiBpZiBub3Qgc2V0IHRvICdub25lJy5cbiAgICpcbiAgICogJ3NpZ24nIC0gVXNlcyByZXF1ZXN0IGhlYWRlcnMgZm9yIGF1dGguXG4gICAqICdwcmVzaWduJyAtIFVzZXMgcXVlcnkgc3RyaW5nIGZvciBhdXRoLlxuICAgKlxuICAgKiBJZiBlbmFibGVkLFxuICAgKlxuICAgKiBOb3RlOiBpZiAnc2lnbicgb3IgJ3ByZXNpZ24nLCBjcmVhdGVzIE9yaWdpblJlcXVlc3QgTGFtYmRhIEAgRWRnZSBmdW5jdGlvbiBmb3IgQVBJIEdhdGV3YXkgT3JpZ2luXG4gICAqIEBkZWZhdWx0ICdzaWduJ1xuICAgKi9cbiAgcmVhZG9ubHkgc2lnbmluZ01vZGU/OiAnc2lnbicgfCAncHJlc2lnbicgfCAnbm9uZSc7XG5cbiAgLyoqXG4gICAqIE9yaWdpbiByZWdpb24gdGhhdCBBUEkgR2F0ZXdheSB3aWxsIGJlIGRlcGxveWVkIHRvLCB1c2VkXG4gICAqIGZvciB0aGUgY29uZmlnLnltbCBvbiB0aGUgRWRnZSBmdW5jdGlvbiB0byBzaWduIHJlcXVlc3RzIGZvclxuICAgKiB0aGUgY29ycmVjdCByZWdpb25cbiAgICpcbiAgICogQGRlZmF1bHQgdW5kZWZpbmVkXG4gICAqL1xuICByZWFkb25seSBvcmlnaW5SZWdpb24/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgR2VuZXJhdGVFZGdlVG9PcmlnaW5Db25maWdPcHRpb25zIHtcbiAgcmVhZG9ubHkgb3JpZ2luUmVnaW9uOiBzdHJpbmc7XG4gIHJlYWRvbmx5IHNpZ25pbmdNb2RlOiAnc2lnbicgfCAncHJlc2lnbicgfCAnJztcbiAgcmVhZG9ubHkgYWRkWEZvcndhcmRlZEhvc3RIZWFkZXI6IGJvb2xlYW47XG4gIHJlYWRvbmx5IHJlcGxhY2VIb3N0SGVhZGVyOiBib29sZWFuO1xufVxuXG4vKipcbiAqIENyZWF0ZSBhIG5ldyBNaWNyb0FwcHMgRWRnZSB0byBPcmlnaW4gRnVuY3Rpb24gdy8gYGNvbmZpZy55bWxgXG4gKi9cbmV4cG9ydCBjbGFzcyBNaWNyb0FwcHNFZGdlVG9PcmlnaW4gZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJTWljcm9BcHBzRWRnZVRvT3JpZ2luIHtcbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSB5YW1sIGNvbmZpZyBmb3IgdGhlIGVkZ2UgbGFtYmRhXG4gICAqIEBwYXJhbSBwcm9wc1xuICAgKiBAcmV0dXJuc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBnZW5lcmF0ZUVkZ2VUb09yaWdpbkNvbmZpZyhwcm9wczogR2VuZXJhdGVFZGdlVG9PcmlnaW5Db25maWdPcHRpb25zKSB7XG4gICAgcmV0dXJuIGBvcmlnaW5SZWdpb246ICR7cHJvcHMub3JpZ2luUmVnaW9ufVxuJHtwcm9wcy5zaWduaW5nTW9kZSA9PT0gJycgPyAnJyA6IGBzaWduaW5nTW9kZTogJHtwcm9wcy5zaWduaW5nTW9kZX1gfVxuYWRkWEZvcndhcmRlZEhvc3RIZWFkZXI6ICR7cHJvcHMuYWRkWEZvcndhcmRlZEhvc3RIZWFkZXJ9XG5yZXBsYWNlSG9zdEhlYWRlcjogJHtwcm9wcy5yZXBsYWNlSG9zdEhlYWRlcn1gO1xuICB9XG5cbiAgcHJpdmF0ZSBfZWRnZVRvT3JpZ2luRnVuY3Rpb246IGxhbWJkYS5GdW5jdGlvbiB8IGNmLmV4cGVyaW1lbnRhbC5FZGdlRnVuY3Rpb247XG4gIHB1YmxpYyBnZXQgZWRnZVRvT3JpZ2luRnVuY3Rpb24oKTogbGFtYmRhLkZ1bmN0aW9uIHwgY2YuZXhwZXJpbWVudGFsLkVkZ2VGdW5jdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMuX2VkZ2VUb09yaWdpbkZ1bmN0aW9uO1xuICB9XG5cbiAgcHJpdmF0ZSBfZWRnZVRvT3JpZ2luTGFtYmRhczogY2YuRWRnZUxhbWJkYVtdO1xuICBwdWJsaWMgZ2V0IGVkZ2VUb09yaWdpbkxhbWJkYXMoKTogY2YuRWRnZUxhbWJkYVtdIHtcbiAgICByZXR1cm4gdGhpcy5fZWRnZVRvT3JpZ2luTGFtYmRhcztcbiAgfVxuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBNaWNyb0FwcHNFZGdlVG9PcmlnaW5Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdwcm9wcyBtdXN0IGJlIHNldCcpO1xuICAgIH1cblxuICAgIGNvbnN0IHtcbiAgICAgIHJlbW92YWxQb2xpY3ksXG4gICAgICBhc3NldE5hbWVSb290LFxuICAgICAgYXNzZXROYW1lU3VmZml4LFxuICAgICAgc2lnbmluZ01vZGUgPSAnc2lnbicsXG4gICAgICBhZGRYRm9yd2FyZGVkSG9zdEhlYWRlciA9IHRydWUsXG4gICAgICByZXBsYWNlSG9zdEhlYWRlciA9IHRydWUsXG4gICAgICBvcmlnaW5SZWdpb24sXG4gICAgfSA9IHByb3BzO1xuXG4gICAgLy8gQ3JlYXRlIHRoZSBlZGdlIGZ1bmN0aW9uIGNvbmZpZyBmaWxlIGZyb20gdGhlIGNvbnN0cnVjdCBvcHRpb25zXG4gICAgY29uc3QgZWRnZVRvT3JpZ2luQ29uZmlnWWFtbCA9IE1pY3JvQXBwc0VkZ2VUb09yaWdpbi5nZW5lcmF0ZUVkZ2VUb09yaWdpbkNvbmZpZyh7XG4gICAgICBvcmlnaW5SZWdpb246IG9yaWdpblJlZ2lvbiB8fCBBd3MuUkVHSU9OLFxuICAgICAgYWRkWEZvcndhcmRlZEhvc3RIZWFkZXIsXG4gICAgICByZXBsYWNlSG9zdEhlYWRlcixcbiAgICAgIHNpZ25pbmdNb2RlOiBzaWduaW5nTW9kZSA9PT0gJ25vbmUnID8gJycgOiBzaWduaW5nTW9kZSxcbiAgICB9KTtcblxuICAgIC8vXG4gICAgLy8gQ3JlYXRlIHRoZSBFZGdlIHRvIE9yaWdpbiBGdW5jdGlvblxuICAgIC8vXG4gICAgY29uc3QgZWRnZVRvT3JpZ2luRnVuY1Byb3BzOiBPbWl0PGxhbWJkYS5GdW5jdGlvblByb3BzLCAnaGFuZGxlcicgfCAnY29kZSc+ID0ge1xuICAgICAgZnVuY3Rpb25OYW1lOiBhc3NldE5hbWVSb290ID8gYCR7YXNzZXROYW1lUm9vdH0tZWRnZS10by1vcmlnaW4ke2Fzc2V0TmFtZVN1ZmZpeH1gIDogdW5kZWZpbmVkLFxuICAgICAgbWVtb3J5U2l6ZTogMTc2OSxcbiAgICAgIGxvZ1JldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xNF9YLFxuICAgICAgdGltZW91dDogRHVyYXRpb24uc2Vjb25kcyg1KSxcbiAgICAgIGluaXRpYWxQb2xpY3k6IFtcbiAgICAgICAgLy8gVGhpcyBjYW4ndCBoYXZlIGEgcmVmZXJlbmNlIHRvIHRoZSBodHRwQXBpIGJlY2F1c2UgaXQgd291bGQgbWVhblxuICAgICAgICAvLyB0aGUgcGFyZW50IHN0YWNrICh0aGlzIHN0YWNrKSBoYXMgdG8gYmUgY3JlYXRlZCBiZWZvcmUgdGhlIHVzLWVhc3QtMVxuICAgICAgICAvLyBjaGlsZCBzdGFjayBmb3IgdGhlIEVkZ2UgTGFtYmRhIEZ1bmN0aW9uLlxuICAgICAgICAvLyBUaGF0J3Mgd2h5IHdlIHVzZSBhIHRhZy1iYXNlZCBwb2xpY3kgdG8gYWxsb3cgdGhlIEVkZ2UgRnVuY3Rpb25cbiAgICAgICAgLy8gdG8gaW52b2tlIGFueSBBUEkgR2F0ZXdheSBBUEkgdGhhdCB3ZSBhcHBseSBhIHRhZyB0b1xuICAgICAgICAvLyBXZSBhbGxvdyB0aGUgZWRnZSBmdW5jdGlvbiB0byBzaWduIGZvciBhbGwgcmVnaW9ucyBzaW5jZVxuICAgICAgICAvLyB3ZSBtYXkgdXNlIGN1c3RvbSBjbG9zZXN0IHJlZ2lvbiBpbiB0aGUgZnV0dXJlLlxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogWydleGVjdXRlLWFwaTpJbnZva2UnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtgYXJuOmF3czpleGVjdXRlLWFwaToqOiR7QXdzLkFDQ09VTlRfSUR9OiovKi8qLypgXSxcbiAgICAgICAgICAvLyBVbmZvcnR1bmF0ZWx5LCBBUEkgR2F0ZXdheSBhY2Nlc3MgY2Fubm90IGJlIHJlc3RyaWN0ZWQgdXNpbmdcbiAgICAgICAgICAvLyB0YWdzIG9uIHRoZSB0YXJnZXQgcmVzb3VyY2VcbiAgICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvYWNjZXNzX3RhZ3MuaHRtbFxuICAgICAgICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfYXdzLXNlcnZpY2VzLXRoYXQtd29yay13aXRoLWlhbS5odG1sI25ldHdvcmtpbmdfc3Zjc1xuICAgICAgICAgIC8vIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgICAvLyAgIC8vIFRPRE86IFNldCB0aGlzIHRvIGEgc3RyaW5nIHVuaXF1ZSB0byBlYWNoIHN0YWNrXG4gICAgICAgICAgLy8gICBTdHJpbmdFcXVhbHM6IHsgJ2F3czpSZXNvdXJjZVRhZy9taWNyb2FwcC1tYW5hZ2VkJzogJ3RydWUnIH0sXG4gICAgICAgICAgLy8gfSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgICAgLi4uKHJlbW92YWxQb2xpY3kgPyB7IHJlbW92YWxQb2xpY3kgfSA6IHt9KSxcbiAgICB9O1xuICAgIGlmIChcbiAgICAgIHByb2Nlc3MuZW52Lk5PREVfRU5WID09PSAndGVzdCcgJiZcbiAgICAgIGV4aXN0c1N5bmMocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJy4uJywgJ21pY3JvYXBwcy1lZGdlLXRvLW9yaWdpbicsICdkaXN0JywgJ2luZGV4LmpzJykpXG4gICAgKSB7XG4gICAgICAvLyBFbWl0IHRoZSBjb25maWcgZmlsZSBmcm9tIHRoZSBjb25zdHJ1Y3Qgb3B0aW9uc1xuICAgICAgd3JpdGVGaWxlU3luYyhcbiAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJy4uJywgJ21pY3JvYXBwcy1lZGdlLXRvLW9yaWdpbicsICdkaXN0JywgJ2NvbmZpZy55bWwnKSxcbiAgICAgICAgZWRnZVRvT3JpZ2luQ29uZmlnWWFtbCxcbiAgICAgICk7XG4gICAgICAvLyBjb3B5RmlsZVN5bmMoXG4gICAgICAvLyAgIHBhdGguam9pbihfX2Rpcm5hbWUsICcuLicsICcuLicsICcuLicsICdjb25maWdzJywgJ21pY3JvYXBwcy1lZGdlLXRvLW9yaWdpbicsICdjb25maWcueW1sJyksXG4gICAgICAvLyAgIHBhdGguam9pbihfX2Rpcm5hbWUsICcuLicsICcuLicsICdtaWNyb2FwcHMtZWRnZS10by1vcmlnaW4nLCAnZGlzdCcsICdjb25maWcueW1sJyksXG4gICAgICAvLyApO1xuICAgICAgLy8gVGhpcyBpcyBmb3IgdGVzdHMgcnVuIHVuZGVyIGplc3RcbiAgICAgIC8vIFRoaXMgaXMgYWxzbyBmb3IgYW55dGltZSB3aGVuIHRoZSBlZGdlIGZ1bmN0aW9uIGhhcyBhbHJlYWR5IGJlZW4gYnVuZGxlZFxuICAgICAgdGhpcy5fZWRnZVRvT3JpZ2luRnVuY3Rpb24gPSBuZXcgY2YuZXhwZXJpbWVudGFsLkVkZ2VGdW5jdGlvbih0aGlzLCAnZWRnZS10by1hcGlnd3ktZnVuYycsIHtcbiAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KFxuICAgICAgICAgIHBhdGguam9pbihfX2Rpcm5hbWUsICcuLicsICcuLicsICdtaWNyb2FwcHMtZWRnZS10by1vcmlnaW4nLCAnZGlzdCcpLFxuICAgICAgICApLFxuICAgICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgIC4uLmVkZ2VUb09yaWdpbkZ1bmNQcm9wcyxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSBpZiAoZXhpc3RzU3luYyhwYXRoLmpvaW4oX19kaXJuYW1lLCAnbWljcm9hcHBzLWVkZ2UtdG8tb3JpZ2luJywgJ2luZGV4LmpzJykpKSB7XG4gICAgICAvLyBFbWl0IHRoZSBjb25maWcgZmlsZSBmcm9tIHRoZSBjb25zdHJ1Y3Qgb3B0aW9uc1xuICAgICAgd3JpdGVGaWxlU3luYyhcbiAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJ21pY3JvYXBwcy1lZGdlLXRvLW9yaWdpbicsICdjb25maWcueW1sJyksXG4gICAgICAgIGVkZ2VUb09yaWdpbkNvbmZpZ1lhbWwsXG4gICAgICApO1xuXG4gICAgICAvLyBUaGlzIGlzIGZvciBidWlsdCBhcHBzIHBhY2thZ2VkIHdpdGggdGhlIENESyBjb25zdHJ1Y3RcbiAgICAgIHRoaXMuX2VkZ2VUb09yaWdpbkZ1bmN0aW9uID0gbmV3IGNmLmV4cGVyaW1lbnRhbC5FZGdlRnVuY3Rpb24odGhpcywgJ2VkZ2UtdG8tYXBpZ3d5LWZ1bmMnLCB7XG4gICAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnbWljcm9hcHBzLWVkZ2UtdG8tb3JpZ2luJykpLFxuICAgICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgIC4uLmVkZ2VUb09yaWdpbkZ1bmNQcm9wcyxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBFbWl0IHRoZSBjb25maWcgZmlsZSBmcm9tIHRoZSBjb25zdHJ1Y3Qgb3B0aW9uc1xuICAgICAgd3JpdGVGaWxlU3luYyhcbiAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJy4uJywgJ21pY3JvYXBwcy1lZGdlLXRvLW9yaWdpbicsICdjb25maWcueW1sJyksXG4gICAgICAgIGVkZ2VUb09yaWdpbkNvbmZpZ1lhbWwsXG4gICAgICApO1xuXG4gICAgICAvLyBUaGlzIGJ1aWxkcyB0aGUgZnVuY3Rpb24gZm9yIGRpc3RyaWJ1dGlvbiB3aXRoIHRoZSBDREsgQ29uc3RydWN0XG4gICAgICAvLyBhbmQgd2lsbCBiZSB1c2VkIGR1cmluZyBsb2NhbCBidWlsZHMgYW5kIFBSIGJ1aWxkcyBvZiBtaWNyb2FwcHMtY29yZVxuICAgICAgLy8gaWYgdGhlIG1pY3JvYXBwcy1lZGdlLXRvLW9yaWdpbiBmdW5jdGlvbiBpcyBub3QgYWxyZWFkeSBidW5kbGVkLlxuICAgICAgLy8gVGhpcyB3aWxsIGZhaWwgdG8gZGVwbG95IGluIGFueSByZWdpb24gb3RoZXIgdGhhbiB1cy1lYXN0LTFcbiAgICAgIC8vIFdlIGNhbm5vdCB1c2UgTm9kZWpzRnVuY3Rpb24gYmVjYXVzZSBpdCB3aWxsIG5vdCBjcmVhdGUgaW4gdXMtZWFzdC0xXG4gICAgICB0aGlzLl9lZGdlVG9PcmlnaW5GdW5jdGlvbiA9IG5ldyBsYW1iZGFOb2RlanMuTm9kZWpzRnVuY3Rpb24odGhpcywgJ2VkZ2UtdG8tYXBpZ3d5LWZ1bmMnLCB7XG4gICAgICAgIGVudHJ5OiBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4nLCAnLi4nLCAnbWljcm9hcHBzLWVkZ2UtdG8tb3JpZ2luJywgJ3NyYycsICdpbmRleC50cycpLFxuICAgICAgICBoYW5kbGVyOiAnaGFuZGxlcicsXG4gICAgICAgIGJ1bmRsaW5nOiB7XG4gICAgICAgICAgbWluaWZ5OiB0cnVlLFxuICAgICAgICAgIHNvdXJjZU1hcDogdHJ1ZSxcbiAgICAgICAgICBjb21tYW5kSG9va3M6IHtcbiAgICAgICAgICAgIGJlZm9yZUluc3RhbGw6ICgpID0+IFtdLFxuICAgICAgICAgICAgYmVmb3JlQnVuZGxpbmc6ICgpID0+IFtdLFxuICAgICAgICAgICAgYWZ0ZXJCdW5kbGluZzogKF9pbnB1dERpcjogc3RyaW5nLCBvdXRwdXREaXI6IHN0cmluZykgPT4ge1xuICAgICAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgICAgIGAke29zLnBsYXRmb3JtKCkgPT09ICd3aW4zMicgPyAnY29weScgOiAnY3AnfSAke3BhdGguam9pbihcbiAgICAgICAgICAgICAgICAgIF9fZGlybmFtZSxcbiAgICAgICAgICAgICAgICAgICcuLicsXG4gICAgICAgICAgICAgICAgICAnLi4nLFxuICAgICAgICAgICAgICAgICAgJy4uJyxcbiAgICAgICAgICAgICAgICAgICdjb25maWdzJyxcbiAgICAgICAgICAgICAgICAgICdtaWNyb2FwcHMtZWRnZS10by1vcmlnaW4nLFxuICAgICAgICAgICAgICAgICAgJ2NvbmZpZy55bWwnLFxuICAgICAgICAgICAgICAgICl9ICR7b3V0cHV0RGlyfWAsXG4gICAgICAgICAgICAgIF07XG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIC4uLmVkZ2VUb09yaWdpbkZ1bmNQcm9wcyxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMuX2VkZ2VUb09yaWdpbkxhbWJkYXMgPSBbXG4gICAgICB7XG4gICAgICAgIGV2ZW50VHlwZTogY2YuTGFtYmRhRWRnZUV2ZW50VHlwZS5PUklHSU5fUkVRVUVTVCxcbiAgICAgICAgZnVuY3Rpb25WZXJzaW9uOiB0aGlzLl9lZGdlVG9PcmlnaW5GdW5jdGlvbi5jdXJyZW50VmVyc2lvbixcbiAgICAgICAgaW5jbHVkZUJvZHk6IHRydWUsXG4gICAgICB9LFxuICAgIF07XG4gIH1cbn1cbiJdfQ==