# MatterApi

A python client library for accessing the Mattermost API with sync/async support.


## Features
+ Endpoints generated from the [API documentation](https://api.mattermost.com/)
  - However: if the description is wrong, this library will be wrong
+ Based on [httpx](https://www.python-httpx.org/) and supports sync and async operation
+ Response parsing into models with [pydantic](https://pydantic-docs.helpmanual.io/)
+ Typing support
+ Websocket handling to connect to the Mattermost event stream.

## Getting Started

The driver has synch and async support. Depending on your workflow you will want to choose between
the `SyncDriver` and `AsyncDriver`.


### SyncDriver

First, let's look at an example on how to use the SyncDriver:

```python

from matterapi import SyncDriver

# set the options for the driver
options = { 'url' : 'http://localhost:8095',
    'auth' : { 
      'token' : '<yourtokenhere>' 
      }
}

# Create a sync driver
sd = SyncDriver(options=options)

# Call login to:
# 1. Get a session token if you use user:password based auth
# 2. Populate the driver with the corresponding user object
sd.login()

# The drivers `user` object will hold information on the current user/bot
print(sd.user.id)

# Use the driver

## Get your own user object
sd.users.get_user("me")

## Get channels
channels = sd.channels.get_all_channels(per_page=20)
print(channels[0])

## Get posts for channel
post_list = sd.posts.get_posts_for_channel("<channel_id>")
for post in post_list.posts:
  print(post, post_list.posts[post].id)

```

### AsyncDriver

And this is how you can use the AsyncDriver


```python

import asyncio
from matterapi import AsyncDriver

# set the options for the driver
options = { 'url' : 'https://localhost:8095',
    # User username and password authentication
    'auth' : { 
      'login_id' : 'hansolo', 
      'password' : 'lea1234' 
      },
    # Disable TLS verification for the client
    'client_options' : {
      'verify' : False
    }
}

# Create a async driver
ad = AsyncDriver(options=options)

async def do_something():
  # Call login to:
  # 1. Get a session token if you user user:password based auth
  # 2. Populate the driver with the corresponding user object
  await ad.login()

  # Use the driver
  users = await ad.users.get_users()
  print(users)

  # To upload files, you could for example use the following request
  data = {
          "files": {
              "test1.png": open("testfile1.png", "rb"),
              "test2.png": open("testfile2.png", "rb"),
          }
      }

  file_infos = await ad.files.upload_file(
      channel_id="7bzsijaqopfczygxm1qc3r63do",
      multipart_data=data)

  print(file_info)

asyncio.run(do_something())
```

## Websocket connection

You can also listen for event data from Mattermost

```python

def handler(message):
  print(message)

# either use the sync version
sd.start_ws_sync(handler)

# or the async version with 
async def wrapper():
  await ad.start_ws(handler)

```


Contributing
------------

The actual endpoints in Matterapi are autogenerated with a fork of [openapi-python-client](https://github.com/gmerz/openapi-python-client). If anything there needs changing, please refer to the generator project.

Some endpoints might return wrong results or miss parameters. Currently, I can image the following two reasons:

1. The generator is broken and/or does not support the required feature set.
    - Create an issue in the [matterapi-generator](https://github.com/gmerz/matterapi-generator) repository
    - If you know how to fix it, consider to do a pull request

2. The [mattermost api documentation](https://github.com/mattermost/mattermost-api-reference), which this client is generated from, is not correct
    - Sometimes entries in the documentation might not be completely correct
    - Consider fixing the API documentation/do a pull request/post an issue there
    - This will help everybody who is using the documentation

3. The API documentation changed or included new endpoints, but the library was not updated.
    - Currently creation of the api client is not automated, so you have two options:
        1. Help with the automation so that this proejct is automatically updated with changes
        2. Create a new issue here that the client is out of sync
        3. Clone the generator repo, update the mattermost api with the generator and do a pull request here. (This is currently the only save way to get changes into the matterapi folder)


Credits
-------

Credits where credits are due:

+ This library is autogenerated from the Mattermost API documentation using a fork of [openapi-python-client](https://github.com/triaxtec/openapi-python-client). 
+ It's heavily inspired by (but not a 1:1 drop-in replacement for) [mattermostdriver](https://github.com/Vaelor/python-mattermost-driver), which I used for several years already. This is still your go-to if you look for something stable that has been in use for years by lot's of people.

