# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyopenuv']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.8.0', 'asynctest>=0.13.0,<0.14.0']

setup_kwargs = {
    'name': 'pyopenuv',
    'version': '2023.1.0',
    'description': 'A simple Python API data from openuv.io',
    'long_description': '# ☀️ pyopenuv: A simple Python API for data from openuv.io\n\n[![CI][ci-badge]][ci]\n[![PyPI][pypi-badge]][pypi]\n[![Version][version-badge]][version]\n[![License][license-badge]][license]\n[![Code Coverage][codecov-badge]][codecov]\n[![Maintainability][maintainability-badge]][maintainability]\n\n<a href="https://www.buymeacoffee.com/bachya1208P" target="_blank"><img src="https://cdn.buymeacoffee.com/buttons/default-orange.png" alt="Buy Me A Coffee" height="41" width="174"></a>\n\n`pyopenuv` is a simple Python library for retrieving UV-related information from\n[openuv.io][openuv].\n\n- [Installation](#installation)\n- [Python Versions](#python-versions)\n- [API Key](#api-key)\n- [Usage](#usage)\n- [Contributing](#contributing)\n\n# Installation\n\n```bash\npip install pyopenuv\n```\n\n# Python Versions\n\n`pyopenuv` is currently supported on:\n\n- Python 3.9\n- Python 3.10\n- Python 3.11\n\n# API Key\n\nYou can get an API key from [the OpenUV console][openuv-console].\n\n# Usage\n\n```python\nimport asyncio\n\nfrom pyopenuv import Client\nfrom pyopenuv.errors import OpenUvError\n\n\nasync def main():\n    client = Client(\n        "<OPENUV_API_KEY>", "<LATITUDE>", "<LONGITUDE>", altitude="<ALTITUDE>"\n    )\n\n    try:\n        # Get the current status of the OpenUV API:\n        print(await client.api_status())\n        # >>> True\n\n        # Get current UV info:\n        print(await client.uv_index())\n        # >>> { "result": { ... } }\n\n        # Get forecasted UV info:\n        print(await client.uv_forecast())\n        # >>> { "result": { ... } }\n\n        # Get UV protection window:\n        print(await client.uv_protection_window())\n        # >>> { "result": { ... } }\n\n        # Get API usage info/statistics:\n        print(await client.api_statistics())\n        # >>> { "result": { ... } }\n    except OpenUvError as err:\n        print(f"There was an error: {err}")\n\n\nasyncio.run(main())\n```\n\n## Checking API Status Before Requests\n\nIf you would prefer to not call `api_status` manually, you can configure the `Client` object\nto automatically check the status of the OpenUV API before executing any of the API\nmethods—simply pass the `check_status_before_request` parameter:\n\n```python\nimport asyncio\n\nfrom pyopenuv import Client\nfrom pyopenuv.errors import ApiUnavailableError, OpenUvError\n\n\nasync def main():\n    client = Client(\n        "<OPENUV_API_KEY>",\n        "<LATITUDE>",\n        "<LONGITUDE>",\n        altitude="<ALTITUDE>",\n        check_status_before_request=True,\n    )\n\n    try:\n        print(await client.uv_index())\n    except ApiUnavailableError:\n        print("The API is unavailable")\n    except OpenUvError as err:\n        print(f"There was an error: {err}")\n\n\nasyncio.run(main())\n```\n\n## Connection Pooling\n\nBy default, the library creates a new connection to OpenUV with each coroutine. If you\nare calling a large number of coroutines (or merely want to squeeze out every second of\nruntime savings possible), an [`aiohttp`][aiohttp] `ClientSession` can be used for\nconnection pooling:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\nfrom pyopenuv import Client\nfrom pyopenuv.errors import OpenUvError\n\n\nasync def main():\n    async with ClientSession() as session:\n        client = Client(\n            "<OPENUV_API_KEY>",\n            "<LATITUDE>",\n            "<LONGITUDE>",\n            altitude="<ALTITUDE>",\n            session=session,\n        )\n\n        try:\n            print(await client.uv_index())\n        except OpenUvError as err:\n            print(f"There was an error: {err}")\n\n\nasyncio.run(main())\n```\n\n# Contributing\n\nThanks to all of [our contributors][contributors] so far!\n\n1. [Check for open features/bugs][issues] or [initiate a discussion on one][new-issue].\n2. [Fork the repository][fork].\n3. (_optional, but highly recommended_) Create a virtual environment: `python3 -m venv .venv`\n4. (_optional, but highly recommended_) Enter the virtual environment: `source ./.venv/bin/activate`\n5. Install the dev environment: `script/setup`\n6. Code your new feature or bug fix on a new branch.\n7. Write tests that cover your new functionality.\n8. Run tests and ensure 100% code coverage: `poetry run pytest --cov pyopenuv tests`\n9. Update `README.md` with any new documentation.\n10. Submit a pull request!\n\n[aiohttp]: https://github.com/aio-libs/aiohttp\n[ci-badge]: https://github.com/bachya/pyopenuv/workflows/CI/badge.svg\n[ci]: https://github.com/bachya/pyopenuv/actions\n[codecov-badge]: https://codecov.io/gh/bachya/pyopenuv/branch/dev/graph/badge.svg\n[codecov]: https://codecov.io/gh/bachya/pyopenuv\n[contributors]: https://github.com/bachya/pyopenuv/graphs/contributors\n[fork]: https://github.com/bachya/pyopenuv/fork\n[issues]: https://github.com/bachya/pyopenuv/issues\n[license-badge]: https://img.shields.io/pypi/l/pyopenuv.svg\n[license]: https://github.com/bachya/pyopenuv/blob/main/LICENSE\n[maintainability-badge]: https://api.codeclimate.com/v1/badges/a03c9e96f19a3dc37f98/maintainability\n[maintainability]: https://codeclimate.com/github/bachya/pyopenuv/maintainability\n[new-issue]: https://github.com/bachya/pyopenuv/issues/new\n[new-issue]: https://github.com/bachya/pyopenuv/issues/new\n[openuv]: https://openuv.io/\n[openuv-console]: https://www.openuv.io/console\n[pypi-badge]: https://img.shields.io/pypi/v/pyopenuv.svg\n[pypi]: https://pypi.python.org/pypi/pyopenuv\n[version-badge]: https://img.shields.io/pypi/pyversions/pyopenuv.svg\n[version]: https://pypi.python.org/pypi/pyopenuv\n',
    'author': 'Aaron Bach',
    'author_email': 'bachya1208@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/bachya/pyopenuv',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9.0,<4.0.0',
}


setup(**setup_kwargs)
