# AUTOGENERATED! DO NOT EDIT! File to edit: projects/m5/02_WRMSSE_metric.ipynb (unless otherwise specified).

__all__ = ['get_agg', 'get_df_weights', 'combine_cols', 'append_df_unique_id', 'WRMSSE']

# Cell
#export
import os
from time import time
import gc

import pandas as pd
import numpy as np
from scipy.sparse import csr_matrix
import matplotlib.pyplot as plt
import seaborn as sns
sns.set()

plt.rcParams['figure.figsize'] = (14,6)
plt.rcParams['font.size'] = 16

# Cell
def get_agg(df_stv):
    """Gets a sparse aggregaion matrix and index to align weights and scales."""

    # Take the transpose of each dummy matrix to correctly orient the matrix
    dummy_frames = [
        pd.DataFrame({'Total': np.ones((df_stv.shape[0],)).astype('int8')}, index=df_stv.index).T,
        pd.get_dummies(df_stv.state_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.store_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.cat_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.dept_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.state_id + '_' + df_stv.cat_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.state_id + '_' + df_stv.dept_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.store_id + '_' + df_stv.cat_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.store_id + '_' + df_stv.dept_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.item_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.item_id + '_' + df_stv.state_id, dtype=np.int8).T,
        pd.get_dummies(df_stv.item_id + '_' + df_stv.store_id, dtype=np.int8).T
    ]
    agg_matrix = pd.concat(dummy_frames, keys=range(1,13), names=['level', 'id'])

    # Save the index for later use
    agg_index = agg_matrix.index

    # Sparse format will save space and calculation time
    agg_matrix_csr = csr_matrix(agg_matrix)
    return agg_matrix_csr, agg_index

# Cell
def get_df_weights(df_stv, df_cal, df_prices, agg_index, agg_matrix_csr, start_test=1914):
    """Returns the weight, scale, and scaled weight of all series,
    in a dataframe aligned with the agg_index, created in get_agg()

    ##### Weights  steps
    We need to convert the sales data into dollar sales
    data so that we can correctly weight each series.
    To begin, we consider only the last 28 days of
    data before START_TEST. We then put the data into
    "long" format so we can merge the calendar
    and price information.

    Now we will get the total dollar sales for each
    item/store combination. Be sure to set sort=False
    so that our index stays in the proper order.
    We don't need df anymore

    We want to build a weight, scales,
    and scaled weight columns
    that are aligned with agg_index. We
    will divide dollar_sales by the total
    dollar sales to get the weight W
    for each series. We don't need dollar_sales anymore.

    ##### Scaling factor steps
    We also need to calculate each series scaling factor S,
    which is the denominator in the WRMSSE cacluation. It can
    be pulled out of the square root and combined with the
    series weight to make a single weight W/sqrt(S),
    simplifying our calculations a bit.

    S is the average squared difference of day to daily sales
    for a series, excluding leading zeros, for all training
    days leading up to START_TEST.

    Aggregate all series, and replace leading
    zeros with np.nan so that we can do numpy calculations
    that will ignore the np.nan.

    Now we can finish our weights and scales dataframe by
    adding scale and scaled_weight columns.
    """

    d_cols = [f'd_{i}' for i in range(start_test - 28, start_test)]
    df = df_stv[['store_id', 'item_id'] + d_cols]
    df = df.melt(id_vars=['store_id', 'item_id'],
                           var_name='d',
                           value_name = 'sales')
    df = df.merge(df_cal[['d', 'wm_yr_wk']], on='d', how='left')
    df = df.merge(df_prices, on=['store_id', 'item_id', 'wm_yr_wk'], how='left')
    df['dollar_sales'] = df.sales * df.sell_price

    # Now we will get the total dollar sales
    dollar_sales = df.groupby(['store_id', 'item_id'], sort=False)['dollar_sales'].sum()
    del df

    # Build a weight, scales, and scaled weight columns
    # that are aligned with agg_index.
    df_weights = pd.DataFrame(index = agg_index)
    df_weights['dollar_sales'] = agg_matrix_csr * dollar_sales
    df_weights['weight'] = df_weights.dollar_sales / df_weights.dollar_sales.values[0]
    del df_weights['dollar_sales']

    ##################### Scaling factor #######################
    df = df_stv.loc[:, :f'd_{start_test - 1}'].iloc[:, 6:]
    agg_series = agg_matrix_csr * df.values
    no_sale = np.cumsum(agg_series, axis=1) == 0
    agg_series = np.where(no_sale, np.nan, agg_series)
    scale = np.nanmean(np.diff(agg_series, axis=1) ** 2, axis=1)

    df_weights['scale'] = 1 / np.sqrt(scale)
    df_weights['scaled_weight'] = df_weights.weight * df_weights.scale

    return df_weights

# Cell
def combine_cols(df, cols: list, sep='__', name='id', reverse=False):
    """Returns a copy of `df` with `cols` combined into a single coloumn `name`,
    separated by `sep`, or with the `name` column expanded into `cols` if `reverse` is True."""

    df = df.copy()
    if reverse:
        final_cols = cols + df.drop(name, axis=1).columns.tolist()
        df[cols] = df[name].str.split(sep).tolist()
    else:
        final_cols = [name] + df.drop(cols, axis=1).columns.tolist()
        df[name] = df[cols].astype(str).apply(sep.join, axis=1)
    return df[final_cols]

# Cell
def append_df_unique_id(df, df_new, id_col='id') -> pd.DataFrame:
    """Returns a copy of df with df_new appended to it with '(n)_'
    prepended to the id_col if the new column value is already in
    the original df. This is used to track scores and ensure there
    are not copies of a unique identifier.

    `id_col` should be of string type.
    """

    if not (id_col in df.columns and id_col in df_new.columns):
        return df.append(df_new) # No issues
    df = df.copy()
    df_new = df_new.copy()
    ids = df[id_col].tolist()
    new_id = df_new[id_col][0]
    if new_id in ids:
        x = 1
        while f'({x})_' + new_id in ids:
            x += 1
        new_id = f'({x})_' + new_id
        df_new[id_col] = f'({x})_' + df_new[id_col]
    return df.append(df_new)

# Cell
class WRMSSE():
    def __init__(self, PATH_DATA_RAW: str='data/raw', start_test: int=1914, horizon: int=28, df_stv_trunc: pd.DataFrame=None):
        """The main object that will hold data, weights and scales which are
        associated with the forecast horizon starting on `start_test`,
        extending horizon `days`.
        """

        if type(df_stv_trunc) == pd.DataFrame: # Provided after filtering out certain items
            self.df_stv = df_stv_trunc
        else:
            self.df_stv = pd.read_csv(os.path.join(PATH_DATA_RAW, 'sales_train_evaluation.csv'))
        self.df_cal = pd.read_csv(os.path.join(PATH_DATA_RAW, 'calendar.csv'))
        self.df_prices = pd.read_csv(os.path.join(PATH_DATA_RAW, 'sell_prices.csv'))
        self.df_ss = pd.read_csv(os.path.join(PATH_DATA_RAW, 'sample_submission.csv'))

        self.start_test = start_test
        self.end_test = start_test + horizon - 1
        self.preds, self.actuals = None, None
        self.df_series_scores, self.model_name = None, None

        path = os.path.join(PATH_DATA_RAW, '..', 'scores.csv')
        if os.path.exists(path):
            self.scores = pd.read_csv(path)
        else:
            self.scores = pd.DataFrame()

        if f'd_{self.end_test}' in self.df_stv.columns:
            self.actuals = self.df_stv.loc[:, f'd_{start_test}': f'd_{self.end_test}'].values

        self.agg_matrix_csr, self.agg_index = get_agg(self.df_stv)
        self.df_weights = get_df_weights(self.df_stv, self.df_cal, self.df_prices,
                                         self.agg_index, self.agg_matrix_csr, start_test)
        self.w_12 = self.df_weights.loc[12]
        self.w_12.index += '_evaluation'

    def score(self, preds: np.array, fast: bool=True, model_name: str=None) -> float:
        """Scores preds against `self.actuals`. If `fast` is set to True, nothing
        will be saved. If `fast` is set to False, `self.df_series_scores` will be
        set to a dataframe with the scores for each of the 40280 series, and if
        `model_name` name is also passed, `self.model_name` is set, `self.scores`
        will be updated with the 12 levels scores along with total score and then
        saved to csv.
        """

        if type(preds) == pd.DataFrame:
            preds = preds.values

        base_errors = self.actuals - preds
        errors = self.agg_matrix_csr * base_errors
        mse = np.sqrt(np.mean((errors)**2, axis=1))
        wrmsse_by_series = mse * self.df_weights.scaled_weight
        wrmsse = np.sum(wrmsse_by_series) / 12

        if not fast:
            self.preds = preds
            self.df_series_scores = pd.DataFrame(wrmsse_by_series).rename(
                mapper={'scaled_weight': 'WRMSSE'}, axis=1)
            if model_name:
                self.model_name = model_name
                print(f'Saving level scores with model name: {model_name}')
                self._append_level_scores(self.df_series_scores, model_name)

        return wrmsse

    def feval(self, preds, train_data) -> tuple:
        """For custom metric in lightgbm"""
        preds = preds.reshape(self.actuals.shape[1], -1).T
        score = self.score(preds)
        return 'WRMSSE', score, False

    @staticmethod
    def get_weighted_mse_feval(w_12_eval, weight_col) -> callable:
        """Returns a weighted root mean squared error metric function for lightgbm.
        w_12_eval must be aligned with grid_df like
        w_12_eval = w_12.reindex(grid_df[eval_mask].id)
        """

        weight = w_12_eval[weight_col] / w_12_eval[weight_col].mean()
        def feval(preds, eval_data) -> tuple:
            actuals = eval_data.get_label()
            diff = preds - actuals
            res = np.mean(diff ** 2 * weight)
            return f'mse_feval_{weight_col}', res, False
        return feval

    @staticmethod
    def get_weighted_mae_feval(w_12_eval, weight_col) -> callable:
        """Returns a weighted mean absolute error metric function for lightgbm.
        w_12_eval must be aligned with grid_df like
        w_12_eval = w_12.reindex(grid_df[eval_mask].id)
        """

        weight = w_12_eval[weight_col] / w_12_eval[weight_col].mean()
        def feval(preds, eval_data) -> tuple:
            actuals = eval_data.get_label()
            diff = preds - actuals
            res = np.mean(np.abs(diff ** 2 * weight))
            return f'mae_feval_{weight_col}', res, False
        return feval

    @staticmethod
    def get_weighted_mse_fobj(w_12_train, weight_col, weight_hess=True) -> callable:
        """Returns a weighted mean squared error objective function for lightgbm.
        w_12_train must be aligned with grid_df like
        w_12_train = w_12.reindex(grid_df[train_mask].id)
        """

        weight = w_12_train[weight_col] / w_12_train[weight_col].mean()
        def fobj(preds, train_data) -> tuple:
            actuals = train_data.get_label()
            diff = preds - actuals
            grad = diff * weight
            hess = weight if weight_hess else np.ones_like(diff)
            return grad, hess
        return fobj

    @staticmethod
    def get_weighted_mae_fobj(w_12_train, weight_col, weight_hess=True) -> callable:
        """Returns a weighted mean absolute error objective function for lightgbm.
        w_12_train must be aligned with grid_df like
        w_12_train = w_12.reindex(grid_df[train_mask].id)
        """

        weight = w_12_train[weight_col] / w_12_train[weight_col].mean()
        def fobj(preds, train_data) -> tuple:
            actuals = train_data.get_label()
            diff = preds - actuals
            grad = np.sign(diff) * weight
            hess = weight if weight_hess else np.ones_like(diff)
            return grad, hess
        return fobj



    def _append_level_scores(self, df_series_scores, model_name) -> None:
        # level_scores
        level_scores = df_series_scores.groupby(level=0).sum()
        level_scores.loc[13] = level_scores.mean()
        level_scores['model_name'] = model_name
        level_scores['start_test'] = start_test
        level_scores.reset_index(inplace=True)
        cols, sep, name = ['model_name', 'level', 'start_test'], '__', 'id'
        level_scores = combine_cols(level_scores, cols, sep, name)
        self.scores = append_df_unique_id(self.scores, level_scores)

    def dump_scores(self, path_dir: str='.') -> None:
        """Saves `self.scores`, which contains scores of each level for
        each `model_name` `start_test` combination.
        """

        self.scores.to_csv(os.path.join(path_dir, 'scores.csv'), index=False)

    def plot_scores(self, df_series_scores=None, model_name: str=None) -> tuple:
        """Returns a tuple: fig, ax with a seaborn plot of the 12 levels of the wrmsse."""

        if not df_series_scores: df_series_scores = self.df_series_scores
        if not model_name: model_name = self.model_name

        fig, ax = plt.subplots()
        level_scores = df_series_scores.groupby(level=0).sum()
        sns.barplot(x=level_scores.index, y=level_scores['WRMSSE'])
        plt.axhline(level_scores.mean()[0], color='blue', alpha=.5, ls=':')
        name_and_days = f'{model_name} test {self.start_test} to {self.end_test}'
        title = f'{name_and_days} WRMSSE: {round(level_scores.mean()[0], 4)}'
        plt.title(title, fontsize=20, fontweight='bold')
        for i in range(12):
            ax.text(i, level_scores['WRMSSE'][i+1],
                    str(round(level_scores['WRMSSE'][i+1], 4)),
                    color='black', ha='center', fontsize=15)
        plt.show()
        return fig, ax

    def make_sub(self, preds: np.array=None, test=False, model_name='no_name', path_dir='.') -> None:
        """Creates and writes a csv file that is ready for submission. If `test` is
        set to True, it will be for the final test set, otherwise, the predictions
        are for the validation set.

        The files name will be at `path_dir`/sub_`model_name`.csv"""
        if not preds: preds = self.preds
        model_name = self.model_name if self.model_name else 'no_name'

        df_preds = pd.DataFrame(preds, index=df_scores.loc[12].index).reset_index()

        if not test: df_preds['id'] = df_preds['id'] + '_validation'
        else: df_preds['id'] = df_preds['id'] + '_evaluation'

        df_sub = df_ss[['id']].merge(df_preds, on='id', how='left').fillna(0)
        file_name = 'sub_' + model_name + '.csv'
        df_sub.to_csv(os.path.join(path_dir, file_name), index=False)

    def get_oos_scale(self, oos_train_df) -> None:
        """Gets the series scaling factor for the level 12 series and adds this
        as a column to self.w_12. Used for 'out of stock' fixed data and possibly
        a better otpimizing metric or objective function."""

        rec = oos_train_df.iloc[:, 6:-28]
        rdiff = np.diff(rec, axis=1)
        self.w_12['oos_scale'] = 1 / np.sqrt(np.nanmean(rdiff**2, axis=1))
        self.w_12['oos_scaled_weight'] = self.w_12['weight'] * self.w_12['oos_scale']

    def add_total_scaled_weight(self) -> None:
        """Creates a column in self.w_12 that is meant to appropriately weight
        each level 12 series to optimize for the wrmsse metric, while still using
        mean squared error on level 12 series as an objective function.

        Explanation of the problem:
        ---------------------------
        Each sereies the 12th level has a scaled weight associated with it.
        If we were only being scored with the 12th level wrmsse, this would
        seem to be an ideal weight. But we are being scored on all 12 levels
        of aggregation equally. Even if our objective function is not optimizing
        for the wrmsse on all 12 levels directly, we may be able to properly
        weight the level 12 series so that they reflect their overall impact
        to all of the 12 levels of aggregation.

        To illustrate the point, there could be item x, which costs $1000, and
        item y, which costs $1. Assuming they have the same sales fluctuations,
        and thus the same scaling factor, item x will have a weight 1000 times
        that of y. But both x and y appear in 11 other series, where they are
        equally as important as every other item in that series. For instance,
        x and y have equal importance on level 1, 'Total sales of all items'.

        Solution:
        ---------
        For each level 12 series x we find the weights (scaled weight since this
        is the 'true weight' of the series) of the series to which x belongs on
        all other 11 levels of aggregation and add them up. Our hope is this
        will allow us to optimize for the wrmsse metric, while still using mean
        squared error on level 12 series as an objective function.

        """
        w_df = self.df_weights
        tmp = self.w_12.copy()
        tmp['level_1_sw'] = w_df.loc[1].scaled_weight[0]
        tmp['level_2_sw'] = tmp.index.map(lambda x: w_df.loc[(2,x.split('_')[3])].scaled_weight)
        tmp['level_3_sw'] = tmp.index.map(lambda x: w_df.loc[(3,x[-15: -11])].scaled_weight)
        tmp['level_4_sw'] = tmp.index.map(lambda x: w_df.loc[(4,x.split('_')[0])].scaled_weight)
        tmp['level_5_sw'] = tmp.index.map(lambda x: w_df.loc[(5, x.split('_')[0] + '_' +  x.split('_')[1])].scaled_weight)
        tmp['level_6_sw'] = tmp.index.map(lambda x: w_df.loc[(6, x.split('_')[3] + '_' +  x.split('_')[0])].scaled_weight)
        tmp['level_7_sw'] = tmp.index.map(lambda x: w_df.loc[(7, x.split('_')[3] + '_' +  x.split('_')[0] + '_' +  x.split('_')[1])].scaled_weight)
        tmp['level_8_sw'] = tmp.index.map(lambda x: w_df.loc[(8, x.split('_')[3] + '_' +  x.split('_')[4] + '_' +  x.split('_')[0])].scaled_weight)
        tmp['level_9_sw'] = tmp.index.map(lambda x: w_df.loc[(9, x.split('_')[3] + '_' +  x.split('_')[4] + '_' +  x.split('_')[0] + '_' +  x.split('_')[1])].scaled_weight)
        tmp['level_10_sw'] = tmp.index.map(lambda x: w_df.loc[(10, x.split('_')[0] + '_' +  x.split('_')[1] + '_' +  x.split('_')[2])].scaled_weight)
        tmp['level_11_sw'] = tmp.index.map(lambda x: w_df.loc[(11, x.split('_')[0] + '_' +  x.split('_')[1] + '_' +  x.split('_')[2] + '_' +  x.split('_')[3])].scaled_weight)
        self.w_12['total_scaled_weight'] = tmp[['scaled_weight', 'level_1_sw', 'level_2_sw', 'level_3_sw', 'level_4_sw',
               'level_5_sw', 'level_6_sw', 'level_7_sw', 'level_8_sw', 'level_9_sw',
               'level_10_sw', 'level_11_sw']].sum(axis=1)