# AUTOGENERATED! DO NOT EDIT! File to edit: projects/m5/03_feature_engineering.ipynb (unless otherwise specified).

__all__ = ['nan_leading_zeros', 'make_grid_df', 'add_base', 'create_price_fe', 'add_price_fe', 'add_cal_fe',
           'add_snap_transform_1', 'add_snap_transform_2', 'add_event_features', 'fe_base_features', 'encode_target',
           'fe_encodings', 'make_lag_col', 'add_lags', 'fe_lags', 'rolling_window', 'split_array', 'make_rolling_col',
           'diff_mean', 'diff_nanmean', 'mean_decay', 'add_rolling_cols', 'fe_rw_stats', 'get_days_since_sale',
           'add_dow_means', 'fe_dow_means', 'add_shift_cols', 'fe_shifts_momentum', 'fe_ipca_lags', 'fe']

# Cell
#export
import os
import sys
import gc
import time
import logging
from typing import Union

import numpy as np
import pandas as pd
from sklearn.decomposition import PCA, IncrementalPCA
from sklearn.preprocessing import StandardScaler
import matplotlib.pyplot as plt
import matplotlib.dates as mdates
import seaborn as sns
from fastcore.script import call_parse, Param

from ..core import *

# Cell
def nan_leading_zeros(rec):
    """Leading zeros indicate an item was not for sale. We
    will mark as np.nan to ensure they are not used for training."""

    rec = rec.astype(np.float64)
    zero_mask = rec.cumsum(axis=1) == 0
    rec[zero_mask] = np.nan
    return rec

# Cell
def make_grid_df(df: 'df or path_to_train_file',
                 pred_horizon=28) -> (pd.DataFrame, np.array):
    """Specific to the the M5 competition data.
    Returns a grid_df to allign all features and the sales
    data in a "rectangle" data representation, a 2D numpy array
    where ever row is an items time series.
    """

    start_time = time.time()
    logging.info("#" * 72)
    logging.info("Making grid_df")

    # If passed a path to raw data instead of a dataframe, create df
    if type(df) == str:
        df = pd.read_csv(df)
    else:
        df = df.copy()

    start_test = int(df.columns[-1][2:]) + 1
    if pred_horizon:
        for i in range(start_test, start_test + pred_horizon):
            df[f'd_{i}'] = np.nan

    d_cols = [col for col in df.columns if col.startswith('d_')]

    # Turn leading zeros into np.nan
    rec = nan_leading_zeros(df[d_cols].values)
    sales = rec.T.reshape(-1,)

    grid_df = pd.DataFrame({
        'id': pd.Series(np.tile(df.id, len(d_cols))).astype('category'),
        'd': np.concatenate([[int(s[2:])] * df.shape[0] for s in d_cols]).astype(np.int16),
        'sales': sales.astype(np.float16)
    })

    logging.info(time_taken(start_time))
    return grid_df, rec

# Cell
def add_base(grid_df, df_stv, rec):
    """ Adds the basic categorical features to grid_df. """
    logging.info('#' * 72)
    logging.info('Adding basic categorical features to grid_df')
    start_time = time.time()
    for col in ['item_id', 'dept_id', 'cat_id', 'store_id', 'state_id']:
        grid_df[col] = pd.Series(np.tile(df_stv[col], rec.shape[1])).astype('category')
    logging.info(time_taken(start_time))
    return grid_df

# Cell
def create_price_fe(df_prices):
    """Adds price features onto price_df. This is the step we take
    before merging prices onto our grid_df."""

    start_time = time.time()
    logging.info('#' * 72)
    logging.info('Creating price features')
    ################# Standard statistics #######################
    group = df_prices.groupby(['item_id', 'store_id'])
    df_prices['price_min'] =    group['sell_price'].transform('min').astype('float16')
    df_prices['price_max'] =    group['sell_price'].transform('max').astype('float16')
    df_prices['price_median'] = group['sell_price'].transform('median').astype('float16')
    df_prices['price_mode'] =   group['sell_price'].transform(lambda x: x.value_counts().idxmax()).astype('float16')
    df_prices['price_mean'] =   group['sell_price'].transform('mean').astype('float16')
    df_prices['price_std'] =    group['sell_price'].transform('std').astype('float16')
    df_prices['sell_price'] =   df_prices['sell_price'].astype('float16')

    ################ Normalization of sell_price ###############
    df_prices['price_norm_max'] = (df_prices['sell_price'] / df_prices['price_max']).astype('float16')
    df_prices['price_norm_mode'] = (df_prices['sell_price'] / df_prices['price_mode']).astype('float16')
    df_prices['price_norm_mean'] = (df_prices['sell_price'] / df_prices['price_mean']).astype('float16')

    ################# Momentum ########################
    # Lets also try to incroporate some week to week price
    # comparison, which we will call 'momentum'.
    df_prices['price_momentum'] = (df_prices.sell_price / df_prices.groupby(['item_id', 'store_id'])['sell_price']\
        .transform(lambda x: x.shift(1))).astype('float16')

    # We also want to compare the sell price to recent
    # rolling averages such as the last 4 weeks and 24 weeks.
    df_prices['price_roll_momentum_4'] = (df_prices.sell_price / df_prices.groupby(['item_id', 'store_id'])['sell_price']\
        .transform(lambda x: x.rolling(4).mean())).astype('float16')
    df_prices['price_roll_momentum_24'] = (df_prices.sell_price / df_prices.groupby(['item_id', 'store_id'])['sell_price']\
        .transform(lambda x: x.rolling(24).mean())).astype('float16')

    ################### Last digits #####################
    # Its possible the last two digits of the price
    # indicate if the product is on sale or clearance.
    df_prices['price_end_digits'] = ((df_prices.sell_price - np.floor(df_prices.sell_price)) * 100).astype('float16')

    logging.info(time_taken(start_time))
    return df_prices

# Cell
def add_price_fe(grid_df, df_prices, df_cal):
    """ Adds on price features to grid_df. """

    logging.info('#' * 72)
    logging.info('Adding price features to grid')
    start_time = time.time()
    df_cal = df_cal.copy()

    # df_prices needs an id column to merge with grid_df
    df_prices['id'] = df_prices.item_id + '_' + df_prices.store_id + '_evaluation'

    # Calendar d column must be int to match grid_df
    if type(df_cal.d[1]) != np.int16:
        df_cal['d'] = np.int16(df_cal.d.str[2:])

    # Grid_df will need a 'wm_yr_wk' column
    grid_df = merge_by_concat(grid_df, df_cal[['wm_yr_wk', 'd']], ['d'])

    # We want to make sure our features are in alignment
    original_cols = list(grid_df)
    grid_df = merge_by_concat(grid_df, df_prices, ['id', 'wm_yr_wk'])
    keep_cols = [col for col in list(grid_df) if col not in original_cols and 'id' not in col]
    grid_df = grid_df[['id', 'd', 'sales'] + keep_cols]

    logging.info(time_taken(start_time))
    return grid_df

# Cell
def add_cal_fe(grid_df, df_cal):
    """ Adds calendar features onto grid_df. """

    start_time = time.time()
    logging.info('#' * 72)
    logging.info('Creating calendar features grid')

    df_cal = df_cal.copy()

    # Change calendar d column to int
    if type(df_cal.d[1]) != np.int16:
        df_cal['d'] = np.int16(df_cal.d.str[2:])

    ########### Merge part of calendar data ##############
    grid_df = grid_df.merge(df_cal[[
        'd', 'date','event_name_1', 'event_type_1', 'event_name_2',
        'event_type_2','snap_CA', 'snap_TX', 'snap_WI'
    ]], on='d', how='left')

    ########## Categorify object columns ############
    cat_cols = ['event_name_1', 'event_type_1', 'event_name_2', 'event_type_2','snap_CA', 'snap_TX', 'snap_WI']
    for col in cat_cols:
        grid_df[col] = grid_df[col].astype('category')

    ############# Create standard date features ##############
    grid_df['date'] = pd.to_datetime(grid_df['date'])

    # Make some features from date
    grid_df['tm_d'] = grid_df['date'].dt.day.astype(np.int8)
    grid_df['tm_w'] = grid_df['date'].dt.isocalendar().week.astype(np.int8)
    grid_df['tm_m'] = grid_df['date'].dt.month.astype(np.int8)
    grid_df['tm_y'] = grid_df['date'].dt.year
    grid_df['tm_y'] = (grid_df['tm_y'] - grid_df['tm_y'].min()).astype(np.int8)
    grid_df['tm_wm'] = grid_df['tm_d'].apply(lambda x: np.ceil(x/7)).astype(np.int8)
    grid_df['tm_dw'] = grid_df['date'].dt.dayofweek.astype(np.int8)
    grid_df['tm_w_end'] = (grid_df['tm_dw']>=5).astype(np.int8)

    # Remove date
    del grid_df['date']

    # Change index back to RangeIndex to save space
    grid_df.index = pd.RangeIndex(grid_df.index[-1] + 1)

    logging.info(time_taken(start_time))
    return grid_df

# Cell
def add_snap_transform_1(grid_df):
    """Adds a column that shows which of the 10 snap days it is.
    The value is 0 if it is not a snap day.
    """

    ca = grid_df[grid_df['snap_CA'] == 1].tm_d.unique()
    tx = grid_df[grid_df['snap_TX'] == 1].tm_d.unique()
    wi = grid_df[grid_df['snap_WI'] == 1].tm_d.unique()

    ################## Make mappers #################
    ca_31 = list(ca) + [d for d in range(1,32) if d not in ca]
    list_to_map_to = list(range(1,11)) + [0] * 21
    ca_mapper = dict(zip(ca_31, list_to_map_to))

    tx_31 = list(tx) + [d for d in range(1,32) if d not in tx]
    list_to_map_to = list(range(1,11)) + [0] * 21
    tx_mapper = dict(zip(tx_31, list_to_map_to))

    wi_31 = list(wi) + [d for d in range(1,32) if d not in wi]
    list_to_map_to = list(range(1,11)) + [0] * 21
    wi_mapper = dict(zip(wi_31, list_to_map_to))

    ################# Map ############################
    grid_df.loc[grid_df.id.str.contains('CA'), 'snap_transform_1'] = grid_df.tm_d.map(ca_mapper)
    grid_df.loc[grid_df.id.str.contains('TX'), 'snap_transform_1'] = grid_df.tm_d.map(tx_mapper)
    grid_df.loc[grid_df.id.str.contains('WI'), 'snap_transform_1'] = grid_df.tm_d.map(wi_mapper)

    #################### Save as int8 ################
    grid_df['snap_transform_1'] = grid_df['snap_transform_1'].astype(np.int8)
    return grid_df

# Cell
def add_snap_transform_2(grid_df):
    """This maps snap days and non snap days in way
    that may be more meaningful than `snap_transform_1`.

    Any day above 40 will be a snap day. Lower days are non
    snap, and lowest days are "gap" days in between
    snap days. In this way I'm hoping the model can
    can use this feature as non-categorical, and be
    able to efficiently sort when higher demand days
    may be. Also, 16-21 will always be the days
    following the last snap day and 27-31 will be
    the days leading up to the first snap day. My theory is
    these numbers will encode more meaning with less
    confusion caused by states having different snap days.
    """
    ca = grid_df[grid_df['snap_CA'] == 1].tm_d.unique()
    tx = grid_df[grid_df['snap_TX'] == 1].tm_d.unique()
    wi = grid_df[grid_df['snap_WI'] == 1].tm_d.unique()

    ca_mapper = dict(zip(ca, ca + 40))
    tx_mapper = dict(zip(tx, ca + 40))
    wi_mapper = dict(zip(wi, ca + 40))

    ns_ca = dict(zip([day for day in range(1,32) if day not in ca],
                     [16,17,18,19,20,21] + [22,22,23,23,24,24,25,25,26,26] + [27, 28, 29, 30, 31]))
    ns_tx = dict(zip([day for day in range(1,32) if day not in tx],
                     list(range(1,6)) + list(range(16,32))))
    ns_wi = dict(zip([day for day in range(1,32) if day not in wi],
                     [31] + list(range(1,5)) + list(range(16,22)) + [22] + list(range(22,31))))

    ca_mapper.update(ns_ca)
    tx_mapper.update(ns_tx)
    wi_mapper.update(ns_wi)

    ################# Map ############################
    grid_df.loc[grid_df.id.str.contains('CA'), 'snap_transform_2'] = grid_df.tm_d.map(ca_mapper)
    grid_df.loc[grid_df.id.str.contains(
        'TX'), 'snap_transform_2'] = grid_df.tm_d.map(tx_mapper)
    grid_df.loc[grid_df.id.str.contains('WI'), 'snap_transform_2'] = grid_df.tm_d.map(wi_mapper)

    #################### Save as int8 ################
    grid_df['snap_transform_2'] = grid_df['snap_transform_2'].astype(np.int8)
    return grid_df

# Cell
def add_event_features(grid_df, n_items=30490, n_days_in_data=1969):
    """Adds some features related to special events like holidays to
    grid_df. The columns added are:
    next_event_type_1
    last_event_type_1
    days_since_event
    days_until_event
    """

    # Replace event 1 with "better" event 2 on double event days.
    mask = (grid_df['event_type_1'] == 'Religious') & (grid_df['event_type_2'] == 'Cultural')
    grid_df.loc[mask, 'event_type_1'] = 'Cultural'

    #################### Create next/last event features #######################
    # I don't want to include the actual event, name because I think my model
    # will learn what time of year it is, which I don't necessarily want it to
    # do, because I think it will hurt its generalization.
    grid_df['next_event_type_1'] = grid_df.event_type_1.fillna(method='bfill')
    grid_df['last_event_type_1'] = grid_df.event_type_1.fillna(method='ffill')

    ##################### Days since event feature ###########################
    old_col = grid_df.event_type_1.cat.codes
    counter = 1
    new_col = []
    for i in range(n_days_in_data):
        if old_col[i*n_items] == -1:
            new_col += [counter] * n_items
            counter += 1
        else:
            new_col += [0] * n_items
            counter = 1
    grid_df['days_since_event'] = np.array(new_col).astype(np.int16)

    ############### Days until event feature ########################
    old_col = grid_df.event_type_1.cat.codes[::-1].tolist()
    counter = 1
    new_col = []
    for i in range(n_days_in_data):
        if old_col[i*n_items] == -1:
            new_col += [counter] * n_items
            counter += 1
        else:
            new_col += [0] * n_items
            counter = 1
    grid_df['days_until_event'] = np.array(new_col)[::-1].astype(np.int16)

    return grid_df

# Cell
@call_parse
def fe_base_features(path_data_raw: Param('path to raw data folder', str)='data/raw',
                     path_features: Param('path to feature folder', str)='data/features',
                     path_to_train_file: Param('path to train data', str)=None) -> None:
    """Creates the basic categorical, price, and calendar features using
    the functions `add_base`, `create_price_fe`, `add_price_fe`, and `add_cal_fe`,
    `add_snap_transform_1`, `add_snap_transform_1`, and `add_event_features`.

    Parameters
    ----------
    path_data_raw: Param('path to raw data folder', str)='data/raw'
    path_features: Param('path to feature folder', str)='data/features'
    path_to_train_file: Param('path to train data', str)=None
        This is so we can `oos_sales_train_evaluation.csv` file which is
        the result of filling zero streaks with NaN where we think an item
        is out of stock.
    """

    logging.basicConfig(format='%(asctime)s   %(levelname)s   %(message)s',
                        level=logging.DEBUG,
                        filename='log.log')
    logging.getLogger().addHandler(logging.StreamHandler(sys.stdout))
    logging.info('#' * 72)
    logging.info('#' * 72)
    logging.info('Creating base features')
    start = time.time()

    if not path_to_train_file:
        path_to_train_file = os.path.join(path_data_raw, 'sales_train_evaluation.csv')

    # Raw data
    df_stv = pd.read_csv(path_to_train_file)
    df_prices = pd.read_csv(os.path.join(path_data_raw, 'sell_prices.csv'))
    df_cal = pd.read_csv(os.path.join(path_data_raw, 'calendar.csv'))

    # Grid for alignment of all features
    grid_df, rec = make_grid_df(path_to_train_file)
    original_cols = grid_df.columns

    # Base features
    add_base(grid_df, df_stv, rec)
    path = os.path.join(path_features, 'fe_base.csv')
    save_file(grid_df, path)
    del df_stv
    grid_df = grid_df[original_cols]
    gc.collect()

    # Price features
    df_prices = create_price_fe(df_prices)
    grid_df = add_price_fe(grid_df, df_prices, df_cal)
    del df_prices
    gc.collect()
    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_features, 'fe_price.csv')
    save_file(grid_df, path, usecols)
    grid_df.drop(columns=usecols, inplace=True)
    gc.collect()

    # Calendar features
    grid_df = add_cal_fe(grid_df, df_cal)
    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_features, 'fe_cal.csv')
    save_file(grid_df, path, usecols)

    cols_needed_for_snap = original_cols.tolist() + [
        "event_name_1", "event_type_1", "event_name_2",
        "event_type_2", "snap_CA", "snap_TX", "snap_WI", "tm_d"
    ]
    drop_cols = [c for c in grid_df.columns if c not in cols_needed_for_snap]
    grid_df.drop(columns=drop_cols, inplace=True)
    gc.collect()

    # Snap and event features
    old_cols = grid_df.columns
    grid_df = add_snap_transform_1(grid_df)
    grid_df = add_snap_transform_2(grid_df)
    grid_df = add_event_features(grid_df, n_items=rec.shape[0], n_days_in_data=rec.shape[1])
    usecols = [c for c in grid_df.columns if c not in old_cols]
    path = os.path.join(path_features, 'fe_snap_event.csv')
    save_file(grid_df, path, usecols)

    logging.info(time_taken(start))

# Cell
def encode_target(df: pd.DataFrame, target: str, cols: Union[list, str], func: Union[str, callable], verbose=True):
    """Uses pandas groupby(col)[target].transform(`func`) to encode
    each col in `cols`. The `target` col can be any numerical column.

    Parameters
    ----------
    df: pd.DataFrame
        DataFrame to use
    target: str
        Name of column to be encoded
    cols: Union[list, str]
        Name of column or list of colunm groups to be groupbed by.
        Each item in the list must work with df.groupby(item), so
        multiple column groups should be in a list themselves, such
        as `cols` = ['item_id', ['store_id', 'state_id']].
    func: callable
        Must work with df.groupby(col)[target].transform(func)
        """
    if type(cols) == str: cols = list(cols)
    if type(func) == str: func_name = func
    else: func_name = func.__name__
    for col in cols:
        if verbose: print(f'Encoding for {col}')
        col_name = '_' + '_'.join(col) + '_'
        df[f'enc{col_name}{func_name}'] = df.groupby(col)[target].transform(func).astype(np.float16)
    return df

# Cell
@call_parse
def fe_encodings(path_features: Param('path to feature folder')='data/features',
                 path_out_features: Param('path to feature folder for output', str)=None,
                 start_test: Param('First day to start nans', int)=1942):
    """Creates target encoding with mean and std for various columns, with sales after `start_test`
    set to np.nan.

    Parameters
    ----------
    path_features: Param('path to feature folder', str)='data/features'
    path_out_features: Param('path to feature folder for output', str)='data/features'
        This is mainly to run on kaggle where `path_features` is set to an input dataset,
        because we need the rolling window stat features to be present, and
        `path_out_features` is set the working directory for the output.
    start_test: Param('First day to start nans', int)=1942
    """

    logging.basicConfig(format='%(asctime)s   %(levelname)s   %(message)s',
                        level=logging.DEBUG,
                        filename='log.log')
    logging.getLogger().addHandler(logging.StreamHandler(sys.stdout))
    logging.info('#' * 72)
    logging.info('#' * 72)
    logging.info('Creating encodings')
    start_time = time.time()

    if not path_out_features:
        path_out_features = path_features

    grid_df = pd.concat([
        load_file(os.path.join(path_features, 'fe_base.csv')),
        load_file(os.path.join(path_features, 'fe_cal.csv'), usecols=['tm_m']),
        load_file(os.path.join(path_features, 'fe_snap_event.csv'), usecols=['snap_transform_1'])
    ], axis=1)
    grid_df.loc[grid_df.d >= start_test, 'sales'] = np.nan
    original_cols = grid_df.columns    # Saving RAM
    grid_df['sales'] = grid_df['sales'].astype(np.float16)
    grid_df['snap_transform_1'] = grid_df.snap_transform_1.astype(np.int8)

    cols =  [
        ['state_id'],
        ['store_id'],
        ['cat_id'],
        ['dept_id'],
        ['state_id', 'cat_id'],
        ['state_id', 'dept_id'],
        ['store_id', 'cat_id'],
        ['store_id', 'dept_id'],
        ['item_id'],
        ['item_id', 'state_id'],
        ['item_id', 'store_id']
    ]

    encode_target(grid_df, 'sales', cols, 'mean')
    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_features, 'fe_enc_mean.csv')
    save_file(grid_df, path, usecols)
    grid_df.drop(columns=usecols, inplace=True)
    gc.collect()

    encode_target(grid_df, 'sales', cols, 'std')
    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_features, 'fe_enc_std.csv')
    save_file(grid_df, path, usecols)
    grid_df.drop(columns=usecols, inplace=True)
    gc.collect()

    cols = [
        ['store_id', 'dept_id', 'snap_transform_1'],
        ['item_id', 'store_id', 'snap_transform_1'],
        ['store_id', 'dept_id', 'tm_m'],
        ['item_id', 'store_id', 'tm_m'],
        ['store_id', 'dept_id', 'snap_transform_1', 'tm_m'],
        ['item_id', 'store_id', 'snap_transform_1', 'tm_m'],
    ]

    encode_target(grid_df, 'sales', cols, 'mean')
    encode_target(grid_df, 'sales', cols, 'std')
    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_features, 'fe_enc_special.csv')
    save_file(grid_df, path, usecols)

    logging.info(time_taken(start_time))

# Cell
def make_lag_col(rec: np.array, lag: int) -> np.array:
    """Transform the 'rectangle' of time series into a lag feature"""
    lag_rec = np.roll(rec, shift=lag, axis=1).astype(np.float16)
    lag_rec[:, :lag] = np.nan
    return lag_rec.T.reshape(-1)

# Cell
def add_lags(grid_df, rec, lags=range(1,16)):
    start_time = time.time()
    logging.info( 72 * '#')
    logging.info('\nAdding lag columns')
    for lag in lags:
        grid_df[f'lag_{lag}'] = make_lag_col(rec, lag)
    logging.info(time_taken(start_time))
    return grid_df

# Cell
@call_parse
def fe_lags(path_data_raw: Param('path to raw data folder', str)='data/raw',
            path_features: Param('path to feature folder', str)='data/features',
            path_to_train_file: Param('path to train data', str)=None) -> None:
    """Creates lags and rolling window features using `add_lags` `add_rolling_cols`

    Parameters
    ----------
    path_data_raw: Param('path to raw data folder', str)='data/raw'
    path_features: Param('path to feature folder', str)='data/features'
    path_to_train_file: Param('path to train data', str)=None
        This is so we can `oos_sales_train_evaluation.csv` file which is
        the result of filling zero streaks with NaN where we think an item
        is out of stock.
    """

    logging.basicConfig(format='%(asctime)s   %(levelname)s   %(message)s',
                        level=logging.DEBUG,
                        filename='log.log')
    logging.getLogger().addHandler(logging.StreamHandler(sys.stdout))
    logging.info('#' * 72)
    logging.info('#' * 72)
    logging.info('Making lags and rolling window features')
    start_time = time.time()

    if not path_to_train_file:
        path_to_train_file = os.path.join(path_data_raw, 'sales_train_evaluation.csv')

    grid_df, rec = make_grid_df(path_to_train_file)
    original_cols = grid_df.columns

    # Lags
    max_lag = 84
    cols_per_file = 14
    for lag in range(1, max_lag, cols_per_file):
        last_lag = lag + cols_per_file
        add_lags(grid_df, rec, lags=range(lag, last_lag))
        usecols = [c for c in grid_df.columns if c not in original_cols]
        path = os.path.join(path_features, f'shift_fe_lags_{lag}_{last_lag - 1}.csv')
        save_file(grid_df, path, usecols)
        grid_df.drop(columns=usecols, inplace=True)
        gc.collect()

    logging.info(time_taken(start_time))

# Cell
def rolling_window(a: np.array, window: int):
    """
    A super fast way of getting rolling window view with size `window` on a numpy array.
    Reference: https://rigtorp.se/2011/01/01/rolling-statistics-numpy.html
    """

    shape = a.shape[:-1] + (a.shape[-1] - window + 1, window)
    strides = a.strides + (a.strides[-1],)
    return np.lib.stride_tricks.as_strided(a, shape=shape, strides=strides)

# Cell
def split_array(ary, sections, axis=0):
    """Works just like np.split, but sections must be a
    single integer. It will work, even when sections doesn't
    evenly divide the length of ary.

    This avoids errors that occur when using
    `make_rolling_col` with high `n_splits` that do not
    divide the number of series evenly.

    Examples
    --------
    >>> x = np.arange(9)
    >>> split_array(x, 4)
    [array([0, 1, 2]), array([3, 4, 5]), array([6, 7, 8])]

    >>> split_array(x, 5)
    [array([0, 1]), array([2, 3]), array([4, 5]), array([6, 7, 8])]"""
    w = int(np.ceil(len(ary)/sections))
    return np.split(ary, [w * i for i in range(1, int(np.ceil(len(ary)//w)))], axis=axis)

# Cell
def make_rolling_col(rw, function, n_splits=10):
    """Returns a one dimensional np.array after `function` has been applied to the rolling window view `rw`.

    Parameters
    ----------
    rw: a rolling window view as defined in this same module
    function: a callable such as np.mean
        Axis must be the second parameter.
    n_splits: int
        Must divide rw.shape[0]. This will avoid a RAM overflow
        that can occur when using

    We need to take off the last columns to
    get the rolling feature shifted one day,
    since we are making features for training.
    For predicting day 10, we only know the
    rolling features up to day 9.
    """

    split_rw = split_array(rw, n_splits, axis=0)
    split_col = [function(rw, -1) for rw in split_rw]
    col = np.concatenate(split_col)
    col = col[:, :-1].T.reshape(-1,)

    # The new column must be prepended with np.nans to account for missing gaps
    nans = np.zeros(rw.shape[-1] * rw.shape[0]) + np.nan
    return np.append(nans, col).astype(np.float16)

# Cell
def diff_mean(rolling_window, axis=-1):
    """For M5 purposes, used on an object generated by the
    rolling_window function. Returns the mean of the first
    difference of a window of sales."""
    return np.diff(rolling_window, axis=axis).mean(axis=axis)

def diff_nanmean(rolling_window, axis=-1):
    """For M5 purposes, used on an object generated by the
    rolling_window function. Returns the mean of the first
    difference of a window of sales."""
    return np.nanmean(np.diff(rolling_window, axis=axis), axis = axis)

def mean_decay(rolling_window, axis=-1):
    """Returns the mean_decay along an axis of a rolling window object,
    which is created by the rolling_window() function."""

    # decay window must be as long as the last
    # dimension in the rolling window
    decay_window = np.power(.9, np.arange(rolling_window.shape[-1]))[::-1]
    decay_sum = decay_window.sum()
    return (rolling_window * decay_window).sum(axis = -1) / decay_sum

# Cell
def add_rolling_cols(
    grid_df: pd.DataFrame,
    rec: np.array,
    windows: list,
    functions: list,
    function_names: list=None,
    n_splits: "list or int"=10,
):
    """Adds rolling features to grid_df."""

    logging.info(72 * '#')
    logging.info('Adding rolling columns')
    start_time = time.time()

    if not function_names: function_names = [f.__name__ for f in functions]
    if not n_splits: n_splits = [1] * len(functions)
    if type(n_splits) == int: n_splits = [n_splits] * len(functions)
    zipped = list(zip(functions, function_names, n_splits))

    for window in windows:
        rw = rolling_window(rec, window)
        for func, f_name, n_n_splits in zipped:
            s_time = time.time()
            grid_df[f'shift_1_rolling_{f_name}_{str(window)}'] = make_rolling_col(rw, func, n_n_splits)
            logging.info(f'{f_name} with window {window}')
            logging.info(time_taken(s_time))

    logging.info(f'Rolling cols total time:')
    logging.info(time_taken(start_time))
    return grid_df

# Cell
@call_parse
def fe_rw_stats(path_data_raw: Param('path to raw data folder', str)='data/raw',
                path_features: Param('path to feature folder', str)='data/features',
                path_to_train_file: Param('path to train data', str)=None) -> None:
    """Creates lags and rolling window features using `add_lags` `add_rolling_cols`

    Parameters
    ----------
    path_data_raw: Param('path to raw data folder', str)='data/raw'
    path_features: Param('path to feature folder', str)='data/features'
    path_to_train_file: Param('path to train data', str)=None
        This is so we can `oos_sales_train_evaluation.csv` file which is
        the result of filling zero streaks with NaN where we think an item
        is out of stock.
    """

    logging.basicConfig(format='%(asctime)s   %(levelname)s   %(message)s',
                        level=logging.DEBUG,
                        filename='log.log')
    logging.getLogger().addHandler(logging.StreamHandler(sys.stdout))
    logging.info('#' * 72)
    logging.info('#' * 72)
    logging.info('Making lags and rolling window features')
    start_time = time.time()

    if not path_to_train_file:
        path_to_train_file = os.path.join(path_data_raw, 'sales_train_evaluation.csv')

    grid_df, rec = make_grid_df(path_to_train_file)
    original_cols = grid_df.columns

    # Rolling window stats
    grid_df = add_rolling_cols(
        grid_df,
        rec,
        windows=[3, 7],
        functions=[np.nanmean, np.nanmedian, mean_decay, diff_nanmean, np.nanmin, np.nanmax, np.nanstd]
    )
    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_features, 'shift_fe_rw_1.csv')
    save_file(grid_df, path, usecols)
    grid_df.drop(columns=usecols, inplace=True)
    gc.collect()

    grid_df = add_rolling_cols(
        grid_df,
        rec,
        windows=[14, 30],
        functions=[np.nanmean, np.nanmedian, mean_decay, diff_nanmean, np.nanmin, np.nanmax, np.nanstd],
        n_splits=10
    )
    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_features, 'shift_fe_rw_2.csv')
    save_file(grid_df, path, usecols)
    grid_df.drop(columns=usecols, inplace=True)
    gc.collect()

    grid_df = add_rolling_cols(
        grid_df,
        rec,
        windows=[60, 140],
        functions=[np.nanmean, np.nanmedian, mean_decay, diff_nanmean, np.nanmin, np.nanmax, np.nanstd],
        n_splits=50
    )
    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_features, 'shift_fe_rw_3.csv')
    save_file(grid_df, path, usecols)

    logging.info(time_taken(start_time))

# Cell
def get_days_since_sale(grid_df, num_series=30490):
    """Returns a column that shows how many days its been
    Since there has been a sale."""
    df = grid_df[['id', 'd', 'sales']]
    df['non-zero'] = df['sales'] > 0
    df['cum_sum'] = df.groupby(['id'])['non-zero'].transform(np.cumsum)
    tmp_df = df[['id', 'd', 'cum_sum']]
    tmp_df.drop_duplicates(['id', 'cum_sum'], inplace=True)
    tmp_df.columns = ['id', 'd_min', 'cum_sum']
    df = df.merge(tmp_df, on=['id', 'cum_sum'], how='left')
    df['days_since_sale'] = df['d'] - df['d_min']

    # We will only know this stat from one day before
    # so we need to shift the column forward by one day.
    df['days_since_sale'] = df['days_since_sale'].shift(num_series)

    return df['days_since_sale'].astype(np.float16)

# Cell
def add_dow_means(grid_df, rec, n_weeks) -> None:
    """Adds features to grid_df for the mean of each day of the week
    for the past `n_weeks`.

    For any row, the column 'mean_{n_weeks}_dow_{i}' represents the mean
    of the last `n_weeks` of sales for the day of the week that is `i`
    days behind the date of this row. So if today is Friday, n_weeks=4
    and i = 1, this column is equal to the mean sales of the last 4
    Thursdays."""
    for i in range(7):
        days = [d for d in range(1,n_weeks * 7 + 1) if d%7 == i]
        add_lags(grid_df, rec, days)
        dow_cols = [f'lag_{d}' for d in days]
        grid_df[f'mean_{n_weeks}_dow_{i}'] = grid_df[dow_cols].mean(axis=1).astype(np.float16)
        grid_df.drop(dow_cols, axis=1, inplace=True)

# Cell
@call_parse
def fe_dow_means(path_data_raw: Param('path to raw data folder', str)='data/raw',
                 path_features: Param('path to feature folder', str)='data/features',
                 path_to_train_file: Param('path to train data', str)=None) -> None:
    """
    Creates the features for day of week means using `add_dow_means`

    We also use 'get_days_since_sale' in this script since there isn't
    another group very similar to this feature.

    Parameters
    ----------
    path_data_raw: Param('path to raw data folder', str)='data/raw'
    path_features: Param('path to feature folder', str)='data/features'
    path_to_train_file: Param('path to train data', str)=None
        This is so we can `oos_sales_train_evaluation.csv` file which is
        the result of filling zero streaks with NaN where we think an item
        is out of stock.

    """

    logging.basicConfig(format='%(asctime)s   %(levelname)s   %(message)s',
                        level=logging.DEBUG,
                        filename='log.log')
    logging.getLogger().addHandler(logging.StreamHandler(sys.stdout))
    logging.info('#' * 72)
    logging.info('#' * 72)
    logging.info('Making day of week means features')
    start_time = time.time()

    if not path_to_train_file:
        path_to_train_file = os.path.join(path_data_raw, 'sales_train_evaluation.csv')
    grid_df, rec = make_grid_df(path_to_train_file)
    original_cols = grid_df.columns

    ######################## Mean day of week ############################
    add_dow_means(grid_df, rec, 4)
    add_dow_means(grid_df, rec, 20)

    ######################### Last sale day ##############################
    grid_df['days_since_sale'] = get_days_since_sale(grid_df, num_series=rec.shape[0])

    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_features, 'shift_fe_dow_means_and_days_since_sale.csv')
    save_file(grid_df, path, usecols)
    grid_df.drop(columns=usecols, inplace=True)
    gc.collect()

    logging.info(time_taken(start_time))

# Cell
def add_shift_cols(grid_df: pd.DataFrame, shifts: list, cols: list, num_series: int=30490, momentum: bool=True):
    """Adds shift_{`shift`} and momentum_{`shift` - 1} features for each
    int `shift` in `shifts` for each column in `cols`. `cols` must be
    a list of columns that begin with 'shift_1' for this function to work.
    """

    for shift_val in shifts:
        for col in cols:
            shift_name = f"{col.replace('shift_1', f'shift_{shift_val}')}"
            grid_df[shift_name] = grid_df[col].shift((shift_val - 1) * num_series)
            if momentum:
                mom_name = col.replace('shift_1', f'momentum_{shift_val - 1}')
                grid_df[mom_name] = grid_df[col] / grid_df[shift_name]
    return grid_df

# Cell
@call_parse
def fe_shifts_momentum(path_features: Param('path to feature folder', str)='data/features',
                       path_out_features: Param('path to feature folder for output', str)='',
                       num_series: Param('Number of series for shifting', int)=30490) -> None:
    """Creates shifts and momentum features using `add_shift_cols`\n

    Parameters\n
    ----------\n
    path_features: Param('path to feature folder', str)='data/features'\n
    path_out_features: Param('path to feature folder for output', str)='data/features'
        This is mainly to run on kaggle where `path_features` is set to an input dataset,
        because we need the rolling window stat features to be present, and
        `path_out_features` is set the working directory for the output.\n
    num_series: Param('Number of series for shifting', int)=30490
    """

    logging.basicConfig(format='%(asctime)s   %(levelname)s   %(message)s',
                        level=logging.DEBUG,
                        filename='log.log')
    logging.getLogger().addHandler(logging.StreamHandler(sys.stdout))
    logging.info('#' * 72)
    logging.info('#' * 72)
    logging.info('Making shifts and momentum features')
    start_time = time.time()

    if not path_out_features:
        path_out_features = path_features
    dict_features = get_file_cols_dict(path_features)

    shifts = [8, 29]
    cols = ["shift_1_rolling_nanmean_7",
            "shift_1_rolling_mean_decay_7",
            "shift_1_rolling_diff_nanmean_7",]
    grid_df = load_features(path_features=path_features, dict_features=dict_features, features=cols)
    original_cols = grid_df.columns
    grid_df = add_shift_cols(grid_df, shifts, cols, num_series)

    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_out_features, 'shift_fe_shifts_mom_1.csv')
    save_file(grid_df, path, usecols)
    grid_df.drop(columns=usecols, inplace=True)
    gc.collect()

    cols = ["shift_1_rolling_nanmean_30",
            "shift_1_rolling_mean_decay_30",
            "shift_1_rolling_diff_nanmean_30",]
    grid_df = load_features(path_features=path_features, dict_features=dict_features, features=cols)
    original_cols = grid_df.columns
    grid_df = add_shift_cols(grid_df, shifts, cols, num_series)

    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_out_features, 'shift_fe_shifts_mom_2.csv')
    save_file(grid_df, path, usecols)
    grid_df.drop(columns=usecols, inplace=True)
    gc.collect()

    shifts = [29, 91]
    cols = ["shift_1_rolling_nanmean_60",
            "shift_1_rolling_mean_decay_60",
            "shift_1_rolling_diff_nanmean_60",]
    grid_df = load_features(path_features=path_features, dict_features=dict_features, features=cols)
    original_cols = grid_df.columns
    grid_df = add_shift_cols(grid_df, shifts, cols, num_series)

    usecols = [c for c in grid_df.columns if c not in original_cols]
    path = os.path.join(path_out_features, 'shift_fe_shifts_mom_3.csv')
    save_file(grid_df, path, usecols)
    grid_df.drop(columns=usecols, inplace=True)
    gc.collect()

    logging.info(72 * '#')
    logging.info(time_taken(start_time))

# Cell
@call_parse
def fe_ipca_lags(path_data_raw: Param('path to raw data folder', str)='data/raw',
                 path_features: Param('Path to feature file', str)='data/features',
                 path_to_train_file: Param('path to train data', str)=None,
                 end: Param('last day to start lags from', int)=1,
                 restart: Param('start if resuming with lags_df.pkl in restart_dir', int)=None,
                 target: Param('Name of target column', str)='sales') -> None:
    """Creates ipca columns for 84 lag days, starting from the end
    and accumulating backward. With 16 GB of RAM, we can only fit 14
    with ipca at a time, so for each iteration, we:

        1) Create 14 new lag days, and use ipca to reduce it to
        the top 7 compnents.
        2) Combine those with the top 7 components from the previous step
        3) Perform ipca on these 14, features, save the output, and
        keep the top 7 components for the next iteration.

    In the end we will have files with the top 14 ipca components
    for each of these separate ranges:
    Days 1_84
    Days 15_84
    Days 29_84
    Days 43_84
    Days 57_84
    Days 71_84

    Parameters
    ----------
    path_data_raw: Param('path to raw data folder', str)='data/raw',
    path_features: Param('path to feature folder', str)='data/features'
    path_to_train_file: Param('path to train data', str)=None
        This is so we can use `oos_sales_train_evaluation.csv` file which is
        the result of filling zero streaks with NaN where we think an item
        is out of stock.
    target: Param('Name of target column', str)='sales'
    """

    logging.basicConfig(format='%(asctime)s   %(levelname)s   %(message)s',
                        level=logging.DEBUG,
                        filename='log.log')
    logging.getLogger().addHandler(logging.StreamHandler(sys.stdout))
    logging.info('#' * 72)
    logging.info('#' * 72)
    logging.info('Using ipca to reduce lag feature dimensions')
    start_time = time.time()

    if not path_to_train_file:
        path_to_train_file = os.path.join(path_data_raw, 'sales_train_evaluation.csv')

    grid_df, rec = make_grid_df(path_to_train_file, 1)

    ############## Start with indexes with non null sales #############
    # Make sure the first test day is not null,
    # since lags will not be null, we need these
    # indexes to be present.
    start_test = grid_df.d.tolist()[-1]
    grid_df.loc[grid_df.d == start_test, target] = 0
    lags_index = grid_df[grid_df.sales.notnull()].index
    full_index = grid_df[[]]
    del grid_df
    gc.collect()

    start = restart if restart else 71
    for start_range in range(start, end - 1, -14):
        logging.info(f'ipca for {start_range}_84')

        lags_df = full_index.copy()
        lags_df = add_lags(
            lags_df, rec, range(start_range, start_range + 14)
        ).reindex(lags_index).fillna(0).astype('float32')
        lags_df[lags_df.columns] = StandardScaler(copy=False).fit_transform(lags_df.values)
        gc.collect()

        ipca = IncrementalPCA(n_components=14, copy=False)
        cols = lags_df.columns[:ipca.n_components]
        lags_df[cols] = ipca.fit_transform(lags_df.values).astype(np.float16)
        logging.info(f'ipca_{start_range}_{start_range + 13} explained variance ratio:')
        logging.info(ipca.explained_variance_ratio_.round(3))

        if start_range == 71:
            name = f'ipca_{start_range}_84'
            lags_df.columns = [name + f'_comp_{i}' for i in range(1, 15)]
            path = os.path.join(path_features, 'shift_fe_' + name + '.csv')
            save_file(lags_df, path, save_index=True)
            lags_df.iloc[:, :7].to_pickle('lags_df.pkl')
            gc.collect()
            continue

        # Combine top 7 components of new lags pca with top 7 components from the last iteration.
        lags_df = lags_df.iloc[:, :7]
        lags_df = pd.concat([lags_df, pd.read_pickle('lags_df.pkl')], axis=1).astype('float32')
        gc.collect()

        ipca = IncrementalPCA(n_components=14, copy=False)
        cols = lags_df.columns[:ipca.n_components]
        lags_df[cols] = ipca.fit_transform(lags_df.values).astype(np.float16)
        logging.info(f'ipca_{start_range}_84 ipca of ipca cols explained variance ratio:')
        logging.info(ipca.explained_variance_ratio_.round(3))

        # Save all components as csv and top 7 as pickle for next iteration.
        name = f'ipca_{start_range}_84'
        lags_df.columns = [name + f'_comp_{i}' for i in range(1, 15)]
        path = os.path.join(path_features, 'shift_fe_' + name + '.csv')
        save_file(lags_df, path, save_index=True)
        lags_df.iloc[:, :7].to_pickle('lags_df.pkl')
        del lags_df
        gc.collect()

    os.remove('lags_df.pkl')

    logging.info(72 * '#')
    logging.info(time_taken(start_time))

# Cell
@call_parse
def fe():
    fe_base_features()
    fe_lags()
    fe_rw_stats()
    fe_dow_means()
    fe_shifts_momentum()
    fe_ipca_lags()