# Copyright 2025 Louder Digital Pty Ltd.
# All Rights Reserved.
"""Test the bigquery.schema class."""

from __future__ import annotations

import datetime
import typing

import orjson
import pydantic
import pytest
from ldr.modelling import bigquery
from ldr.modelling.bigquery import errors, schema, types

State = typing.Literal["ACT", "NSW", "NT", "QLD", "SA", "TAS", "VIC", "WA"]


class Address(pydantic.BaseModel):
    """An address."""

    street: str
    city: str
    state: State
    post_code: int = pydantic.Field(
        description="The Australian postal code (0000-9999).",
    )


class Interest(pydantic.BaseModel):
    """A person's interest."""

    name: str
    description: str | None


class Person(pydantic.BaseModel):
    """A person."""

    name: str
    # Allow Annotations that have nothing to do
    # with the library and fall through to `int`
    age: typing.Annotated[int, lambda x: x]
    height: float | None = pydantic.Field(
        description="The height of the person in centimeters.",
    )
    address: Address
    po_box: Address | None = None
    interests: list[Interest] = pydantic.Field(
        description="The person's interests.",
    )
    created_at: datetime.datetime = pydantic.Field(
        description="The date and time the record was created.",
    )
    date_of_birth: datetime.date | None = pydantic.Field(
        description="The date of birth of the person.",
    )
    metadata: dict[str, str] | None = pydantic.Field(
        description="Metadata about the person.",
    )
    active: bool
    some_time_field: datetime.time
    opt_time: datetime.time | None
    repeated_strings: list[str]
    another_timestamp: typing.Annotated[types.Timestamp, types.TimestampValidator]
    opt_timestamp: typing.Annotated[types.Timestamp, types.TimestampValidator] | None
    geo: typing.Annotated[types.Geography, types.GeographyValidator]
    opt_geo: typing.Annotated[types.Geography, types.GeographyValidator] | None


PERSON_SCHEMA = [
    {
        "name": "name",
        "type": "STRING",
        "mode": "REQUIRED",
    },
    {
        "name": "age",
        "type": "INTEGER",
        "mode": "REQUIRED",
    },
    {
        "name": "height",
        "type": "FLOAT",
        "mode": "NULLABLE",
        "description": "The height of the person in centimeters.",
    },
    {
        "name": "address",
        "type": "RECORD",
        "mode": "REQUIRED",
        "fields": [
            {
                "name": "street",
                "type": "STRING",
                "mode": "REQUIRED",
            },
            {
                "name": "city",
                "type": "STRING",
                "mode": "REQUIRED",
            },
            {
                "name": "state",
                "type": "STRING",
                "mode": "REQUIRED",
            },
            {
                "name": "post_code",
                "type": "INTEGER",
                "mode": "REQUIRED",
                "description": "The Australian postal code (0000-9999).",
            },
        ],
    },
    {
        "name": "po_box",
        "type": "RECORD",
        "mode": "NULLABLE",
        "fields": [
            {
                "name": "street",
                "type": "STRING",
                "mode": "REQUIRED",
            },
            {
                "name": "city",
                "type": "STRING",
                "mode": "REQUIRED",
            },
            {
                "name": "state",
                "type": "STRING",
                "mode": "REQUIRED",
            },
            {
                "name": "post_code",
                "type": "INTEGER",
                "mode": "REQUIRED",
                "description": "The Australian postal code (0000-9999).",
            },
        ],
    },
    {
        "name": "interests",
        "type": "RECORD",
        "mode": "REPEATED",
        "fields": [
            {
                "name": "name",
                "type": "STRING",
                "mode": "REQUIRED",
            },
            {
                "name": "description",
                "type": "STRING",
                "mode": "NULLABLE",
            },
        ],
        "description": "The person's interests.",
    },
    {
        "name": "created_at",
        "type": "DATETIME",
        "mode": "REQUIRED",
        "description": "The date and time the record was created.",
    },
    {
        "name": "date_of_birth",
        "type": "DATE",
        "mode": "NULLABLE",
        "description": "The date of birth of the person.",
    },
    {
        "name": "metadata",
        "type": "JSON",
        "mode": "NULLABLE",
        "description": "Metadata about the person.",
    },
    {
        "name": "active",
        "type": "BOOLEAN",
        "mode": "REQUIRED",
    },
    {
        "name": "some_time_field",
        "type": "TIME",
        "mode": "REQUIRED",
    },
    {
        "name": "opt_time",
        "type": "TIME",
        "mode": "NULLABLE",
    },
    {
        "name": "repeated_strings",
        "type": "STRING",
        "mode": "REPEATED",
    },
    {
        "name": "another_timestamp",
        "type": "TIMESTAMP",
        "mode": "REQUIRED",
    },
    {
        "name": "opt_timestamp",
        "type": "TIMESTAMP",
        "mode": "NULLABLE",
    },
    {
        "name": "geo",
        "type": "GEOGRAPHY",
        "mode": "REQUIRED",
    },
    {
        "name": "opt_geo",
        "type": "GEOGRAPHY",
        "mode": "NULLABLE",
    },
]


def test_bq_base_schema_to_bigquery_schema() -> None:
    """Test the to_schema_schema method."""
    assert bigquery.to_schema_dict(Person) == PERSON_SCHEMA


def test_bq_base_schema_raises_with_none() -> None:
    """Test that the schema throws with an annotated None."""

    class _Example(pydantic.BaseModel):
        test: None

    with pytest.raises(errors.InvalidTypeError):
        bigquery.to_schema(_Example)


def test_bq_base_schema_to_bigquery_schema_ser() -> None:
    """Test passing serialization arguments."""

    class _Example(pydantic.BaseModel):
        name: str

    assert (
        bigquery.to_schema_ser(_Example, serializer=orjson.dumps)
        == b'[{"name":"name","type":"STRING","mode":"REQUIRED"}]'
    )


def test_bq_base_schema_raises_with_unsupported_union() -> None:
    """Test that the schema throws with an unsupported union."""

    class Example(pydantic.BaseModel):
        """A test schema."""

        test: str | int

    with pytest.raises(errors.UnsupportedTypeError):
        bigquery.to_schema(Example)


def test__union_types_is_valid() -> None:
    """Test that the _union_types_is_valid method."""
    assert schema._union_types_is_valid(str | int) is False
    assert schema._union_types_is_valid(str | int | None) is False
    assert schema._union_types_is_valid(str | None) is True


def test__get_builtin_field_type() -> None:
    """Test the _get_builtin_field_type raises when given an unsupported type."""
    with pytest.raises(errors.InvalidTypeError):
        schema._get_flat_field_type(list)

    with pytest.raises(schema.InvalidTypeError):
        schema._get_flat_field_type(dict)


def test__get_field_type() -> None:
    """Test the _get_field_type method."""
    assert schema._get_field_type(str) == "STRING"
    assert schema._get_field_type(int) == "INTEGER"
    assert schema._get_field_type(float) == "FLOAT"
    assert schema._get_field_type(bool) == "BOOLEAN"
    assert schema._get_field_type(list[str]) == "STRING"
    assert schema._get_field_type(set[int]) == "INTEGER"
    assert schema._get_field_type(dict[str, str]) == "JSON"
    assert schema._get_field_type(datetime.datetime) == "DATETIME"
    assert schema._get_field_type(datetime.date) == "DATE"
    assert schema._get_field_type(str | None) == "STRING"
    assert schema._get_field_type(datetime.time) == "TIME"

    with pytest.raises(errors.InvalidTypeError):
        schema._get_field_type(None)

    with pytest.raises(errors.InvalidTypeError):
        schema._get_field_type(list)

    with pytest.raises(errors.InvalidTypeError):
        schema._get_field_type(dict)

    with pytest.raises(errors.UnsupportedTypeError):
        schema._get_field_type(tuple)

    with pytest.raises(errors.UnsupportedTypeError):
        schema._get_field_type(tuple[str])


def test__get_field_mode() -> None:
    """Test the _get_field_mode method."""
    assert schema._get_field_mode(None) == "NULLABLE"
    assert schema._get_field_mode(type(None)) == "NULLABLE"
    assert schema._get_field_mode(str) == "REQUIRED"
    assert schema._get_field_mode(str | None) == "NULLABLE"
    assert schema._get_field_mode(list[str]) == "REPEATED"
    assert schema._get_field_mode(dict[str, str]) == "REQUIRED"
    assert schema._get_field_mode(dict[str, str] | None) == "NULLABLE"

    with pytest.raises(schema.UnsupportedTypeError):
        schema._get_field_mode(str | int)

    with pytest.raises(schema.UnsupportedTypeError):
        schema._get_field_mode(tuple[str])


def test__get_field_name_with_alias() -> None:
    """Test the _get_field_mode_method."""
    assert (
        schema._get_field_name(
            "street_address",
            pydantic.Field(alias="streetAddress"),
            by_alias=True,
        )
        == "streetAddress"
    )


def test__get_field_name_with_serialization_alias() -> None:
    """Test the _get_field_mode_method."""
    assert (
        schema._get_field_name(
            "street_address",
            pydantic.Field(serialization_alias="streetAddress"),
            by_alias=True,
        )
        == "streetAddress"
    )


def test__get_field_name_with_no_alias() -> None:
    """Test the _get_field_mode_method."""
    with pytest.raises(schema.MissingAliasError):
        schema._get_field_name(
            "street_address",
            pydantic.Field(),
            by_alias=True,
        )


class _ForwardRefModel(pydantic.BaseModel):
    id: int
    name: str
    # Defined after use so should become typing.ForwardRef("_ForwardRefInfo")
    info: _ForwardRefInfo


class _ForwardRefInfo(pydantic.BaseModel):
    category: str


def test_to_schema_with_forward_ref() -> None:
    """Test the schema.to_schema method with a forward reference."""
    assert schema.to_schema_dict(_ForwardRefModel) == [
        {
            "name": "id",
            "type": "INTEGER",
            "mode": "REQUIRED",
        },
        {
            "name": "name",
            "type": "STRING",
            "mode": "REQUIRED",
        },
        {
            "name": "info",
            "type": "RECORD",
            "mode": "REQUIRED",
            "fields": [
                {
                    "name": "category",
                    "type": "STRING",
                    "mode": "REQUIRED",
                },
            ],
        },
    ]


class _InvalidForwardRefModel(pydantic.BaseModel):
    id: int
    ref: Ref  # type: ignore[undefined]  # noqa: F821


def test_to_schema_with_invalid_forward_ref() -> None:
    """Check an InvalidTypeError is raised when a ForwardRef cannot be resolved."""
    with pytest.raises(schema.InvalidTypeError):
        schema.to_schema(_InvalidForwardRefModel)
