# Copyright 2025 Louder Digital Pty Ltd.
# All Rights Reserved.
"""Test the bigquery.param.Param class."""

from __future__ import annotations

import re
from datetime import date, datetime
from typing import Annotated, cast

import pytest
from ldr.modelling import bigquery
from ldr.modelling.bigquery.param import Param, _PydanticProperty, _unwrap_type
from ldr.modelling.bigquery.types import (
    BigNumeric,
    BigNumericValidator,
    Geography,
    GeographyValidator,
    Numeric,
    NumericValidator,
    Timestamp,
    TimestampValidator,
)


class Model(bigquery.Model):
    """Test model."""

    id: int
    name: str
    active: bool
    event_date: date
    event_datetime: datetime
    event_timestamp: Annotated[Timestamp, TimestampValidator]
    event_location: Annotated[Geography, GeographyValidator] | None = None
    event_number: Annotated[Numeric, NumericValidator]
    event_bignum: Annotated[BigNumeric, BigNumericValidator]


class EmptyModel(bigquery.Model):
    """Model to test behaviour when no values are set."""

    value: str | None = None


INSTANCE = Model(
    id=1,
    name="test",
    active=True,
    event_date=date.today(),  # noqa: DTZ011
    event_datetime=datetime.now(),  # noqa: DTZ005
    event_timestamp=Timestamp.now(),
    event_location=cast("Geography", "POINT (1 2)"),
    event_number=Numeric("0.999"),
    event_bignum=BigNumeric("12.592"),
)


def test_model_to_param_same_number_of_fields() -> None:
    """Ensure a model with all BigQuery types can be serialised as a Param."""
    params = Param.from_model(INSTANCE)
    assert len(params) == len(Model.model_fields.keys())


def test_model_from_param() -> None:
    """Ensure a model with all BigQuery types can be serialised as a Param."""
    params = Param.from_model(INSTANCE)
    parsed = Param.parse_into(Model, params)

    assert parsed == INSTANCE


def test_param__from_key_info_with_no_values_set() -> None:
    """Param serialises to {"key": None} when no values set."""
    params = Param.from_model(EmptyModel())
    assert [p.as_kv() for p in params] == [{"value": None}]


def test_param_as_kv_with_multiple_values_set() -> None:
    """Param serialises to {"key": None} when no values set."""
    p = Param(key="value", string_value="test", int_value=0)
    with pytest.raises(
        ValueError,
        match=re.escape("Multiple values set for value: ['test', 0]"),
    ):
        p.as_kv()


def test_param__from_key_info_with_invalid_type() -> None:
    """Param raises UnsupportedTypeError when an invalid type is provided."""
    with pytest.raises(bigquery.UnsupportedTypeError):
        Param._from_key_info(
            key="test",
            field_type=dict[str, str],
            info=_PydanticProperty(),
            value={},
        )


def test_param_from_model_with_missing_alias() -> None:
    """Param raises MissingAliasError when no alias is provided."""
    with pytest.raises(bigquery.MissingAliasError):
        Param.from_model(EmptyModel(), by_alias=True)


def test__unwrap_type_with_unparsable_type() -> None:
    """Ensure _unwrap_type fails when an unparsable type is provided."""
    with pytest.raises(bigquery.UnsupportedTypeError):
        _unwrap_type(int | str)  # type: ignore[union] - weirdness with python union types
