# Copyright 2025 Louder Digital Pty Ltd.
# All Rights Reserved.
"""Module for converting models into BigQuery schemas and rows."""

from __future__ import annotations

import json
import typing

import pydantic
from ldr.modelling.bigquery import schema


class Model(pydantic.BaseModel):
    """Pydantic extension methods to generate BigQuery schemas from models."""

    @classmethod
    def to_bigquery_schema(  # pragma: nocover
        cls,
        *,
        by_alias: bool = False,
    ) -> list[schema.Field]:
        """
        Convert the model to a valid BigQuery schema.

        Supports nested models and lists of models, as well
        as optional fields, and `pydantic.Field` descriptions.

        Params
        ------
        by_alias: Whether to use the alias of the field as the name
                  in the schema. Defaults to `False`.

        Returns
        -------
        The BigQuery schema deserialised into python builtins.

        """
        return schema.to_schema(cls, by_alias=by_alias)

    @classmethod
    def to_bigquery_schema_dict(  # pragma: nocover
        cls,
        *,
        by_alias: bool = False,
    ) -> list[dict[str, typing.Any]]:
        """
        Convert the model to a valid BigQuery schema in the form of a python dict.

        Supports nested models and lists of models, as well
        as optional fields, and `pydantic.Field` descriptions.

        Params
        ------
        by_alias: Whether to use the alias of the field as the name
                  in the schema. Defaults to `False`.

        Returns
        -------
        The BigQuery schema deserialised into python builtins.

        """
        return schema.to_schema_dict(cls, by_alias=by_alias)

    @classmethod
    def to_bigquery_schema_ser[  # pragma: nocover
        **P,
        R,
    ](
        cls,
        by_alias: bool = False,  # noqa: FBT001, FBT002
        serializer: typing.Callable[
            typing.Concatenate[list[dict[str, typing.Any]], P],
            R,
        ] = json.dumps,
        *serializer_args: P.args,
        **serializer_kwargs: P.kwargs,
    ) -> R:
        """
        Convert the model to a valid BigQuery schema and serialize it.

        Defaults to the standard libaray `json.dumps` method, but will
        support any function that can take a `list[dict[str, Any]]`
        and return whatever that serialiser returns.

        Params
        ------
        by_alias: Whether to use the alias of the field as the name
                  in the schema. Defaults to `False`.
        serializer: The callable used to serialize the schema. Defaults to
                    the standard library `json.dumps`.
        serializer_args: Any positional arguments to pass to the serializer callable.
        serializer_kwargs: Any keyword arguments to pass to the serializer callable.

        Returns
        -------
        The serialized BigQuery schema.

        """
        return schema.to_schema_ser(
            cls,
            *serializer_args,
            by_alias=by_alias,
            serializer=serializer,
            **serializer_kwargs,
        )
