"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Condition = void 0;
/**
 * A Condition for use in a Choice state branch
 */
class Condition {
    /**
     * Matches if a boolean field has the given value
     */
    static booleanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.BooleanEquals, value);
    }
    /**
     * Matches if a string field has the given value
     */
    static stringEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringEquals, value);
    }
    /**
     * Matches if a string field sorts before a given value
     */
    static stringLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThan, value);
    }
    /**
     * Matches if a string field sorts equal to or before a given value
     */
    static stringLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanEquals, value);
    }
    /**
     * Matches if a string field sorts after a given value
     */
    static stringGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThan, value);
    }
    /**
     * Matches if a string field sorts after or equal to a given value
     */
    static stringGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanEquals, value);
    }
    /**
     * Matches if a numeric field has the given value
     */
    static numberEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericEquals, value);
    }
    /**
     * Matches if a numeric field is less than the given value
     */
    static numberLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThan, value);
    }
    /**
     * Matches if a numeric field is less than or equal to the given value
     */
    static numberLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanEquals, value);
    }
    /**
     * Matches if a numeric field is greater than the given value
     */
    static numberGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThan, value);
    }
    /**
     * Matches if a numeric field is greater than or equal to the given value
     */
    static numberGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanEquals, value);
    }
    /**
     * Matches if a timestamp field is the same time as the given timestamp
     */
    static timestampEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampEquals, value);
    }
    /**
     * Matches if a timestamp field is before the given timestamp
     */
    static timestampLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThan, value);
    }
    /**
     * Matches if a timestamp field is before or equal to the given timestamp
     */
    static timestampLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanEquals, value);
    }
    /**
     * Matches if a timestamp field is after the given timestamp
     */
    static timestampGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThan, value);
    }
    /**
     * Matches if a timestamp field is after or equal to the given timestamp
     */
    static timestampGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanEquals, value);
    }
    /**
     * Combine two or more conditions with a logical AND
     */
    static and(...conditions) {
        return new CompoundCondition(CompoundOperator.And, ...conditions);
    }
    /**
     * Combine two or more conditions with a logical OR
     */
    static or(...conditions) {
        return new CompoundCondition(CompoundOperator.Or, ...conditions);
    }
    /**
     * Negate a condition
     */
    static not(condition) {
        return new NotCondition(condition);
    }
}
exports.Condition = Condition;
/**
 * Comparison Operator types
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator[ComparisonOperator["StringEquals"] = 0] = "StringEquals";
    ComparisonOperator[ComparisonOperator["StringLessThan"] = 1] = "StringLessThan";
    ComparisonOperator[ComparisonOperator["StringGreaterThan"] = 2] = "StringGreaterThan";
    ComparisonOperator[ComparisonOperator["StringLessThanEquals"] = 3] = "StringLessThanEquals";
    ComparisonOperator[ComparisonOperator["StringGreaterThanEquals"] = 4] = "StringGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["NumericEquals"] = 5] = "NumericEquals";
    ComparisonOperator[ComparisonOperator["NumericLessThan"] = 6] = "NumericLessThan";
    ComparisonOperator[ComparisonOperator["NumericGreaterThan"] = 7] = "NumericGreaterThan";
    ComparisonOperator[ComparisonOperator["NumericLessThanEquals"] = 8] = "NumericLessThanEquals";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanEquals"] = 9] = "NumericGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["BooleanEquals"] = 10] = "BooleanEquals";
    ComparisonOperator[ComparisonOperator["TimestampEquals"] = 11] = "TimestampEquals";
    ComparisonOperator[ComparisonOperator["TimestampLessThan"] = 12] = "TimestampLessThan";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThan"] = 13] = "TimestampGreaterThan";
    ComparisonOperator[ComparisonOperator["TimestampLessThanEquals"] = 14] = "TimestampLessThanEquals";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanEquals"] = 15] = "TimestampGreaterThanEquals";
})(ComparisonOperator || (ComparisonOperator = {}));
/**
 * Compound Operator types
 */
var CompoundOperator;
(function (CompoundOperator) {
    CompoundOperator[CompoundOperator["And"] = 0] = "And";
    CompoundOperator[CompoundOperator["Or"] = 1] = "Or";
})(CompoundOperator || (CompoundOperator = {}));
/**
 * Scalar comparison
 */
class VariableComparison extends Condition {
    constructor(variable, comparisonOperator, value) {
        super();
        this.variable = variable;
        this.comparisonOperator = comparisonOperator;
        this.value = value;
        if (!/^\$|(\$[.[])/.test(variable)) {
            throw new Error(`Variable reference must be '$', start with '$.', or start with '$[', got '${variable}'`);
        }
    }
    renderCondition() {
        return {
            Variable: this.variable,
            [ComparisonOperator[this.comparisonOperator]]: this.value,
        };
    }
}
/**
 * Logical compound condition
 */
class CompoundCondition extends Condition {
    constructor(operator, ...conditions) {
        super();
        this.operator = operator;
        this.conditions = conditions;
        if (conditions.length === 0) {
            throw new Error('Must supply at least one inner condition for a logical combination');
        }
    }
    renderCondition() {
        return {
            [CompoundOperator[this.operator]]: this.conditions.map(c => c.renderCondition()),
        };
    }
}
/**
 * Logical unary condition
 */
class NotCondition extends Condition {
    constructor(comparisonOperation) {
        super();
        this.comparisonOperation = comparisonOperation;
    }
    renderCondition() {
        return {
            Not: this.comparisonOperation.renderCondition(),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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