# This file is part of the Reproducible and Reusable Data Analysis Workflow
# Server (flowServ).
#
# Copyright (C) 2019-2021 NYU.
#
# flowServ is free software; you can redistribute it and/or modify it under the
# terms of the MIT License; see LICENSE file for more details.

"""Definitions for the different types of steps in a serial workflow. At this
point we distinguish three types of workflow steps:

:class:`flowserv.model.workflow.step.FunctionStep` and
:class:`flowserv.model.workflow.step.ContainerStep`
:class: `flowserv.model.workflow.step.NotebookStep`.

A :class:`flowserv.model.workflow.step.CodeStep` is used to execute a given
function within the workflow context. The code is executed within the same
thread and environment as the flowserv engine. Code steps are intended for minor
actions (e.g., copying of files or reading results from previous workflow steps).
For these actions it would cause too much overhead to create an external Python
script that is run as a subprocess or a Docker container image.

A :class:`flowserv.model.workflow.step.ContainerStep` is a workflow step that is
executed in a separate container-like environment. The environment can either be
a subprocess with specific environment variable settings or a Docker container.
"""

from __future__ import annotations
from typing import Callable, Dict, List, Optional

import inspect
import os

"""Unique identifier for workflow step types."""
CONTAINER_STEP = 'container'
FUNCTION_STEP = 'func'
NOTEBOOK_STEP = 'notebook'

STEPS = [CONTAINER_STEP, FUNCTION_STEP, NOTEBOOK_STEP]


class WorkflowStep(object):
    """Base class for the different types of steps (actor) in a serial workflow.

    We distinguish several workflow steps including steps that are executed in
    a container-like environment and steps that directly execute Python code.

    The aim of this base class is to provide functions to distinguish between
    these two types of steps and to maintain properties that are common to all
    steps.

    Each step in a serial workflow has a unique identifier (name) and optional
    lists of input files and output files. All files are specified as relative
    path expressions (keys).
    """
    def __init__(
        self, identifier: str, step_type: int, inputs: Optional[List[str]] = None,
        outputs: Optional[List[str]] = None
    ):
        """Initialize the type identifier for the workflow step.

        Raises a ValueError if an invalid type identifier is given.

        Parameters
        ----------
        identifier: str
            Unique workflow step identifier.
        step_type: int
            Either CONTAINER_STEP or FUNCTION_STEP.
        inputs: list of string, default=None
            List of files that are required by the workflow step as inputs.
        outputs: list of string, default=None
            List of files that are generated by the workflow step as outputs.
        """
        if step_type not in STEPS:
            raise ValueError("invalid step type '{}'".format(step_type))
        self.step_type = step_type
        self.identifier = identifier
        self.inputs = inputs if inputs is not None else list()
        self.outputs = outputs if outputs is not None else list()

    def is_container_step(self) -> bool:
        """True if the workflow step is of type
        :class:`flowserv.model.workflow.step.ContainerStep`.

        Returns
        -------
        bool
        """
        return self.step_type == CONTAINER_STEP

    def is_code_step(self) -> bool:
        """True if the workflow step is of type
        :class:`flowserv.model.workflow.step.CodeStep`.

        Returns
        -------
        bool
        """
        return self.step_type == FUNCTION_STEP

    def is_notebook_step(self) -> bool:
        """True if the workflow step is of type
        :class:'flowserv.model.workflow.step.NotebookStep'.

        Returns
        -------
        bool
        """
        return self.step_type == NOTEBOOK_STEP

    @property
    def name(self) -> str:
        """Synonym for the step identifier.

        Returns
        -------
        string
        """
        return self.identifier


class CodeStep(WorkflowStep):
    """Workflow step that executes a given Python function.

    The function is evaluated using the current state of the workflow arguments.
    If the executed function returns a result, the returned object can be added
    to the arguments. That is, the argument dictionary is updated and the added
    object is availble for the following workflows steps.
    """
    def __init__(
        self, identifier: str, func: Callable, arg: Optional[str] = None,
        varnames: Optional[Dict] = None, inputs: Optional[List[str]] = None,
        outputs: Optional[List[str]] = None
    ):
        """Initialize the reference to the executed function and the optional
        return value target and variable name mapping.

        Parameters
        ----------
        identifier: str
            Unique workflow step identifier.
        func: callable
            Python function that is executed by the workflow step.
        arg: string, default=None
            Name of the variable under which the function result is stored in
            the workflow arguments. If None, the function result is discarded.
        varnames: dict, default=None
            Mapping of function argument names to names of workflow arguments.
            This mapping is used when generating the arguments for the executed
            function. By default it is assumed that the names of arguments for
            the given function correspond to the names in the argument dictionary
            for the workflow. This mapping provides the option to map names in
            the function signature that do not occur in the arguments dictionary
            to argument names that are in the dictionary.
        inputs: list of string, default=None
            List of files that are required by the workflow step as inputs.
        outputs: list of string, default=None
            List of files that are generated by the workflow step as outputs.
        """
        super(CodeStep, self).__init__(
            identifier=identifier,
            step_type=FUNCTION_STEP,
            inputs=inputs,
            outputs=outputs
        )
        self.func = func
        self.arg = arg
        self.varnames = varnames if varnames is not None else dict()

    def exec(self, context: Dict):
        """Execute workflow step using the given arguments.

        The given set of input arguments may be modified by the return value of
        the evaluated function.

        Parameters
        ----------
        context: dict
            Mapping of parameter names to their current value in the workflow
            executon state. These are the global variables in the execution
            context.
        """
        # Generate argument dictionary from the signature of the evaluated function
        # and the variable name mapping.
        kwargs = dict()
        for var in inspect.getfullargspec(self.func).args:
            source = self.varnames.get(var, var)
            if source in context:
                kwargs[var] = context[source]
        # Evaluate the given function using the generated argument dictionary.
        result = self.func(**kwargs)
        # Add the function result to the context dictionary if a variable name
        # for the result is given.
        if self.arg is not None:
            context[self.arg] = result


"""Include FunctionStep class for backward compatibility."""
FunctionStep = CodeStep


class ContainerStep(WorkflowStep):
    """Workflow step that is executed in a container environment. Contains a
    reference to the container identifier and a list of command line statements
    that are executed in a given environment.
    """
    def __init__(
        self, identifier: str, image: str, commands: Optional[List[str]] = None,
        env: Optional[Dict] = None, inputs: Optional[List[str]] = None,
        outputs: Optional[List[str]] = None
    ):
        """Initialize the object properties.

        Parameters
        ----------
        identifier: str
            Unique workflow step identifier.
        image: string
            Execution environment identifier.
        commands: list(string), optional
            List of command line statements.
        env: dict, default=None
            Environment variables for workflow step execution.
        inputs: list of string, default=None
            List of files that are required by the workflow step as inputs.
        outputs: list of string, default=None
            List of files that are generated by the workflow step as outputs.
        """
        super(ContainerStep, self).__init__(
            identifier=identifier,
            step_type=CONTAINER_STEP,
            inputs=inputs,
            outputs=outputs
        )
        self.image = image
        self.commands = commands if commands is not None else list()
        self.env = env if env is not None else dict()

    def add(self, cmd: str) -> ContainerStep:
        """Append a given command line statement to the list of commands in the
        workflow step.

        Returns a reference to the object itself.

        Parameters
        ----------
        cmd: string
            Command line statement

        Returns
        -------
        flowserv.model.workflow.serial.Step
        """
        self.commands.append(cmd)
        return self


class NotebookStep(WorkflowStep):
    def __init__(
        self, identifier: str, notebook: str, output: Optional[str] = None,
        requirements: Optional[List[str]] = None, params: Optional[List[str]] = None,
        varnames: Optional[Dict] = None, inputs: Optional[List[str]] = None,
        outputs: Optional[List[str]] = None
    ):
        """Initialize the notebook step.

        Parameters
        ----------
        identifier: str
            Unique workflow step identifier.
        notebook: string
            Relative path to notebook file in the storage volume that is
            associated with the worker that will execute the notebook step.
        output: string, default=None
            Relative path of the output notebook. If not given, the name of
            the output notebook will be the same name as the input notebook
            with the suffix ``.ipynb`` replaced by ``.out.ipynb`` (or with
            suffix ``.out.ipynb`` appended if the input notebook does not
            have a ``.ipynb`` suffix).
        requirements: list of string, default=None
            List of additional Python packages that need to be installed for
            running the notebook. These requirements will only be taken into
            account when running the notebook inside a Docker container using a
            :class:``flowserv.controller.worker.nbdocker.NotebookDockerWorker``
            worker.
        params: list of string, default=None
            List of names for notebook parameters. These parameters will be
            assigned values from the workflow context when the notebook step
            is executed.
        varnames: dict, default=None
            Mapping of notebook parameters names to names of workflow arguments.
            This mapping is used when generating the arguments for running the
            notebook using papermill. By default it is assumed that the names of
            arguments for the given notebook correspond to the names in the
            argument dictionary for the workflow (i.e., the workflow context).
            This mapping provides the option to map names for notebook parameters
            that do not occur in the arguments dictionary to names that are in
            the dictionary.
        inputs: list of string, default=None
            List of files that are required by the workflow step as inputs.
        outputs: list of string, default=None
            List of files that are generated by the workflow step as outputs.
        """
        super(NotebookStep, self).__init__(
            identifier=identifier,
            step_type=NOTEBOOK_STEP,
            inputs=inputs,
            outputs=outputs
        )
        self.notebook = notebook
        self.output = output_notebook(name=output, input=self.notebook)
        self.requirements = requirements if requirements is not None else list()
        self.params = params if params is not None else list()
        self.varnames = varnames if varnames is not None else dict()

    def cli_command(self, context: Dict) -> str:
        """Get command to run notebbok using papermill from command line.

        This method is used when running a notebook inside a Docker container.

        Parameters
        ----------
        context: dict
            Mapping of parameter names to their current value in the workflow
            executon state. These are the global variables in the execution
            context.
        """
        cli_params = list()
        for arg, val in self._get_parameters(context=context).items():
            if isinstance(val, str):
                val = f'"{val}"'
            cli_params.append(f'-p {arg} {val}')
        parameters = ' '.join(cli_params)
        return f'papermill {self.notebook} {self.output} {parameters}'.strip()

    def exec(self, context: Dict, rundir: str):
        """Execute the notebook using papermill in the given workflow context.

        Parameters
        ----------
        context: dict
            Mapping of parameter names to their current value in the workflow
            executon state. These are the global variables in the execution
            context.
        rundir: string
            Directory for the workflow run that contains all the run files.
        """
        import papermill as pm
        # Prepare parameters for running the notebook using papermill.
        kwargs = self._get_parameters(context=context)
        # Change working directory temporarily to the given rundir.
        cwd = os.getcwd()
        os.chdir(rundir)
        try:
            pm.execute_notebook(self.notebook, self.output, parameters=kwargs)
        finally:
            os.chdir(cwd)

    def _get_parameters(self, context: Dict) -> Dict:
        """Get values for notebook parameters from the workflow context.

        Parameters
        ----------
        context: dict
            Mapping of parameter names to their current value in the workflow
            executon state. These are the global variables in the execution
            context.

        Returns
        -------
        dict
        """
        parameters = dict()
        for var in self.params:
            source = self.varnames.get(var, var)
            if source in context:
                parameters[var] = context[source]
        return parameters


# -- Helper Functions ---------------------------------------------------------

def output_notebook(name: str, input: str) -> str:
    """Generate name for output notebook.

    If an output name is given it is returned as it is. Otherwise, the name
    of the input notebook will have the suffix ``.ipynb`` replaced by ``.out.ipynb``.
    If the input notebook does not have a suffix ``.ipynb`` the suffix  ``.out.ipynb``
    is appended to the input notebook name.

    Parameters
    ----------
    name: string
        User-provided name for the output notebook. This value may be None.
    input: string
        Name of the input notebook.

    Returns
    -------
    string
    """
    if name:
        return name
    return input[:-6] + '.out.ipynb' if input.endswith('.ipynb') else input + '.out.ipynb'
