# This file is part of the Reproducible and Reusable Data Analysis Workflow
# Server (flowServ).
#
# Copyright (C) 2019-2021 NYU.
#
# flowServ is free software; you can redistribute it and/or modify it under the
# terms of the MIT License; see LICENSE file for more details.

"""Parser for serial workflow templates."""

from typing import Dict, List, Optional, Tuple

from flowserv.model.template.base import WorkflowTemplate
from flowserv.model.workflow.step import CodeStep, ContainerStep, NotebookStep, WorkflowStep

import flowserv.model.template.parameter as tp
import flowserv.util as util


def Step(
    identifier: str, action: Dict, inputs: Optional[List[str]] = None,
    outputs: Optional[List[str]] = None
) -> WorkflowStep:
    """Create workflow step instance from dictionary serialization.

    The type of the generated workflow step will depend on the elements in
    the given dictionary serialization.

    Raises a ValueError if the given dictionary is not a valid serialization
    for a workflow step.

    Parameters
    ----------
    identifier: string
        Unique step name (identifier).
    action: Dict
        Dictionary serialization for the workflow step.
    inputs: list of string, default=None
        List of files that are required by the workflow step as inputs.
    outputs: list of string, default=None
        List of files that are generated by the workflow step as outputs.

    Returns
    -------
    flowserv.model.workflow.step.WorkflowStep
    """
    if 'environment' in action and 'commands' in action:
        # If the dictionary contains `environment` and `commands` the result
        # is a container step.
        return ContainerStep(
            identifier=identifier,
            image=action.get('environment'),
            commands=action.get('commands', []),
            inputs=inputs,
            outputs=outputs
        )
    elif 'func' in action:
        return CodeStep(
            identifier=identifier,
            func=util.import_obj(action['func']),
            arg=action.get('arg'),
            varnames=parse_varnames(action=action),
            inputs=inputs,
            outputs=outputs
        )
    elif 'notebook' in action:
        return NotebookStep(
            identifier=identifier,
            notebook=action['notebook'],
            output=action.get('output'),
            params=action.get('params'),
            requirements=action.get('requirements'),
            varnames=parse_varnames(action=action),
            inputs=inputs,
            outputs=outputs
        )
    raise ValueError(f"invalid action specification '{action}'")


def parse_template(template: WorkflowTemplate, arguments: Dict) -> Tuple[List[ContainerStep], Dict, List[str]]:
    """Parse a serial workflow template to extract workflow steps and output
    files.

    The expected schema of the workflow specification is as follows:

    .. code-block:: yaml

        workflow:
            files:
                inputs:
                - "str"
                outputs:
                - "str"
            parameters:
            - name: "scalar"
            steps:
            - name: "str"
              files:
                inputs:
                - "str"
                outputs:
                - "str"
              action: "object depending on the step type"


    The schema for the action specification for a workflow step is dependent on
    the step type. For container steps, the expected schema is:

    .. code-block:: yaml

        action:
            environment: "str"
            commands:
            - "str"

    Expands template parameter references in the workflow argument specification
    and the step inputs list. Returns the modified argument list as part of the
    result.

    Parameters
    ----------
    template: flowserv.model.template.base.WorkflowTemplate
        Template for a serial workflow.

    Returns
    -------
    tuple of list of flowsert.controller.serial.workflow.step.ContainerStep, dict and list of string
    """
    # Get the commands from the workflow specification.
    workflow_spec = template.workflow_spec
    steps = list()
    for doc in workflow_spec.get('steps', []):
        # For each workflow step we expect the elements 'name' and 'action' as
        # well as an optional specification of the input and output files.
        step_id = doc['name']
        action = doc['action']
        input_files = doc.get('files', {}).get('inputs')
        output_files = doc.get('files', {}).get('outputs')
        # The action may either be a reference to an input parameter for the
        # workflow step or a dictionary.
        if isinstance(action, str):
            # If the action references a parameter, we replace the action object
            # with workflow step serialization from the parameter value.
            para = template.parameters[tp.get_name(action)]
            if para.name not in arguments and not para.required:
                # Skip this step if no parameter value was provided and is not
                # a required parameter (step).
                continue
            action = para.cast(value=arguments[para.name]).spec
        step = Step(identifier=step_id, action=action, inputs=input_files, outputs=output_files)
        steps.append(step)
    # Get the workflow arguments that are defined in the workflow template.
    # Expand template parameter references using the given argument set.
    run_args = workflow_spec.get('parameters', {})
    for key in run_args.keys():
        run_args[key] = tp.expand_value(
            value=run_args[key],
            arguments=arguments,
            parameters=template.parameters
        )
    # Return tuple of workflow steps and output file list.
    return steps, run_args, workflow_spec.get('files', {}).get('outputs', {})


def parse_varnames(action: Dict) -> Dict:
    """Parse mapping of function or notebook argument names to the names of
    variables (e.g., parameters) in the workflow context.

    Parameters
    ----------
    action: dict
        Workflow step serialization.

    Returns
    -------
    dict
    """
    return {doc['arg']: doc['var'] for doc in action.get('variables', [])}
