import numpy as np
import matplotlib.pyplot as plt
from anndata import AnnData


def linearity_deviation(
    adata: AnnData, start_milestone, end_milestone, ntop_genes: int = 30
):

    """\
    Plot the results generated by `tl.linearity_deviation`.

    Parameters
    ----------
    adata
        Annotated data matrix.
    start_milestone
        tip defining progenitor branch.
    end_milestone
        tips defining the progeny branch.
    percentiles
        pseudotime percentiles to define the progenitor and progeny populations
    n_jobs
        number of cpu processes used to perform the test.
    n_map
        number of cell mappings from which to do the test.
    plot
        plot the cells selection according to percentiles.
    basis
        basis to use in case of plotting
    copy
        Return a copy instead of writing to adata.

    Returns
    -------
    adata : anndata.AnnData
        if `copy=True` it returns subsetted or else subset (keeping only
        significant features) and add fields to `adata`:

        `.var['A->B_rss']`
            pearson residuals of the linear fit.
        `.obs['A->B_lindev_sel']`
            cell selections used for the test.

    """

    name = start_milestone + "->" + end_milestone

    topgenes = adata.var[name + "_rss"].sort_values(ascending=False)[:ntop_genes]
    ymin = np.min(topgenes)
    ymax = np.max(topgenes)
    ymax += 0.3 * (ymax - ymin)

    fig, ax = plt.subplots()
    ax.set_ylim(ymin, ymax)
    ax.set_xlim(-0.9, len(topgenes) - 0.1)
    for ig, gene_name in enumerate(topgenes.index):
        ax.text(
            ig,
            topgenes[gene_name],
            gene_name,
            rotation="vertical",
            verticalalignment="bottom",
            horizontalalignment="center",
        )

    ax.set_xlabel("ranking")
    ax.set_ylabel("deviance from linearity")
