## gossh-python

The purpose of this module is to provide a Python interface to the Golang [ssh](https://golang.org/x/crypto/ssh) module.

It was made very easy with the help of the Golang [gopy](https://github.com/go-python/gopy) module.

It has come about because I found Paramiko to be a bit unwieldy and have some odd behaviours under certain conditions (we connect to a lot
of wireless devices that often have very poor connectivity); I've also done some testing with SSH2-Python but I think it's a bit too young (
good performance when it worked, but odd behaviour that I couldn't explain from time-to-time, especially when comparing Python vs PyPy).

#### Versions

This version (0.2.4) is the last version to support Python 2; all versions after this have been subject to a refactor and support Python 3
only.

#### Limitations

* Python command needs to be prefixed with GODEBUG=cgocheck=0 (or have that in the environment)
* It's implemented in a very basic non-blocking approach (calls to .Read(size) will always finish instantly)
* Seems to have some odd memory problems with PyPy (via CFFI); lots of locks and stuff to try and alleviate that

#### How do I make use of this?

Right now I'm still working on how to put it all together as a Python module, so here are the raw steps.

#### Prerequisites

* Go 1.13
* Python 2.7+
* pip
* virtualenvwrapper
* pkgconfig/pkg-config

#### Setup (for dev)

* ```mkvirtualenvwrapper -p (/path/to/pypy) gossh-python```
* ```pip install -r requirements.txt```
* ```./build.sh```
* ```GODEBUG=cgocheck=0 py.test -v```

#### What's worth knowing if I want to further the development?

* gopy doesn't like Go interfaces; so make sure you don't have any public (exported) interfaces
    * this includes a struct with a public property that may eventually lead to an interface

#### Example Go RPCSession usage (simple session ID, calls return strings)

There's no real reason why you'd want to do this (just use x/crypto/ssh on it's own)- it's more for reference:

```
package main

import (
	"gossh_python"
	"fmt"
    "time"
)

func main() {

	sessionID := gossh_python.NewRPCSession(
		"1.2.3.4",
		"some_username",
        "some_password",
        22,
		5,
	)

	err := gossh_python.RPCConnect(sessionID)
	if err != nil {
		panic(err)
	}

	err := gossh_python.RPCGethell(sessionID) // this starts a goroutine to read from session's STDOUT
	if err != nil {
		panic(err)
	}

	err := gossh_python.RPCWrite(sessionID, "uname -a\n")
	if err != nil {
		panic(err)
	}

    time.Sleep(time.Second * 5) // because .Read(size) is non-blocking, give the buffer some time to fill up

	data, err := gossh_python.RPCRead(sessionID, 1024)
    if err != nil {
        panic(err)
    }

    fmt.Println(data)

	err = gossh_python.RPCClose(sessionID) // stops the goroutine
	if err != nil {
		panic(err)
	}

}
```

#### Example Python usage (uses RPCSession underneath because of memory leaks between Go and Python with structs)

To create an SSH session in Python do the following:

```
import time

from gossh_python import create_session

session = create_session(
    hostname='1.2.3.4',
    username='some_username',
    password='some_password',
)

session.connect()

session.get_shell()

session.write('uname -a\n')

time.sleep(5)

print(session.read())

session.close()
```

To test the build in a docker container

    ./test.sh
