"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
/* eslint-disable no-console,no-process-env */
const AWS = require("aws-sdk");
const AWSXRay = require("aws-xray-sdk-core");
const ec2Client = AWSXRay.captureAWSClient(new AWS.EC2());
const route53Client = AWSXRay.captureAWSClient(new AWS.Route53());
const DEFAULT_TTL = 60;
const hostedZoneId = process.env.HOSTED_ZONE_ID;
if (!hostedZoneId) {
    throw new Error('HOSTED_ZONE_ID environment variable is not set!');
}
const hostedZoneName = process.env.HOSTED_ZONE_NAME;
if (!hostedZoneName) {
    throw new Error('HOSTED_ZONE_NAME environment variable is not set!');
}
const getNetworkInterfaceForTask = async (task, taskId) => {
    const networkInterfaceId = task.attachments
        ?.find(attachment => attachment.type === 'eni')?.details
        ?.find(details => details.name === 'networkInterfaceId')?.value;
    if (!networkInterfaceId) {
        throw new Error(`Task ${taskId} does not have a network interface.`);
    }
    const networkInterfacesResponse = await ec2Client.describeNetworkInterfaces({
        NetworkInterfaceIds: [networkInterfaceId]
    }).promise();
    if (!networkInterfacesResponse.NetworkInterfaces) {
        throw new Error('DescribeNetworkInterfaces did not return any network interfaces!');
    }
    // eslint-disable-next-line no-magic-numbers
    return networkInterfacesResponse.NetworkInterfaces[0];
};
const getTag = (key, tags) => tags?.find(tag => tag.Key === key)?.Value;
const getRequiredTag = (key, error, tags) => {
    const tag = getTag(key, tags);
    if (!tag) {
        throw new Error(error);
    }
    return tag;
};
const handleTaskRunning = async (taskId, networkInterface) => {
    const nameTag = getRequiredTag('public-discovery:name', `Task ${taskId} does not have the 'public-discovery:name' tag.`, networkInterface.TagSet);
    const name = `${nameTag}.${hostedZoneName}`;
    const publicIp = networkInterface.Association?.PublicIp;
    if (!publicIp) {
        throw new Error(`Task ${taskId} does not have a public ip address.`);
    }
    const ttlTag = getTag('public-discovery:ttl', networkInterface.TagSet);
    const ttl = ttlTag ? Number(ttlTag) : DEFAULT_TTL;
    console.log(`UPSERT '${name}' with address '${publicIp}' for set '${taskId}'.`);
    await route53Client.changeResourceRecordSets({
        ChangeBatch: {
            Changes: [{
                    Action: 'UPSERT',
                    ResourceRecordSet: {
                        MultiValueAnswer: true,
                        Name: name,
                        ResourceRecords: [{ Value: publicIp }],
                        SetIdentifier: taskId,
                        TTL: ttl,
                        Type: 'A'
                    }
                }]
        },
        HostedZoneId: hostedZoneId
    }).promise();
};
const getResourceRecordSetByIdentifier = async (setIdentifier, startRecordIdentifier, startRecordName, startRecordType
// eslint-disable-next-line max-params
) => {
    const resourceRecordSetsResponse = await route53Client.listResourceRecordSets({
        HostedZoneId: hostedZoneId,
        StartRecordIdentifier: startRecordIdentifier,
        StartRecordName: startRecordName,
        StartRecordType: startRecordType
    }).promise();
    const resourceRecordSet = resourceRecordSetsResponse.ResourceRecordSets.find(rrs => rrs.MultiValueAnswer &&
        rrs.Type === 'A' && rrs.SetIdentifier === setIdentifier);
    if (resourceRecordSet) {
        return resourceRecordSet;
    }
    if (resourceRecordSetsResponse.IsTruncated) {
        return getResourceRecordSetByIdentifier(setIdentifier, resourceRecordSetsResponse.NextRecordIdentifier, resourceRecordSetsResponse.NextRecordName, resourceRecordSetsResponse.NextRecordType);
    }
    // eslint-disable-next-line no-undefined
    return undefined;
};
const handleTaskStopped = async (taskId) => {
    const resourceRecordSet = await getResourceRecordSetByIdentifier(taskId);
    if (!resourceRecordSet?.ResourceRecords) {
        console.log(`No resource record sets found with set identifier: '${taskId}'.`);
        return;
    }
    // eslint-disable-next-line no-magic-numbers
    const publicIp = resourceRecordSet.ResourceRecords[0].Value;
    console.log(`DELETE '${resourceRecordSet.Name}' with address '${publicIp}' for set '${taskId}'.`);
    await route53Client.changeResourceRecordSets({
        ChangeBatch: {
            Changes: [{
                    Action: 'DELETE',
                    ResourceRecordSet: {
                        MultiValueAnswer: true,
                        Name: resourceRecordSet.Name,
                        ResourceRecords: [{ Value: publicIp }],
                        SetIdentifier: taskId,
                        TTL: resourceRecordSet.TTL,
                        Type: 'A'
                    }
                }]
        },
        HostedZoneId: hostedZoneId
    }).promise();
};
exports.handler = async (event) => {
    const { taskArn } = event.detail;
    if (!taskArn) {
        throw new Error('Unknown task ARN!');
    }
    // eslint-disable-next-line no-magic-numbers
    const taskId = taskArn.substring(taskArn.lastIndexOf('/') + 1);
    if (event.detail.desiredStatus === 'RUNNING') {
        const networkInterface = await getNetworkInterfaceForTask(event.detail, taskId);
        await handleTaskRunning(taskId, networkInterface);
    }
    else {
        await handleTaskStopped(taskId);
    }
};
//# sourceMappingURL=data:application/json;base64,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